/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.local.control;

import java.util.Vector;
import java.util.Enumeration;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import org.eclipse.hyades.internal.execution.local.common.AgentQueryStateCommand;
import org.eclipse.hyades.internal.execution.local.common.DataProcessor;
import org.eclipse.hyades.internal.execution.local.common.RegisterAgentInterestCommand;
import org.eclipse.hyades.internal.execution.local.common.TCPDataServer;
import org.eclipse.hyades.internal.execution.local.common.ControlMessage;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.AttachToAgentCommand;
import org.eclipse.hyades.internal.execution.local.common.DetachFromAgentCommand;
import org.eclipse.hyades.internal.execution.local.common.ErrorCommand;
import org.eclipse.hyades.internal.execution.local.common.StartMonitoringRemoteAgentCommand;
import org.eclipse.hyades.internal.execution.local.common.StopMonitorCommand;
import org.eclipse.hyades.internal.execution.local.common.SetNVPairCommand;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.local.common.MonitorPeerRequestCommand;

public class AgentImpl implements Agent, ProcessListener, ConnectionListener { // Bug 59316
	protected String _name=null;
	protected String _type=null;
	protected String _uuid=null;
	protected Process _process=null;
	protected AgentConfiguration _config;
	protected boolean _autoAttach=false;
	protected boolean _isMonitored=false;
	protected boolean _isActive=false;
	protected boolean _isAttached=false;
	protected TCPDataServer _server=null;
	protected String _profileFile=null;
	
	protected Vector _listeners=new Vector(10);
	
	private Object _lock = new Object();

	public AgentImpl(Process process, String name) {
		this(process, name, null);
	}
	
	public AgentImpl(Process process, String name, String type) {
		this(process, name, type, false);
	}
	
	public AgentImpl(Process process, String name, String type, boolean active) {
		_process=process;
		_name=name;
		_type=type;
		_isActive=active;
		process.addProcessListener(this);
		try {
			process.getNode().getConnection().addConnectionListener(this); // Bug 59316
		} catch (InactiveProcessException e) {
		}
	}

	/**
	 * @see Agent#addAgentListener(AgentListener)
	 */
	public void addAgentListener(AgentListener listener) {
		synchronized(_listeners) {
			if(!_listeners.contains(listener)) {
				_listeners.add(listener);
			}
		}
	}

	/**
	 * @see Agent#removeAgentListener(AgentListener)
	 */
	public void removeAgentListener(AgentListener listener) {
		synchronized(_listeners) {
			if(_listeners.contains(listener)) {
				_listeners.remove(listener);
			}	
		}
	}

	/**
	 * @see Agent#getProcess()
	 */
	public Process getProcess() {
		return _process;
	}

	/**
	 * @see Agent#setAutoAttach(boolean)
	 */
	public void setAutoAttach(boolean auto) {
		_autoAttach=auto;
		Process proc=getProcess();
		
		/* If the process is active we need to send our registration to the remote server */
		if(proc!=null) {
			if(proc.isActive()) {
				ControlMessage message=new ControlMessage();
				RegisterAgentInterestCommand command=new RegisterAgentInterestCommand();
				command.setAgentName(_name);
				try {
					command.setProcessId(new Long(proc.getProcessId()).longValue());
					message.appendCommand(command);
					proc.getNode().getConnection().sendMessage(message, new CommandHandler() {
						public void incommingCommand(Node node, CommandElement element) {
							handleCommand(element);
						}
					});
				}
				catch(InactiveProcessException e) {
					return;
				}
				catch(IOException e) {
				}
			}
		}
	}
	
	/**
	 * @see Agent#isAutoAttach(boolean)
	 */
	public boolean isAutoAttach() {
		return _autoAttach;
	}

	/**
	 * @see Agent#attach()
	 */
	public void attach() throws InactiveAgentException, InactiveProcessException{
		/* We will assume the attach is successful here.  This is in fact
		   incorrect but the transport layer must be changed to determine
		   if an attach is in fact successful.
		*/
		_isAttached=true;
	
		ControlMessage message=new ControlMessage();
		AttachToAgentCommand command=new AttachToAgentCommand();
		command.setAgentName(_name);
		command.setProcessId(Long.valueOf(getProcess().getProcessId()).longValue());
		message.appendCommand(command);
		try {
			_process.getNode().getConnection().sendMessage(message, new CommandHandler() {
						public void incommingCommand(Node node, CommandElement command) {
							handleCommand(command);
						}
			});
		}
		catch(IOException e) {
			/* We need to handle this */
		}
	}

	/**
	 * @see Agent#detach()
	 */
	public void detach() throws InactiveAgentException, InactiveProcessException {
		_isAttached=false;
		stopMonitoring();
		_server = null; /* 9707 */
			
		ControlMessage message=new ControlMessage();
		DetachFromAgentCommand command=new DetachFromAgentCommand();
		command.setAgentName(_name);
		command.setProcessId(Long.valueOf(getProcess().getProcessId()).longValue());
		message.appendCommand(command);
		try {
			_process.getNode().getConnection().sendMessage(message, new CommandHandler() {
					public void incommingCommand(Node node, CommandElement command) {
						handleCommand(command);
					}
					
			});
		}
		catch(IOException e) {
			/* We need to handle this */
		}	
	}


	/**
	 * @see Agent#getType()
	 */
	public String getType() {
		return _type;
	}

	/**
	 * @see Agent#getName()
	 */
	public String getName() {
		return _name;
	}
	
	/**
	 * @see Agent#getUUID()
	 */
	public String getUUID() {
		return _uuid;	
	}

	/**
	 * @see Agent#isActive()
	 */
	public boolean isActive() {
		return _isActive;
	}

	/**
	 * @see Agent#isMonitored()
	 */
	public boolean isMonitored() {
		return _isMonitored;
	}
	
	// Bug 54376 begins
	/**
	 * @see Agent#isAttached()
	 */
	public boolean isAttached() {
		return isAttached(false);
	}
	
	/**
	 * @see Agent#isAttached()
	 */
	public boolean isAttached(boolean remote) {
		if(!remote) {
			return _isAttached;
		}

		ControlMessage message = new ControlMessage();
		AgentQueryStateCommand command = new AgentQueryStateCommand();

		synchronized(_lock) {
			try {
				command.setAgentName(_name);
				command.setProcessId(Long.valueOf(getProcess().getProcessId()).longValue());
				message.appendCommand(command);
				_process.getNode().getConnection().sendMessage(message, new CommandHandler() {
					public void incommingCommand(Node node, CommandElement command) {
						switch((int)command.getTag()) {
							case (int)org.eclipse.hyades.internal.execution.local.common.Constants.RA_AGENT_ATTACHED:
								_isAttached = true;
								synchronized(_lock) {
									_lock.notify();
								}
								break;
							case (int)org.eclipse.hyades.internal.execution.local.common.Constants.RA_AGENT_DETACHED:
								_isAttached = false;
								synchronized(_lock) {
									_lock.notify();
								}
								break;
						}
					}
				});
				_lock.wait(5000); // 5 sec wait max
			}
			catch(InactiveProcessException e1) {
				// this should not occur
			}
			catch(IOException e2) {
				// this should not occur
			}
			catch (InterruptedException e3) {
				// this should not occur
			}
		}

		return _isAttached;
	}
	// Bug 54376 ends
	
	/**
	 * @see Agent#startMonitoring()
	 */
	public void startMonitoring(DataProcessor processor) throws InactiveAgentException {
		if(!_isAttached) {
			return;
		}
		if(_server == null) {
			_server= new TCPDataServer();

			/* 9707 */
			try {
				_server.startServer(processor);
			}
			catch(Exception e) {
				/* Need to handle these exceptions */
			}	
		}
		else {
			_server.resumeServer(processor); /* 9707 */
		}

		ControlMessage message=new ControlMessage();
		StartMonitoringRemoteAgentCommand command=new StartMonitoringRemoteAgentCommand();
 		command.setAgentName(_name);
		command.setPort((short)_server.getPort());
		try {
			command.setProcessId(Long.parseLong(_process.getProcessId()));
			message.appendCommand(command);
			_process.getNode().getConnection().sendMessage(message, null);
		}
		catch(InactiveProcessException e) {
			throw new InactiveAgentException();
		}
		catch(IOException e) {
			/* We need to handle this */
		}
		
		_isMonitored=true;
	}
	
	/**
	 * @see Agent#stopMonitoring()
	 */
	public void stopMonitoring() throws InactiveAgentException {
		if(_server!=null) {
			_server.stopServer();
//			_server=null;  /* 9707 */
		}
		if(!_isMonitored) {
			return;
		}
		ControlMessage message=new ControlMessage();
		StopMonitorCommand command=new StopMonitorCommand();
		command.setAgentName(_name);
		try {
			command.setProcessId(Long.parseLong(_process.getProcessId()));
			message.appendCommand(command);
			_process.getNode().getConnection().sendMessage(message, null);
		}
		catch(InactiveProcessException e) {
			throw new InactiveAgentException();
		}
		catch(IOException e) {
			/* We need to handle this */
		}
		_isMonitored=false;
	}
	
	/**
	 * @see Agent#setConfiguration()
	 */
	public void setConfiguration(AgentConfiguration config) {	
		_config=config;
	}
	
	/**
	 * @see Agent#getConfiguration
	 */
	public AgentConfiguration getConfiguration() {
		if(_config==null) {
			_config=new AgentConfiguration();
		}
		return _config;
	}
	
	/**
	 * @see Agent#publishConfiguration
	 */
	public void publishConfiguration() throws InactiveAgentException {
		if(!_isActive) {
			throw new InactiveAgentException();
		}
		if(_config==null) {
			return;
		}
		int entryCount=0;
		ControlMessage message=new ControlMessage();
		synchronized(_config) {
			for(int i=0; i<_config.size(); i++) {
				AgentConfigurationEntry entry=_config.getEntryAt(i);
				if(entry != null  && entry.isEnabled()) {
					entryCount++;
					SetNVPairCommand command=new SetNVPairCommand();
					try {
						command.setProcessId(Long.parseLong(_process.getProcessId()));
					}
					catch(InactiveProcessException e) {
						throw new InactiveAgentException();
					}
					command.setAgentName(_name);
					command.setType(entry.getType());
					command.setName(entry.getName());
					command.setValue(entry.getValue());
					message.appendCommand(command);
				}
			}
		}
		if(entryCount > 0) {
			try {
				_process.getNode().getConnection().sendMessage(message, null);
			}
			catch(InactiveProcessException e) {
				throw new InactiveAgentException();
			}
			catch(IOException e) {
				/* We need to handle this */
			}
		}
	}
	
	/**
	 * @see Agent#invokeCustomCommand()
	 */
	public void invokeCustomCommand(CustomCommand command) throws InactiveAgentException {
		if(!_isActive) {
			throw new InactiveAgentException();
		}
		ControlMessage message=new ControlMessage();
		message.appendCommand(command);
		command.setAgentName(_name);
		try {
			command.setProcessId(Long.parseLong(_process.getProcessId()));
			_process.getNode().getConnection().sendMessage(message, new CommandHandler() {
				public void incommingCommand(Node node, CommandElement element) {
					handleCommand(element);
				}
			});
		}
		catch(InactiveProcessException e) {
			throw new InactiveAgentException();
		}
		catch(IOException e) {
			/* We need to handle this */
		}
	}
	
	public void setUUID(String uuid) {
		_uuid=uuid;	
	}
	
	
	/**
	 * This is the local handler for
	 */
	public void handleCommand(final CommandElement command) {
		switch((int)command.getTag()) {
		case (int)org.eclipse.hyades.internal.execution.local.common.Constants.RA_AGENT_ACTIVE:
			synchronized(_listeners) {
				_isActive=true;
				if(_autoAttach) {
					_isAttached=true;
				}
				Enumeration elements=_listeners.elements();
				while(elements.hasMoreElements()) {
					((AgentListener)elements.nextElement()).agentActive(this);
				}
			}
			break;
		case (int)org.eclipse.hyades.internal.execution.local.common.Constants.RA_AGENT_INACTIVE:
			synchronized(_listeners) {
				_isActive=false;
				Enumeration elements=_listeners.elements();
				while(elements.hasMoreElements()) {
					((AgentListener)elements.nextElement()).agentInactive(this);
				}
			}
			break;
		case (int)org.eclipse.hyades.internal.execution.local.common.Constants.RA_ERROR_STRING:
			synchronized(_listeners) {
				Enumeration elements=_listeners.elements();
				while(elements.hasMoreElements()) {
					((AgentListener)elements.nextElement()).error(this, ((ErrorCommand)command).getErrorId(), ((ErrorCommand)command).getErrorString());
				}
				
			}
			break;
		case (int)org.eclipse.hyades.internal.execution.local.common.Constants.RA_CONTROLLER_REQUEST_MONITOR:
			synchronized(_listeners) {
				/* We need to ensure we have the peer node/process/agent in our model */
				try {
					InetAddress address=((MonitorPeerRequestCommand)command).getPeerNode();
					if(address!=null) {
						final Node node=NodeFactory.createNode(address.getHostAddress());
						
						/* RKD:  We shouldn't do the connect here.  This should be delegated to
						 * the listener so that the proper connection can be opened and the appropriate
						 * authenticationListeners can be added to the connection.
						 */
						if(!node.isConnected()) {
							node.connect(org.eclipse.hyades.internal.execution.local.common.Constants.CTL_PORT_NUM_SERVER);		
						}
						
						final Agent agent=this;
						
						
						/* My initial plan was to call list processes here but that blocks the calling thread
						   until the background processing is done.  The problem with this is that the background thread
						   is this thread..so we deadlock. So the new plan is to create another thread to do the remainder
						   of the notification.
						*/
						
						Runnable delegate=new Runnable() {
							public void run() {
								
								/* Refresh our process list on the node */
								String peerProcessId=Long.toString(((MonitorPeerRequestCommand)command).getProcessId());
								Enumeration processes=null;
						
								try {
									processes=node.listProcesses();
								}
								catch(NotConnectedException e) {
									/* We are the connnection thread so this cannot happen */	
								}
								
								Process proc=null;
								while(processes.hasMoreElements()) {
									proc=(Process)processes.nextElement();
									try {
										if(proc.getProcessId().equals(peerProcessId)) {
											break;
										}
										proc=null;
									}
									catch(InactiveProcessException e) {
										/* Ignore inactive processes */	
									}
									
								}
								
								/* The process may have exited.  Onceagainwe should probably inform the requesting Agent Controller */
								if(proc==null) {
									return;	
								}
								
								Agent peer=proc.getAgent(((MonitorPeerRequestCommand)command).getAgentName());
								
								if(peer==null) {
									return;	
								}
								
								
								
								/* Inform the listeners */
								Enumeration elements=_listeners.elements();
								while(elements.hasMoreElements()) {
									AgentListener listener=((AgentListener)elements.nextElement());
									if(listener instanceof AgentPeerListener) {
										((AgentPeerListener)listener).peerWaiting(agent, peer);		
									}
								}
							}
						};
						
						Thread newThread=new Thread(delegate, "PeerDelegate");
						newThread.start();
					}
				}
				catch(UnknownHostException e) {
					/* We can't resolve the target host, for now we will break.  We may need
					   to send an error message back to the forwarding RAC.
					*/
					break;	
				}
				catch(AgentControllerUnavailableException e) {
					/* We cannot reach the RAC on this machine, this hould only occur as a result
					   of bad name resolution as the RAC requested a connect.
					*/  	
				}
				
			}
			break;
		case (int)org.eclipse.hyades.internal.execution.local.common.Constants.RA_PROCESS_EXITED:
			/* Beacuse of context mappings we may directly hear that this process has exited rather then the command
			 * being routed to the process. This is because context on the server side is held in the agent.
			 */
			 try {
			 	ProcessImpl process=(ProcessImpl)this.getProcess();
			 	process.handleCommand(command);
			 }
			 catch(ClassCastException e) {
			 	/* We can ignore this */	
			 }
			 break;
		default:
			synchronized(_listeners) {
				Enumeration elements=_listeners.elements();
				while(elements.hasMoreElements()) {
					((AgentListener)elements.nextElement()).handleCommand(this, command);
				}
				
			}
		}	
	}
	
	/**
	  * @see ProcessListener#processLaunched
	  */
	public void processLaunched(Process process) {	
	}
	
	/**
	  * @see ProcessListener#processExited
	  */
	public void processExited(Process process) {
		/* It may be that the process dies unexpectantly and the agent believes
	       it is still active.
		*/

		if ( _server != null )
		{
			_server.shutdownServer(); /* 9707 */ /* Close the data port when process exits */
		}

		// Bug 59316 begins
		/*
		 * This has to be done no matter _server exists or not since the client might just
		 * attach to the agent but doesn't start monitoring (so that _server is not created.
		 */
		if(process == _process && _isActive) {
			synchronized(_listeners) {
				_isActive = false;
				Enumeration elements = _listeners.elements();
				while(elements.hasMoreElements()) {
					((AgentListener)elements.nextElement()).agentInactive(this);
				}
			}	
		}
		// Bug 59316 ends
	}

	/**
	 * Determine if this agent is sending data to a profiling file
	 */
	public boolean isToProfileFile()
	{
		return (_profileFile!=null && _profileFile.trim().length()!=0);
	}
	/**
	 * Get the fully quality path of profiling file
	 * @return String
	 */
	public String getProfileFile() {
		return _profileFile;
	}

	/**
	 * Set the fully quality path of profiling file
	 * @param _profileFile The _profileFile to set
	 */
	public void setProfileFile(String _profileFile) {
		this._profileFile = _profileFile;
	}

	// Bug 59316 begins
	public void connectionClosed(Connection connection) {
		synchronized(_listeners) {
			_isActive = false;
			Enumeration elements = _listeners.elements();
			while(elements.hasMoreElements()) {
				((AgentListener)elements.nextElement()).agentInactive(this);
			}
		}	
	}
	// Bug 59316 ends

}

