/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.local.control;

import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.Socket;
import java.net.SocketException;
import java.util.Enumeration;
import java.util.Vector;

import org.eclipse.hyades.internal.execution.local.common.AuthenticationFailedCommand;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.Constants;
import org.eclipse.hyades.internal.execution.local.common.ControlMessage;
import org.eclipse.hyades.internal.execution.local.common.Message;
import org.eclipse.hyades.internal.execution.local.common.ServerSecurityInfoCommand;
import org.eclipse.hyades.internal.execution.security.AuthenticationListener;
import org.eclipse.hyades.internal.execution.security.LoginFailedException;
import org.eclipse.hyades.internal.execution.security.SecureConnectionRequiredException;
import org.eclipse.hyades.internal.execution.security.UntrustedAgentControllerException;

public class ConnectionImpl implements Connection {
    
	protected Socket _socket;
	protected Node _node;
	protected int _port;
	
	private static long _context = 1;
	private ContextMapper _contextMapper = null;
	private CommandHandler _cmdHandler = null;
	private boolean _isComplete = false;
	
	private final Vector _listeners = new Vector();
	private final Vector _authenticationlisteners = new Vector();
	
	private final Object _connectionLock = new Object();
	private final Object _loginLock = new Object();
	
	private static final Object contextLock = new Object();
	
	private boolean _isInitialized=false;

	private SecureConnectionRequiredException _secureConnectionRequiredException;
	private LoginFailedException _loginFailed;
	private boolean _loginPending = false;


	public ConnectionImpl() {
		super();	
	}

	public void connect(Node node, int port)
	    throws IOException,
	            SecureConnectionRequiredException,
	            LoginFailedException,
	            UntrustedAgentControllerException {
		_port=port;
		int offset=0;
		InetAddress[] addrs=node.getAllInetAddresses();

		do {
			/* Connect to the remote machine */
			try {
				_socket=new Socket(addrs[offset], port);
				break;
			}
			catch(IOException e) {
				offset++;
				//System.out.println(e.getMessage());
				if(offset==addrs.length) {
					throw e;
				}
			}
		}while(offset<addrs.length);
		
		_node=node;
		this.init();
	}
	
	
	/* Init */
	protected void init() throws IOException , SecureConnectionRequiredException, LoginFailedException{
		
		/* Set the timeout on the socket reader to be 1 second */
		try {
			_socket.setSoTimeout(1000);
			_socket.setTcpNoDelay(true);
		}
		catch(SocketException e) {
			/* We can ignore this */
		}
		
		_contextMapper=new ContextMapper();
		
		final Connection connection=this;
		
		_cmdHandler=new CommandHandler() {
			public void incommingCommand(Node node, CommandElement command) {
				long contextId=command.getContext();
				
				
				/* Intercept authentication requests */
				switch((int)command.getTag()) {
					case (int)Constants.RA_SERVER_SECURITY_REQUIREMENTS:
						_secureConnectionRequiredException=new SecureConnectionRequiredException((ServerSecurityInfoCommand)command);
						disconnect();
						break;
					case (int)Constants.RA_AUTHENTICATION_FAILED:
						synchronized(_authenticationlisteners) {
							AuthenticationFailedCommand failedCommand=(AuthenticationFailedCommand)command;
							if(node.getUser()!=null && failedCommand.getTicket()==0) {
								try {
									_loginPending=true;
									node.getUser().login(connection);
									break;
								}
								catch(Exception e) {
									/* Swallow the exception and defer to the authentication listeners */
								}	
							}
							
							Enumeration enum=_authenticationlisteners.elements();
							while(enum.hasMoreElements()) {
								if(failedCommand.getTicket()==0) {
									((AuthenticationListener)enum.nextElement()).authenticationRequired(connection);
								}
								else {
									((AuthenticationListener)enum.nextElement()).authenticationFailed(connection);
									
								}
							}
							synchronized(_loginLock) {
								synchronized(_connectionLock) {
									_loginFailed=new LoginFailedException(getPort());
									_loginPending=false;
									disconnect();
									_connectionLock.notifyAll();
									_loginLock.notifyAll();
								}
							}
						}
						break;
					case (int)Constants.RA_AUTHENTICATION_SUCCESSFUL:
						synchronized(_authenticationlisteners) {
							Enumeration enum=_authenticationlisteners.elements();
							while(enum.hasMoreElements()) {
								((AuthenticationListener)enum.nextElement()).authenticationSuccessful(connection);
							}
						}
						synchronized(_loginLock) {
							synchronized(_connectionLock) {
								_loginPending=false;
								_connectionLock.notifyAll();
								_loginLock.notifyAll();
							}
						}
						break;
					default:
						/* Find the command handler associated with this contextId and
						   forward the message appropriately.
						*/
						CommandHandler ch=_contextMapper.getHandler(contextId);
						if(ch != null) {
							//System.out.println("Forwarding to command handler");
							ch.incommingCommand(_node, command);
						}	
						else {
							//System.out.println("Could not find command handler");
						}
				}
				
				
			}
				
		};
		
		/* RKD:  With the V5 Agent Controller it first accepts the socket connection and then determines is the connection
		 * is allowed.  If the connection is not allowed the the connection is cut.  We need to first connect
		 * and then determine if we have been cut off.  The connection has already occurred above us in the
		 * stack.  We then will wait until either a read timeout has occured on the reader thread or the
		 * connection gets cut by the other side.
		 */
		
		synchronized(_connectionLock) {
			SocketReaderThread reader=new SocketReaderThread();
			if(_node.getUser()==null) {
				reader.setName(_node.getName()+"_connection");
			}
			else {
				reader.setName(_node.getName()+"_"+_node.getUser().getName()+"_connection");
			}
			reader.setDaemon(true);
			reader.start();
			
			try {
				_connectionLock.wait();
			}
			catch(InterruptedException e) {
				/* We should ignore this */
			}
			
			_isInitialized=true;
			
			
			/* If this connection has a pending exception because the server is secure we need to throw the
			 * exception so that the caller can create a new connection with the proper security settings.
			 */
			if(_secureConnectionRequiredException!=null) {
				SecureConnectionRequiredException temp=_secureConnectionRequiredException;
				_secureConnectionRequiredException=null;
				_loginPending=false;
				throw temp;
			}
				
				
			/* If we are currently trying to login to the server we should wait until this
			 * login hasa chance to complete.
			 */
			synchronized(_loginLock) {
				if(_loginPending) {
					try {
						_loginLock.wait();
					}
					catch(InterruptedException e) {
						/* We should ignore this */
					}
				}
			}
			
			if(_loginFailed!=null) {
				LoginFailedException temp=_loginFailed;
				_loginFailed=null;
				_loginPending=false;
				throw temp;
			}
			
			if(_isComplete) {
				throw new IOException();	
			}
			
		}
		
	}
	
	public void sendMessage(ControlMessage msg, CommandHandler handler) throws IOException {
	    
		int count=msg.getSize();
		byte[] buffer=new byte[count];

		int commandCount = msg.getCommandCount();
		
		/* 
		 * Need to store the context for rerouting on return, lock is acquired so context
		 * static variable can be incremented safely, synchronized surrounds loop instead
		 * of inside loop to minimize acquisition of locks within loop.
		 * 
		 * The part of the loop that did not deal with incrementing and storing the context
		 * has been extracted out into another loop as to minimize time of lock.
		 */
		synchronized (ConnectionImpl.contextLock) {
		    for(int i=0; i < commandCount; i++) {
		        msg.getCommand(i).setContext(ConnectionImpl._context++);
		    }
		}
		
		/*
		 * The command count loop is repeated outside of the synchronized block to
		 * minimize duration of the context lock.
		 */
		for (int i=0; i < commandCount; i++) {
		    this._contextMapper.addContext(msg.getCommand(i).getContext(), handler);
		}
		
		msg.writeToBuffer(buffer, 0);
		
		OutputStream stream=_socket.getOutputStream();
		stream.write(buffer);
		stream.flush();
		
	}
	
	public void disconnect() {
		synchronized(_connectionLock) {
			if(!_isComplete) {
				_isComplete=true;
				_connectionLock.notifyAll();
				try {
					_socket.close();
				}
				catch(IOException e) {
					/* We can ignore this */
				}
				
				/* If this is a nodeimpl clear the keystore spec so it tries an insecure connect next time */
				if(_node instanceof NodeImpl) {
					((NodeImpl)_node).setSecurityParameters(null);
				}
				
				/* Inform each of the listeners that the connection is closed. */
				synchronized(_listeners) {
					Enumeration e=_listeners.elements();
					while(e.hasMoreElements()) {
						ConnectionListener listener=(ConnectionListener)e.nextElement();
						listener.connectionClosed(this);
					}
				}
			}
		}
	}
	
	public Node getNode() {
		return _node;
	}

	public boolean isActive() {
		if(_isInitialized) {
			return !_isComplete;
		}
		return false;
	}
	
	public int getPort() {
		return _port;
	}
	
	/**
	 * @see Connection#addConnectionListener(ConnectionListener)
	 */
	public void addConnectionListener(ConnectionListener listener) {
		synchronized(_listeners) {
			if(!_listeners.contains(listener)) {
				_listeners.add(listener);
			}	
		}
	}

	/**
	 * @see Connection#removeConnectionListener(ConnectionListener)
	 */
	public void removeConnectionListener(ConnectionListener listener) {
		synchronized(_listeners) {
			if(_listeners.contains(listener)) {
				_listeners.remove(listener);
			}	
		}
	}
	
	/**
	 * @see Connection#addAuthenticationListener(AuthenticationListener)
	 */
	public void addAuthenticationListener(AuthenticationListener listener) {
		synchronized(_authenticationlisteners) {
			if(!_authenticationlisteners.contains(listener)) {
				_authenticationlisteners.add(listener);
			}	
		}
		
	}
	
	/**
	 * @see Connection#removeAuthenticationListener(AuthenticationListener)
	 */
	public void removeAuthenticationListener(AuthenticationListener listener) {
		synchronized(_authenticationlisteners) {
			if(_authenticationlisteners.contains(listener)) {
				_authenticationlisteners.remove(listener);
			}	
		}
	}
	
	
	protected int processControlMessage(byte[] buffer, int offset, int length) {
		if(_cmdHandler != null) {
			ControlMessage msg=new ControlMessage();
			int current=-1;
			try {
				current=msg.readFromBuffer(buffer, offset);
				
				/* If we read more bytes then were valid, return -1 */
				if(current>offset+length) {
					return -1;
				}
			}
			catch(IndexOutOfBoundsException e) {   /* 185463 changed to superclass because both String and Array IndexOutOfBoundsExceptions can be thrown */
				/* We have reached the end of the buffer so we have an incomplete message, return -1 */
				return -1;
			}
			catch(Exception e) {
		            /* There was some other exception reading the message so we can't handle the message.
		               return -1 to indicate an error
		            */
		            return -1;   /* 185463 */
			}

	         /* If we have parsed the message successfully  Then process it */
		    if (current == msg.getLength()+offset) {
				/* Valid pass on each command */
				int count=msg.getCommandCount();
				for(int i=0; i<count; i++) {
					_cmdHandler.incommingCommand(_node, msg.getCommand(i));
				}
			}
			return current;
		}
		return -1;
	}
	
class SocketReaderThread extends Thread implements org.eclipse.hyades.internal.execution.local.common.Constants {
	
	public void run() {
		/* Run forever */
		byte[] buffer=new byte[MAX_MESSAGE_LENGTH];
		int masterOffset=0;
		boolean incompleteMsg;  /* 185463 */
		int timeoutCount=0;

outer:	while(!_isComplete) {
			   incompleteMsg = false;  /* 185463 */
			
			try {
				
				/* Get the InputStream for this socket so we can read some data */
				InputStream inStream=_socket.getInputStream();
				int bytesRead=inStream.read(buffer, masterOffset, buffer.length-masterOffset);
				
				if(bytesRead==-1) {
						break;
				}
				
				
				
/* 185463 begin */					
		        /* Reset bytesRead to be the total message size read from the socket so far
		           (to handle the case where we had an incomplete message)
		        */
		        if (masterOffset > 0) {
		        	bytesRead += masterOffset;
					masterOffset = 0;
		        }
/* 185463 end */					
					
				/* Is this a valid message type */
				Message incommingMessage=new Message();
				
				while(masterOffset<bytesRead) {
					int newOffset=0;
					/* First we try and read the message header information.  It may be
					   that we overrun our buffer or we overrun the number of bytes written.
					   If either is the case, compress and re-read.
					*/
					try {
						newOffset=incommingMessage.readFromBuffer(buffer, masterOffset);
						/* Did we overrun the bytes written? ie do we have an incomplete message header? */
						if(newOffset>bytesRead) {
				                	/* Assumption is that we have processed other parts of the message so masterOffset > 0 */
				                     	/* Compress the buffer and read the socket to get the remainder of the message */
							for(int i=masterOffset, j=0; i<bytesRead; i++, j++) {
								buffer[j]=buffer[i];
							}
							/* 185463 begin */					
							masterOffset=bytesRead-masterOffset;
                     					incompleteMsg = true;
							break;
							/* 185463 end */					
						}
					}
					catch(IndexOutOfBoundsException e) {  /* 185463 */
						/* We overran our buffer so we have an incomplete message.
				                     Compress the buffer and read the socket to get the remainder of the message
				                     Assumption is that we have processed other parts of the message so masterOffset > 0
				                */
						for(int i=masterOffset, j=0; i<bytesRead; i++, j++) {
							buffer[j]=buffer[i];
						}
						/* 185463 begin */					
						masterOffset=bytesRead-masterOffset;
                  				incompleteMsg = true;
						break;
					}
					catch(Exception e) {
						/* Don't know what happened here so let's ignore this message and get the next one */
                  				/* Should log a message here */
						break;
						/* 185463 end */					
					}
					
					/* We have the header information, now lets try and get the entire message.  Once
					   again the same error conditions can occur.
					*/
					if(incommingMessage.getType()!=RA_ACKNOWLEDGEMENT_MESSAGE) {
						newOffset=processControlMessage(buffer, masterOffset, bytesRead);
						/* If there was an error processing the message or we overran the data read from the socket
                     				   then we have an incomplete message so we have to go back and read the rest of the message
                     				   from the socket
                  				*/
						if(newOffset==-1) {
							/* If we processed some of the message  Then  move the remainder of the message
				                           to the beginning of the buffer so we have more room to read from the socket
				                        */
							if(masterOffset>0) {
								for(int i=masterOffset, j=0; i<bytesRead; i++, j++) {
									buffer[j]=buffer[i];
								}
							}
							/* 185463 begin */					
         					/* Else check if we have reached the end of the buffer without getting a complete message */
         					else if (bytesRead == buffer.length) {
		                        /* Allocate a bigger buffer in anticipation of reading the remainder of the message */
		                        int len = buffer.length * 2;
		                        byte[] tmpbuffer=new byte[len];
		                        /* Copy necessary data over to new buffer */
		                        System.arraycopy(buffer, masterOffset, tmpbuffer, 0, bytesRead-masterOffset);
		                        buffer = tmpbuffer;
                 			}
							masterOffset=bytesRead-masterOffset;
                     					incompleteMsg = true;
							break;
							/* 185463 end */					
						}
						masterOffset=newOffset;
					}
				} /* end of inner while */

				/* 185463 begin */
				if (!incompleteMsg) {
					masterOffset=0;
			        }
				/* 185463 end */					
			}
			catch(InterruptedIOException e) {
				/* This happens as the read has timed out.  Just continue as the socket is still valid.  If we are still locked because
				 * we are unsure if the connection is good then we should make sure we allow it to continue.
				 */
				 if(timeoutCount>6 && !_isInitialized) {
				 	synchronized(_connectionLock) {
				 		_connectionLock.notifyAll();
				 	}
				 }
				 timeoutCount++;
				
			}
			catch(SocketException e) {
				/* this is thrown when the RAC cuts our connection because we must support security or if the
				 * host has been denied access
				 */
				
				 break;	
				
			}
			catch(IOException e) {
				break;
			}	
		} /* end of outer while */
		/* The server is now stopping */
		disconnect();
	}
}


}