/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.trace.ui.internal.actions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.internal.ui.PDProjectExplorer;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetsManager;
import org.eclipse.hyades.trace.ui.internal.navigator.ProfileDetailItem;
import org.eclipse.hyades.trace.ui.internal.wizard.OpenMonitorWizard;
import org.eclipse.hyades.trace.ui.internal.wizard.OpenProcessWizard;
import org.eclipse.hyades.trace.ui.internal.wizard.OpenProfAgentWizard;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.hyades.ui.extension.INavigatorContribution;
import org.eclipse.hyades.ui.extension.INavigatorItem;
import org.eclipse.hyades.ui.internal.extension.NavigatorExtensionUtil;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.ui.actions.ActionContext;
import org.eclipse.ui.dialogs.PropertyDialogAction;

/**
 * Group action for the Trace Navigator.
 * 
 * <p>This implementation is based on the 
 * {@link org.eclipse.ui.views.navigator.MainActionGroup} class.
 * 
 */
public class TraceNavigatorActionGroup extends TraceLogActionGroup
{

	protected OpenWizardAction fOpenMonitorAction;	
	protected OpenWizardAction fOpenProfAgentAction;
	protected PropertyDialogAction fPropertyDialogAction;
	protected OpenWizardAction fOpenProcessAction;	
	
	/**
	 * Constructor for NavigatorActionGroup
	 * @param testNavigator
	 */
	public TraceNavigatorActionGroup(INavigator viewer)
	{
		super(viewer);
	}
	
	/**
	 * @see org.eclipse.ui.actions.ActionGroup#dispose()
	 */
	public void dispose()
	{	
		deleteActions();
		super.dispose();
	}
		
		
		private void deleteActions()
		{
		
			if (fOpenMonitorAction != null)
			{
				fOpenMonitorAction.dispose();
				fOpenMonitorAction = null;
			}
			if (fOpenProcessAction != null)
			{
				fOpenProcessAction.dispose();
				fOpenProcessAction = null;
			}
			if (fPropertyDialogAction != null)
			{
				fPropertyDialogAction.dispose();
				fPropertyDialogAction = null;
			}
			
			if (fOpenProfAgentAction != null)
			{
				fOpenProfAgentAction.dispose();
				fOpenProfAgentAction = null;
			}
	}

	/**
	 * @see org.eclipse.ui.actions.ActionGroup#setContext(org.eclipse.ui.actions.ActionContext)
	 */
	public void setContext(ActionContext context)
	{
		super.setContext(context);
		/*
		sorterActionGroup.setContext(context);
		newActionGroup.setContext(context);
		workingSetFilterActionGroup.setContext(context);
		frameListActionGroup.setContext(context); 
		*/
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup#createActions()
	 */
	protected void createActions()
	{
		super.createActions();
				
		fNewSnapshotGrp = new Separator(TraceConstants.PROFILE_NEW_SNAPSHOT_GROUP);
		fGCGroup = new Separator(TraceConstants.PROFILE_GC_GROUP);
		
		fTerminateGrp = new Separator(TraceConstants.PROFILE_TERMINATE_GROUP);
		
		fSaveToFile = new SaveAction(getNavigator());
	
		fOpenMonitorAction =
			new OpenWizardAction(
		(PDProjectExplorer)getNavigator(),
				OpenMonitorWizard.class,
				UIPlugin.getResourceString("PROPERTIES"));
		fOpenProfAgentAction =
			new OpenWizardAction(
		(PDProjectExplorer)getNavigator(),
				OpenProfAgentWizard.class,
				UIPlugin.getResourceString("PROPERTIES"));
		fOpenProcessAction =
			new OpenWizardAction(
		(PDProjectExplorer)getNavigator(),
				OpenProcessWizard.class,
				UIPlugin.getResourceString("PROPERTIES"));
	
		fDeleteAction = new DeleteAction(getNavigator());
		fPropertyDialogAction = new PropertyDialogAction(getNavigator().getViewSite().getShell(), getNavigator().getViewer());
	}

	/**
	 * @see org.eclipse.ui.actions.ActionGroup#fillContextMenu(org.eclipse.jface.action.IMenuManager)
	 */
	public void fillContextMenu(IMenuManager menu)
	{
		super.fillContextMenu(menu);
		
		IStructuredSelection selection = (IStructuredSelection)getContext().getSelection();
		boolean sameClass = false;
		boolean multiSelection = (selection.size() > 1);
		if (multiSelection)
			sameClass = areSameClass(selection);
				
		if (selection.isEmpty()) {
			fImportActionGroup.fillContextMenu(menu);
			menu.appendToGroup(INavigatorContribution.GROUP_REFRESH,fRefreshTree);
			return;
		}
		
		Object sel = selection.iterator().next();
		updateActions(selection);

		if(sel instanceof TRCMonitor)
		{
			menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, new Separator(GROUP_ADD_VIEWS));		
			addViews(menu, sel);
			menu.appendToGroup(INavigatorContribution.GROUP_REFRESH, fRefreshGrp);
		}
		else if(sel instanceof TRCNode)
		{
			menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, new Separator(GROUP_ADD_VIEWS));		
			addViews(menu, sel); 
			menu.appendToGroup(INavigatorContribution.GROUP_REFRESH, fRefreshGrp);
		}
		else if(sel instanceof TRCProcessProxy)
		{
			if (!multiSelection) {
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fGCGroup);
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fTerminateGrp);
				
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, new Separator(GROUP_ADD_VIEWS));		
				addViews(menu, sel);
				menu.appendToGroup(INavigatorContribution.GROUP_REFRESH, fRefreshGrp);
				
			}
	    	
		}
		else if(sel instanceof TRCAgentProxy)
		{
			if (!multiSelection || sameClass) {
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fMonitorGrp);
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fGCGroup);
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fAttachGrp);				
			}
			
			if(!multiSelection)
			{
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fNewSnapshotGrp);
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fTerminateGrp);
				
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, new Separator(GROUP_ADD_VIEWS));		
				addViews(menu, sel);
				menu.appendToGroup(INavigatorContribution.GROUP_REFRESH, fRefreshGrp);
			}
	    	
		}
		else if (sel instanceof INavigatorItem ){
			if (!multiSelection) {
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fMonitorGrp);
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fGCGroup);
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, fAttachGrp);				
				menu.appendToGroup(INavigatorContribution.GROUP_ADDITIONS, new Separator(GROUP_ADD_VIEWS));		
				addViews(menu, sel);
				menu.appendToGroup(INavigatorContribution.GROUP_REFRESH, fRefreshGrp);
			}
		}

		menu.appendToGroup(INavigatorContribution.GROUP_DELETE,fDeleteAction);
		menu.appendToGroup(INavigatorContribution.GROUP_SAVE,fSaveToFile);
		menu.appendToGroup(INavigatorContribution.GROUP_REFRESH,fRefreshTree);
		menu.appendToGroup(INavigatorContribution.GROUP_REFRESH,fRefreshLocal);

		if (!multiSelection) {
			
			if (fPropertyDialogAction.isApplicableForSelection())
				menu.appendToGroup(INavigatorContribution.GROUP_PROPERTIES,fPropertyDialogAction);
	
			if (sel instanceof TRCMonitor) {
				menu.appendToGroup(INavigatorContribution.GROUP_PROPERTIES,fOpenMonitorAction);
			} else if (sel instanceof TRCNode) {
				menu.appendToGroup(INavigatorContribution.GROUP_PROPERTIES,fOpenNodeAction);
			} else if (sel instanceof TRCProcessProxy) {
				menu.appendToGroup(INavigatorContribution.GROUP_PROPERTIES,fOpenProcessAction); 
			} else if (sel instanceof TRCAgentProxy) {
				if (((TRCAgentProxy) sel).getType().equals(TraceConstants.PROFILE_AGENT_TYPE)) {
					menu.appendToGroup(INavigatorContribution.GROUP_PROPERTIES,fOpenProfAgentAction);
				} else {
						menu.appendToGroup(INavigatorContribution.GROUP_PROPERTIES,fOpenLogAgentAction);
				}
			}
		}

		fImportActionGroup.fillContextMenu(menu);		
	}

	/**
	 * Convenience method for adding the given views to
	 * the menu.
	 */
	private void addActions(IMenuManager menu, Collection views) {
		menu.add(new Separator());
		Iterator iter = views.iterator();
		while (iter.hasNext()) {
			Object next = iter.next();
			if (next != null && next instanceof IAction) {
				menu.add((IAction)next);
			}
		}
	}

	/**
	 * Adds all appropriate views to this object's menu.
	 */
	private void addViews(IMenuManager menu, Object obj) {
		if (obj != null && ((obj instanceof EObject) || (obj instanceof INavigatorItem))) {
			MenuManager popup = new MenuManager(UIPlugin.getResourceString("MENU_MANAGER"));
			menu.appendToGroup(GROUP_ADD_VIEWS,popup);
			
			// sorted set
			Set views = new TreeSet(new Comparator() {
				public int compare(Object o1, Object o2) {
					return ((IAction)o1).getText().compareTo(((IAction)o2).getText());
				}
			});
			
			if (obj instanceof TRCMonitor || obj instanceof TRCNode) {
				getViews(obj, views, true, false);
				addActions(popup, views);
				views.clear();
				getViews(obj, views, false, true);
				addActions(popup, views);
			}
			else {
				getViews(obj, views, true, true);
				addActions(popup, views);
			}
		}
	}
	
	/**
	 * Adds all views to the set for the specified object. If
	 * includeSelf is false, it will omit the views for obj. If
	 * includeChildren is true, it will recursively add all views
	 * for all descendants.
	 */
	private void getViews(Object obj, Set set, boolean includeSelf, boolean includeChildren) {
		String type = null;
		Collection children = new ArrayList();
		if (obj instanceof TRCMonitor) {
			type = TraceConstants.MONITOR_TYPE;
			children.addAll(((TRCMonitor)obj).getNodes());
		}
		else if (obj instanceof TRCNode) {
			type = TraceConstants.HOST_TYPE;
			children.addAll(((TRCNode)obj).getProcessProxies());
		}
		else if (obj instanceof TRCProcessProxy) {
			type = null;
			children.addAll(((TRCProcessProxy)obj).getAgentProxies());
		}
		else if (obj instanceof TRCAgentProxy) {
			type = ((TRCAgentProxy)obj).getType();
			
			/*
			 * If there are no profiling types under this agent,
			 * add the views for ALL profiling types. 
			 */
			if (type.equals(TraceConstants.PROFILE_AGENT_TYPE) && !hasProfilingTypes(obj)) {
				Map typesMap = ProfilingSetsManager.instance().getProfilingTypes();
				children = new ArrayList();
				Iterator iter = typesMap.values().iterator();
				while (iter.hasNext()) {
					IProfilingSetType next = (IProfilingSetType)iter.next();
					children.add(new ProfileDetailItem(obj, next.getName(), next.getId(), null));
				}
			}
		}
		else if (obj instanceof INavigatorItem) {
			type = ((INavigatorItem)obj).getType();
		}

		/*
		 * Add all the navigator contribution children..
		 */
		Collection items = null;
		items = NavigatorExtensionUtil.getAllChildren(obj, PDProjectExplorer.ID);
		if (items != null) {
			if (children == null) {
				children = items;
			} else {
				children.addAll(items);
			}
		}
		
		if (includeSelf && type != null) {
			IAction[] actions = getViewMappingForType(type);
			for (int i=0;i<actions.length;++i) {
				set.add(actions[i]);
			}
		}
		if (includeChildren && children != null) {
			Iterator iter = children.iterator();
			while (iter.hasNext()) {
				getViews(iter.next(), set, true, true);
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup#handleKeyPressed(org.eclipse.swt.events.KeyEvent)
	 */
	public void handleKeyPressed(KeyEvent event) {
		if(event.character == SWT.DEL && (event.stateMask == 0)) {
			updateActions(HyadesUtil.getActiveNavigator().getStructuredSelection());
			if(fDeleteAction.isEnabled()) {
				fDeleteAction.run();
			}
		}
	}
}
