/**********************************************************************
 * Copyright (c) 2003, 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.launcher;

import org.eclipse.core.runtime.*;
import org.eclipse.debug.core.*;
import org.eclipse.debug.ui.*;
import org.eclipse.hyades.trace.internal.ui.*;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.ui.internal.core.*;
import org.eclipse.jface.preference.*;
import org.eclipse.swt.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;

public class ProfileHostTab extends AbstractLaunchConfigurationTab
{
	private TraceHostUI _ui;
	private ProfileAgentsTab _agents;

	private String _oldHost;
	private int _oldPort = -1;

	public ProfileHostTab()
	{
		this(null);
	}

	public ProfileHostTab(ProfileAgentsTab agents)
	{
		_agents = agents;
	}

	public void createControl(Composite parent)
	{
		Composite content = new Composite(parent, SWT.NULL);
		setControl(content);
		content.setLayout(new GridLayout());
		createVerticalSpacer(content, 1);

		_ui = new TraceHostUI();
		_ui.createControl(content);
		_ui.addListener(new Listener()
		{
			public void handleEvent(Event event)
			{
				String host = _ui.getHost();
				int port = _ui.getPort();
				
				/*
				 * The agents tab is sensitive to changes in this
				 * tab. If the values changed, the agents tab becomes
				 * invalidated.
				 */
				if (!(_agents == null || _oldPort == -1 || _oldHost == null || (_oldHost.equals(host) && _oldPort == port)))
					_agents.reset();

				_oldHost = host;
				_oldPort = port;

				// the host changed; update
				updateLaunchConfigurationDialog();
				
				/*
				 * #63289: We have no way to know when the user clicked
				 * Apply. The only thing we can do is keep the preferences
				 * up to date as soon as user adds/deletes hosts.
				 */
				updatePreferences();
			}
		});
	}

	public boolean isValid(ILaunchConfiguration conf)
	{
		setErrorMessage(null);
		setMessage(null);

		if (_ui.getHost() == null)
		{
			setErrorMessage(UIPlugin.getResourceString("TAB_ERROR_HOST"));
			return false;
		}

		return true;
	}

	public void setDefaults(ILaunchConfigurationWorkingCopy wc)
	{
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
		wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, "localhost");
		wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, store.getDefaultInt(HyadesConstants.LOCALHOST_PORT));
	}

	public void initializeFrom(ILaunchConfiguration conf)
	{
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();

		try
		{
			String host = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, "localhost");
			int port = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, store.getDefaultInt(HyadesConstants.LOCALHOST_PORT));
			String info = host + ":" + port;

			_ui.addHost(info);
			_ui.selectHost(info);
			_oldHost = host;
			_oldPort = port;
		}
		catch (CoreException e)
		{
			e.printStackTrace();
		}
	}

	public void performApply(ILaunchConfigurationWorkingCopy wc)
	{
		wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, _ui.getHost());
		wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, _ui.getPort());
		
		if (_agents != null)
			_agents.performApply(wc);
	}

	public String getName()
	{
		return UIPlugin.getResourceString("TAB_NAME_HOST");
	}

	public Image getImage()
	{
		return PDPluginImages.getImage(PDPluginImages.IMG_UI_NODE);
	}
	
	public boolean isLocal()
	{
		return _ui.isLocalHost(_ui.getHost());
	}
	
	private void updatePreferences()
	{
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();

		StringBuffer buf = new StringBuffer();
		
		TableItem[] items = _ui.getHosts();
		for(int idx=0; idx<items.length; idx++)	{
			buf.append(',');
			buf.append(items[idx].getText().trim());		
		}

		String hosts = buf.toString();
		if(hosts.startsWith(",") && hosts.length() > 1) {
			hosts = hosts.substring(1);
		}

		store.setValue(HyadesConstants.HOST_KEY, hosts);
	}
}
