/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.view;


import java.util.Iterator;

import org.eclipse.hyades.uml2sd.ui.SDViewerPlugin;
import org.eclipse.hyades.uml2sd.ui.actions.FilterInternalMessages;
import org.eclipse.hyades.uml2sd.ui.actions.GoToCalled;
import org.eclipse.hyades.uml2sd.ui.actions.GoToCaller;
import org.eclipse.hyades.uml2sd.ui.actions.GoToFirstLifelineMessage;
import org.eclipse.hyades.uml2sd.ui.actions.GoToLastLifelineMessage;
import org.eclipse.hyades.uml2sd.ui.actions.GoToNextLifeline;
import org.eclipse.hyades.uml2sd.ui.actions.GoToNextLifelineMessage;
import org.eclipse.hyades.uml2sd.ui.actions.GoToPrevLifeline;
import org.eclipse.hyades.uml2sd.ui.actions.GoToPrevLifelineMessage;
import org.eclipse.hyades.uml2sd.ui.actions.MoveSDDown;
import org.eclipse.hyades.uml2sd.ui.actions.MoveSDLeft;
import org.eclipse.hyades.uml2sd.ui.actions.MoveSDRight;
import org.eclipse.hyades.uml2sd.ui.actions.MoveSDUp;
import org.eclipse.hyades.uml2sd.ui.actions.MoveToMessage;
import org.eclipse.hyades.uml2sd.ui.actions.NextPage;
import org.eclipse.hyades.uml2sd.ui.actions.OpenSDFiltersDialog;
import org.eclipse.hyades.uml2sd.ui.actions.OpenSDFindDialog;
import org.eclipse.hyades.uml2sd.ui.actions.OpenSDPagesDialog;
import org.eclipse.hyades.uml2sd.ui.actions.PrevPage;
import org.eclipse.hyades.uml2sd.ui.actions.ShowNodeEnd;
import org.eclipse.hyades.uml2sd.ui.actions.ShowNodeStart;
import org.eclipse.hyades.uml2sd.ui.actions.Zoom;
import org.eclipse.hyades.uml2sd.ui.actions.provider.IExtendedFilterProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.IExtendedFindProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDAdvancedPagingProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDCollapseProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDExtendedActionBarProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDFilterProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDFindProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDGraphNodeSupporter;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDInternalMesFilterProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDPagingProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDPropertiesProvider;
import org.eclipse.hyades.uml2sd.ui.core.BaseMessage;
import org.eclipse.hyades.uml2sd.ui.core.Frame;
import org.eclipse.hyades.uml2sd.ui.core.GraphNode;
import org.eclipse.hyades.uml2sd.ui.core.SyncMessage;
import org.eclipse.hyades.uml2sd.ui.core.SyncMessageReturn;
import org.eclipse.hyades.uml2sd.ui.load.IUml2SDLoader;
import org.eclipse.hyades.uml2sd.ui.load.LoadersManager;
import org.eclipse.hyades.uml2sd.ui.preferences.SDViewPref;
import org.eclipse.hyades.uml2sd.ztest.OpenToolBox;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.internal.PluginActionContributionItem;
import org.eclipse.ui.part.ViewPart;
import org.eclipse.ui.texteditor.IWorkbenchActionDefinitionIds;
import org.eclipse.ui.views.properties.IPropertySheetPage;


/**
 * @author sveyrier
 *
 */
public class SDView extends ViewPart {
	
	protected static final String ACTION_PREVPAGE = "org.eclipse.hyades.uml2sd.ui.prevpage"; //$NON-NLS-1$
	protected static final String ACTION_NEXTPAGE = "org.eclipse.hyades.uml2sd.ui.nextpage"; //$NON-NLS-1$
	
	protected SDWidget sdWidget = null;
	protected TimeCompressionBar timeCompressionBar = null;
	protected ISDFindProvider sdFindProvider = null;
	protected ISDPagingProvider sdPagingProvider = null;
	protected ISDFilterProvider sdFilterProvider = null;
	protected IExtendedFilterProvider sdExFilterProvider = null;
	protected IExtendedFindProvider sdExFindProvider = null;
	protected ISDInternalMesFilterProvider sdIntFilterProvider = null;
	protected ISDExtendedActionBarProvider sdExtendedActionBarProvider = null;
	protected ISDPropertiesProvider sdPropertiesProvider = null;
	
	protected NextPage nextPageButton = null;
	protected PrevPage prevPageButton = null;
	protected FilterInternalMessages internalFilterButton = null;
	protected Action goToMessageForKeyBinding = null;
	protected Action findForKeyBinding = null;
	
	private SDViewPref pref;
	
	private String extFilterId = null;
	private String extFindId = null;
	
	private MenuManager menuMgr = null;
	
	
	private boolean needInit=true;
	
	public SDWidget getSDWidget()
	{
		return sdWidget;
	}
	
	
	public void createPartControl (Composite c)
	{
		Composite parent = new Composite(c,SWT.NONE);
		GridLayout parentLayout = new GridLayout();
		parentLayout.numColumns=2;
		parentLayout.marginWidth=0;
		parentLayout.marginHeight=0;
		parent.setLayout(parentLayout);
		
		GridData timeLayoutdata = new GridData(GridData.GRAB_VERTICAL|GridData.VERTICAL_ALIGN_FILL);
		timeLayoutdata.widthHint=10;
		GridData seqDiagLayoutData = new GridData(GridData.HORIZONTAL_ALIGN_FILL|GridData.GRAB_HORIZONTAL|
					GridData.GRAB_VERTICAL|GridData.VERTICAL_ALIGN_FILL);
		timeCompressionBar=new TimeCompressionBar (parent, SWT.NONE);
		timeCompressionBar.setLayoutData(timeLayoutdata);
		sdWidget = new SDWidget(parent, SWT.NONE);//SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL);
		sdWidget.setLayoutData(seqDiagLayoutData);
		sdWidget.setSite(this);
		sdWidget.setTimeBar(timeCompressionBar);
		pref = SDViewPref.getInstance();
		
		createCoolbarContent();
		
		goToMessageForKeyBinding = new MoveToMessage(this);
		goToMessageForKeyBinding.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.GoToMessage"); //$NON-NLS-1$
		getSite().getKeyBindingService().registerAction(goToMessageForKeyBinding);
		
		MoveSDUp MoveUp= new MoveSDUp(this);
		MoveUp.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.MoveSDUp");//$NON-NLS-1$
		getSite().getKeyBindingService().registerAction(MoveUp);
		MoveSDDown MoveDown= new MoveSDDown(this);
		MoveDown.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.MoveSDDown");//$NON-NLS-1$
		getSite().getKeyBindingService().registerAction(MoveDown);
		MoveSDLeft MoveLeft= new MoveSDLeft(this);
		MoveLeft.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.MoveSDLeft");//$NON-NLS-1$
		getSite().getKeyBindingService().registerAction(MoveLeft);
		MoveSDRight MoveRight= new MoveSDRight(this);
		MoveRight.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.MoveSDRight");//$NON-NLS-1$
		getSite().getKeyBindingService().registerAction(MoveRight);
		
		findForKeyBinding = new OpenSDFindDialog(this);
		getViewSite().getActionBars().setGlobalActionHandler(
				ActionFactory.FIND.getId(),findForKeyBinding);
		findForKeyBinding.setActionDefinitionId(IWorkbenchActionDefinitionIds.FIND_REPLACE); //$NON-NLS-1$
		getSite().getKeyBindingService().registerAction(findForKeyBinding);
		findForKeyBinding.setEnabled(false);
		
		hookContextMenu();
		
		timeCompressionBar.setVisible(false);
		parent.layout(true);
		
		needInit = restoreLoader();
	}
	
	/**
	 * Load a blank page that is supposed to explain that a kind of interaction must be choosen
	 */
	private void loadBlank()
	{
		IUml2SDLoader l = new IUml2SDLoader() {
			/**
			 * @see org.eclipse.hyades.uml2sd.ui.load.IUml2SDLoader#setViewer(org.eclipse.hyades.uml2sd.ui.view.SDView)
			 */
			public void setViewer(SDView viewer) {
				// Nothing to do
				Frame f = new Frame();
				f.setName(SDViewerPlugin.getResourceString("NO_INTERACTIONS_CHOOSEN")); //$NON-NLS-1$
				viewer.setFrame(f);
			}
			/**
			 * @see org.eclipse.hyades.uml2sd.ui.load.IUml2SDLoader#getTitleString()
			 */
			public String getTitleString() {
				return SDViewerPlugin.getResourceString("NO_INTERACTIONS_CHOOSEN"); //$NON-NLS-1$
			}
			/**
			 * @see org.eclipse.hyades.uml2sd.ui.load.IUml2SDLoader#aboutToBeReplaced()
			 */
			public void aboutToBeReplaced() {
				// Nothing to do
			}
		};
		l.setViewer(this);
		setContentDescription(l.getTitleString());
	}
	
	public void setFocus()
	{
		if (sdWidget != null)
			sdWidget.setFocus();
		if (isViewReady()&&needInit)
		{
			needInit = restoreLoader();
		}
	}
	
	/**
	 * Set the find provider for the opened sequence diagram viewer<br>
	 * If the provider is not set, the find menu item will not be available in the viewer<br>
	 * A find provider is called back when the user perform a find action<br>
	 * The find provider is reponsible to move the sequence diagram to the GraphNode
	 * which match the find criteria as well as to highlight the GraphNode
	 * @param provider the search provider
	 */
	public void setSDFindProvider(ISDFindProvider provider)
	{
		sdFindProvider = provider;
		sdExFindProvider = null;
		createCoolbarContent();
		if (provider!=null)
			findForKeyBinding.setEnabled(true);
		else findForKeyBinding.setEnabled(false);
	}
	
	public void setExtendedFindProvider(IExtendedFindProvider provider)
	{
		sdExFindProvider = provider;
		sdFindProvider = null;
		createCoolbarContent();
		if (provider!=null)
			findForKeyBinding.setEnabled(true);
		else findForKeyBinding.setEnabled(false);
	}
	
	public IExtendedFindProvider getExtendedFindProvider()
	{
		return sdExFindProvider;
	}
	
	public void resetProviders()
	{
		findForKeyBinding.setEnabled(false);
		sdFindProvider = null;
		sdExFindProvider = null;
		sdFilterProvider = null;
		sdExFilterProvider = null;
		sdIntFilterProvider = null;
		sdPagingProvider = null;
		sdExtendedActionBarProvider = null;
		sdPropertiesProvider = null;
		if ((sdWidget!=null)&&(!sdWidget.isDisposed()))
			sdWidget.setCollapseProvider(null);
	}
	
	/**
	 * Set the filter provider for the opened sequence diagram viewer<br>
	 * If the provider is not set, the filter menu item will not be available in the viewer<br>
	 * A filter provider is called back when the user perform a filter action<br>
	 * @param provider the filter provider
	 */
	public void setSDFilterProvider(ISDFilterProvider provider)
	{
		sdFilterProvider = provider;
		sdExFilterProvider = null;
		createCoolbarContent();
	}
	
	/**
	 * Register the given provider to support Drag and Drop collapsing.
	 * This provider is responsible of updating the Frame.
	 * @param provider - the provider to register
	 */
	public void setCollapsingProvider(ISDCollapseProvider provider)
	{
		if ((sdWidget!=null)&&(!sdWidget.isDisposed()))
			sdWidget.setCollapseProvider(provider);
	}
	
	public void setExtendedFilterProvider(IExtendedFilterProvider provider)
	{
		sdExFilterProvider = provider;
		sdFilterProvider = null;
		createCoolbarContent();
	}
	
	public IExtendedFilterProvider getExtendedFilterProvider()
	{
		return sdExFilterProvider;
	}
	
	public void setInternalMessageFilterProvider(ISDInternalMesFilterProvider provider)
	{
		sdIntFilterProvider = provider;
		createCoolbarContent();
	}
	
	public ISDInternalMesFilterProvider getInternalMessageFilterProvider()
	{
		return sdIntFilterProvider;
	}
	
	/**
	 * Set the page provider for the opened sequence diagram viewer<br>
	 * If the sequence diagram provided (see setFrame) need to be split in many parts,
	 * a paging provider must be provided in order to handle page change requested by the user<br>
	 * Set a page provider will create the next and prev page buttons in the viewer coolBar
	 * @param provider the paging provider
	 */
	public void setSDPagingProvider(ISDPagingProvider provider)
	{
		sdPagingProvider = provider;
		createCoolbarContent();
	}
	
	/**
	 * Returns the current page provider for the view
	 * @return the paging provider
	 */
	public ISDPagingProvider getSDPagingProvider()
	{
		return sdPagingProvider;
	}
	
	/**
	 * Returns the current find provider for the view
	 * @return the find provider
	 */
	public ISDGraphNodeSupporter getSDFindProvider()
	{
		return sdFindProvider;
	}
	
	/**
	 * Returns the current filter provider for the view
	 * @return the filter provider
	 */
	public ISDFilterProvider getSDFilterProvider()
	{
		return sdFilterProvider;
	}

	/**
	 * Set the extended action bar provider for the opened sequence diagram viewer<br>
	 * This allow to add programmatically actions in the coolbar and/or in the dropdown menu 
	 * @param provider the search provider
	 */
	public void setSDExtendedActionBarProvider(ISDExtendedActionBarProvider provider)
	{
		sdExtendedActionBarProvider = provider;
		createCoolbarContent();
	}
	
	/**
	 * Returns the current extended action bar provider for the view
	 * @return the extended action bar provider
	 */
	public ISDExtendedActionBarProvider getSDExtendedActionBarProvider()
	{
		return sdExtendedActionBarProvider;
	}
	
	/**
	 * Set the properties view provider for the opened sequence diagram viewer<br>
	 * @param provider the properties provider
	 */
	public void setSDPropertiesProvider(ISDPropertiesProvider provider)
	{
		sdPropertiesProvider = provider;
	}
	
	/**
	 * Returns the current extended action bar provider for the view
	 * @return the extended action bar provider
	 */
	public ISDPropertiesProvider getSDPropertiesProvider()
	{
		return sdPropertiesProvider;
	}
	
	/**
	 * Creates the basic sequence diagram menu
	 *
	 */
	private void hookContextMenu() 
	{
		menuMgr = new MenuManager("#PopupMenu"); //$NON-NLS-1$
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager manager) {
				fillContextMenu(manager);
			}
		});
		Menu menu = menuMgr.createContextMenu(sdWidget.getViewControl());
		sdWidget.getViewControl().setMenu(menu);
		getSite().registerContextMenu(menuMgr, sdWidget.getSelectionProvider());
	}
	
	/**
	 * Returns the context menu manager
	 * @return the menu manager
	 */
	public MenuManager getMenuManager()
	{
		return menuMgr; 
	}
	
	/**
	 * Fills the basic sequence diagram menu
	 * and define the dynamic menu item insertion point
	 * @param manager the menu manager
	 */
	private void fillContextMenu(IMenuManager manager) 
	{
		manager.add(new Separator("Additions")); //$NON-NLS-1$
		if (getSDWidget()!=null&&getSDWidget().currentGraphNode!=null)
		{
			ISelectionProvider selProvider = sdWidget.getSelectionProvider();
			ISelection sel=selProvider.getSelection();
			Object selectedNode= null;
			int nbMessage=0;
			Iterator  it = ((StructuredSelection)sel).iterator();
			while (it.hasNext())
			{
				Object node = it.next();
				if (node instanceof BaseMessage)
					nbMessage++;
			}
			if (nbMessage!=1)
				return;
			GraphNode node = getSDWidget().currentGraphNode;
			if (node instanceof SyncMessageReturn)
				if (((SyncMessageReturn)node).getMessage()!=null)
				{
					Action goToMessage = new MoveToMessage(this);
					goToMessage.setText(SDViewerPlugin.getResourceString("POPUP_MENU_GO_TO_MESSAGE")); //$NON-NLS-1$
					goToMessage.setImageDescriptor(SDViewerPlugin.getResourceImage("hide_image_maplist.gif")); //$NON-NLS-1$
					manager.add(goToMessage);
					goToMessage.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.GoToMessage"); //$NON-NLS-1$
					getSite().getKeyBindingService().registerAction(goToMessage);
				}
			if (node instanceof SyncMessage)
				if (((SyncMessage)node).getMessageReturn()!=null)
				{
					Action goToMessage = new MoveToMessage(this);
					goToMessage.setText(SDViewerPlugin.getResourceString("POPUP_MENU_GO_TO_MESSAGE_RET")); //$NON-NLS-1$
					goToMessage.setImageDescriptor(SDViewerPlugin.getResourceImage("show_image_maplist.gif")); //$NON-NLS-1$
					manager.add(goToMessage);
					goToMessage.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.GoToMessage"); //$NON-NLS-1$
					getSite().getKeyBindingService().registerAction(goToMessage);
				}
		}
		manager.add(new Separator("MultiSelectAdditions")); //$NON-NLS-1$
	}
	
	public void setEnableAction(String actionName, boolean state)
	{
		IActionBars bar = getViewSite().getActionBars();
		if (bar!=null)
		{
			IContributionItem item = bar.getMenuManager().find(actionName);
			if ((item !=null)&&(item instanceof ActionContributionItem))
			{
				IAction action = ((ActionContributionItem)item).getAction();
				if (action != null)
					action.setEnabled(state);
				item.setVisible(state);
				bar.updateActionBars();
			}
		}
	}
	
	/**
	 * Creates the coolBar icon depending on the actions supported by the 
	 * Sequence Diagram provider<br>
	 * - Navigation buttons are displayed if ISDPovider.HasPaging return true<br>
	 * - Navigation buttons are enabled depending on the value
	 * return by ISDPovider.HasNext and HasPrev<br>
	 * @see ISDGraphNodeSupporter Action support definition
	 * @see SDView#setSDFilterProvider(ISDFilterProvider) 
	 * @see SDView#setSDFindProvider(ISDFindProvider)
	 * @see SDView#setSDPagingProvider(ISDPagingProvider)
	 */
	protected void createCoolbarContent()
	{
		IActionBars bar = getViewSite().getActionBars();
		
		bar.getMenuManager().removeAll();
		bar.getToolBarManager().removeAll();

		createMenuGroup();
		
		Zoom resetZoom = new Zoom(this);
		resetZoom.setText(SDViewerPlugin.getResourceString("ACTION_RESET_ZOOM")); //$NON-NLS-1$
		resetZoom.setToolTipText(SDViewerPlugin.getResourceString("ACTION_RESET_ZOOM_TOOLTIP")); //$NON-NLS-1$
		resetZoom.setId("org.eclipse.hyades.uml2sd.ui.actions.ResetZoom");//$NON-NLS-1$
		resetZoom.setImageDescriptor(SDViewerPlugin.getResourceImage("home_nav.gif")); //$NON-NLS-1$
		bar.getMenuManager().appendToGroup("UML2SD_OTHER_COMMANDS",resetZoom);//$NON-NLS-1$
		bar.getToolBarManager().appendToGroup("UML2SD_OTHER_COMMANDS",resetZoom); //$NON-NLS-1$
		
		Zoom noZoom = new Zoom(this);
		noZoom.setText(SDViewerPlugin.getResourceString("ACTION_NO_ZOOM")); //$NON-NLS-1$
		noZoom.setToolTipText(SDViewerPlugin.getResourceString("ACTION_NO_ZOOM_TOOLTIP")); //$NON-NLS-1$
		noZoom.setId("org.eclipse.hyades.uml2sd.ui.actions.NoZoom");//$NON-NLS-1$
		noZoom.setImageDescriptor(SDViewerPlugin.getResourceImage("move.gif")); //$NON-NLS-1$
		bar.getMenuManager().appendToGroup("UML2SD_OTHER_COMMANDS",noZoom);//$NON-NLS-1$
		bar.getToolBarManager().appendToGroup("UML2SD_OTHER_COMMANDS",noZoom); //$NON-NLS-1$
		
		Zoom zoomIn = new Zoom(this);
		zoomIn.setText(SDViewerPlugin.getResourceString("ACTION_ZOOM_IN_NAME")); //$NON-NLS-1$
		zoomIn.setToolTipText(SDViewerPlugin.getResourceString("ACTION_ZOOM_IN_TOOLTIP")); //$NON-NLS-1$
		zoomIn.setId("org.eclipse.hyades.uml2sd.ui.actions.ZoomInCoolBar");//$NON-NLS-1$
		zoomIn.setImageDescriptor(SDViewerPlugin.getResourceImage("zoomin_nav.gif")); //$NON-NLS-1$
		bar.getMenuManager().appendToGroup("UML2SD_OTHER_COMMANDS",zoomIn);//$NON-NLS-1$
		bar.getToolBarManager().appendToGroup("UML2SD_OTHER_COMMANDS",zoomIn); //$NON-NLS-1$
		
		Zoom zoomOut = new Zoom(this);
		zoomOut.setText(SDViewerPlugin.getResourceString("ACTION_ZOOM_OUT_NAME")); //$NON-NLS-1$
		zoomOut.setToolTipText(SDViewerPlugin.getResourceString("ACTION_ZOOM_OUT_TOOLTIP")); //$NON-NLS-1$
		zoomOut.setId("org.eclipse.hyades.uml2sd.ui.actions.ZoomOutCoolBar");//$NON-NLS-1$
		zoomOut.setImageDescriptor(SDViewerPlugin.getResourceImage("zoomout_nav.gif")); //$NON-NLS-1$
		bar.getMenuManager().appendToGroup("UML2SD_OTHER_COMMANDS",zoomOut);//$NON-NLS-1$
		bar.getToolBarManager().appendToGroup("UML2SD_OTHER_COMMANDS",zoomOut); //$NON-NLS-1$
		
		MenuManager navigation=new MenuManager(SDViewerPlugin.getResourceString("NAVIGATION"));//$NON-NLS-1$
		
		GoToCalled goToCalled = new GoToCalled(this);
		goToCalled.setText(SDViewerPlugin.getResourceString("SHOW_CALLED_ACTION_NAME")); //$NON-NLS-1$
		goToCalled.setToolTipText(SDViewerPlugin.getResourceString("SHOW_CALLED_ACTION_DESCRIPTION")); //$NON-NLS-1$
		goToCalled.setId("org.eclipse.hyades.uml2sd.ui.actions.GoToCalled");//$NON-NLS-1$
		goToCalled.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.GoToCalled");//$NON-NLS-1$
		navigation.add(goToCalled);
		getSite().getKeyBindingService().registerAction(goToCalled);
		
		GoToCaller goToCaller = new GoToCaller(this);
		goToCaller.setText(SDViewerPlugin.getResourceString("SHOW_CALLER_ACTION_NAME")); //$NON-NLS-1$
		goToCaller.setToolTipText(SDViewerPlugin.getResourceString("SHOW_CALLER_ACTION_DESCRIPTION")); //$NON-NLS-1$
		goToCaller.setId("org.eclipse.hyades.uml2sd.ui.actions.GoToCaller");//$NON-NLS-1$
		goToCaller.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.GoToCaller");//$NON-NLS-1$
		navigation.add(goToCalled);
		getSite().getKeyBindingService().registerAction(goToCaller);
		
		GoToFirstLifelineMessage goToFL = new GoToFirstLifelineMessage(this);
		goToFL.setText(SDViewerPlugin.getResourceString("SHOW_FIRST_MESS_ACTION_NAME")); //$NON-NLS-1$
		goToFL.setToolTipText(SDViewerPlugin.getResourceString("SHOW_FIRST_MESS_ACTION_DESCRIPTION")); //$NON-NLS-1$
		goToFL.setId("org.eclipse.hyades.uml2sd.ui.actions.GoToFirstLifelineMessage");//$NON-NLS-1$
		goToFL.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.GoToFirstLifelineMessage");//$NON-NLS-1$
		navigation.add(goToFL);
		getSite().getKeyBindingService().registerAction(goToFL);
		
		GoToLastLifelineMessage goToLL = new GoToLastLifelineMessage(this);
		goToLL.setText(SDViewerPlugin.getResourceString("SHOW_LAST_MESS_ACTION_NAME")); //$NON-NLS-1$
		goToLL.setToolTipText(SDViewerPlugin.getResourceString("SHOW_LAST_MESS_ACTION_DESCRIPTION")); //$NON-NLS-1$
		goToLL.setId("org.eclipse.hyades.uml2sd.ui.actions.GoToLastLifelineMessage");//$NON-NLS-1$
		goToLL.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.GoToLastLifelineMessage");//$NON-NLS-1$
		navigation.add(goToLL);
		getSite().getKeyBindingService().registerAction(goToLL);
		
		GoToNextLifelineMessage goToNL = new GoToNextLifelineMessage(this);
		goToNL.setText(SDViewerPlugin.getResourceString("SHOW_NEXT_MESS_ACTION_NAME")); //$NON-NLS-1$
		goToNL.setToolTipText(SDViewerPlugin.getResourceString("SHOW_NEXT_MESS_ACTION_DESCRIPTION")); //$NON-NLS-1$
		goToNL.setId("org.eclipse.hyades.uml2sd.ui.actions.GoToNextLifelineMessage");//$NON-NLS-1$
		goToNL.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.GoToNextLifelineMessage");//$NON-NLS-1$
		navigation.add(goToNL);
		getSite().getKeyBindingService().registerAction(goToNL);
		
		GoToPrevLifelineMessage goToPL = new GoToPrevLifelineMessage(this);
		goToPL.setText(SDViewerPlugin.getResourceString("SHOW_PREV_MESS_ACTION_NAME")); //$NON-NLS-1$
		goToPL.setToolTipText(SDViewerPlugin.getResourceString("SHOW_PREV_MESS_ACTION_DESCRIPTION")); //$NON-NLS-1$
		goToPL.setId("org.eclipse.hyades.uml2sd.ui.actions.GoToPrevLifelineMessage");//$NON-NLS-1$
		goToPL.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.GoToPrevLifelineMessage");//$NON-NLS-1$
		navigation.add(goToPL);
		getSite().getKeyBindingService().registerAction(goToPL);
		
		GoToNextLifeline goToNextLifeline = new GoToNextLifeline(this);
		goToNextLifeline.setText(SDViewerPlugin.getResourceString("SHOW_NEXT_LIFELINE_ACTION_NAME")); //$NON-NLS-1$
		goToNextLifeline.setToolTipText(SDViewerPlugin.getResourceString("SHOW_NEXT_LIFELINE_ACTION_DESCRIPTION")); //$NON-NLS-1$
		goToNextLifeline.setId("org.eclipse.hyades.uml2sd.ui.actions.GoToNextLifeline");//$NON-NLS-1$
		goToNextLifeline.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.GoToNextLifeline");//$NON-NLS-1$
		navigation.add(goToNextLifeline);
		getSite().getKeyBindingService().registerAction(goToNextLifeline);
		
		GoToPrevLifeline goToPrevLifeline = new GoToPrevLifeline(this);
		goToPrevLifeline.setText(SDViewerPlugin.getResourceString("SHOW_PREV_LIFELINE_ACTION_NAME")); //$NON-NLS-1$
		goToPrevLifeline.setToolTipText(SDViewerPlugin.getResourceString("SHOW_PREV_LIFELINE_ACTION_DESCRIPTION")); //$NON-NLS-1$
		goToPrevLifeline.setId("org.eclipse.hyades.uml2sd.ui.actions.GoToPrevLifeline");//$NON-NLS-1$
		goToPrevLifeline.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.GoToPrevLifeline");//$NON-NLS-1$
		navigation.add(goToPrevLifeline);
		getSite().getKeyBindingService().registerAction(goToPrevLifeline);
		
		ShowNodeStart showNodeStart = new ShowNodeStart(this);
		showNodeStart.setText(SDViewerPlugin.getResourceString("SHOW_NODE_START_ACTION_NAME")); //$NON-NLS-1$
		showNodeStart.setToolTipText(SDViewerPlugin.getResourceString("SHOW_NODE_START_ACTION_DESCRIPTION")); //$NON-NLS-1$
		showNodeStart.setId("org.eclipse.hyades.uml2sd.ui.actions.ShowNodeStart");//$NON-NLS-1$
		showNodeStart.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.ShowNodeStart");//$NON-NLS-1$
		navigation.add(showNodeStart);
		getSite().getKeyBindingService().registerAction(showNodeStart);
		
		ShowNodeEnd showNodeEnd = new ShowNodeEnd(this);
		showNodeEnd.setText(SDViewerPlugin.getResourceString("SHOW_NODE_END_ACTION_NAME")); //$NON-NLS-1$
		showNodeEnd.setToolTipText(SDViewerPlugin.getResourceString("SHOW_NODE_END_ACTION_DESCRIPTION")); //$NON-NLS-1$
		showNodeEnd.setId("org.eclipse.hyades.uml2sd.ui.actions.ShowNodeEnd");//$NON-NLS-1$
		showNodeEnd.setActionDefinitionId("org.eclipse.hyades.uml2sd.ui.actions.ShowNodeEnd");//$NON-NLS-1$
		navigation.add(showNodeEnd);
		getSite().getKeyBindingService().registerAction(showNodeEnd);
		
		bar.getMenuManager().appendToGroup("UML2SD_OTHER_COMMANDS",navigation); //$NON-NLS-1$
		
		//Do we need to display a paging item
		if (sdPagingProvider != null)
		{
			nextPageButton = new NextPage(this);
			bar.getToolBarManager().appendToGroup("UML2SD_OTHER_COMMANDS",nextPageButton); //$NON-NLS-1$
			nextPageButton.setEnabled(sdPagingProvider.hasNextPage());
			bar.getMenuManager().appendToGroup("UML2SD_OTHER_COMMANDS",nextPageButton); //$NON-NLS-1$
			
			prevPageButton = new PrevPage(this);
			bar.getToolBarManager().appendToGroup("UML2SD_OTHER_COMMANDS",prevPageButton); //$NON-NLS-1$
			prevPageButton.setEnabled(sdPagingProvider.hasPrevPage());
			bar.getMenuManager().appendToGroup("UML2SD_OTHER_COMMANDS",prevPageButton); //$NON-NLS-1$
		}
		
		if (sdIntFilterProvider !=null)
		{
			internalFilterButton = new FilterInternalMessages(this);
			internalFilterButton.setChecked(sdIntFilterProvider.getInternalMessageFiltered());
			bar.getMenuManager().prependToGroup("UML2SD_FILTERING",internalFilterButton); //$NON-NLS-1$
		}
		
		if (SDViewerPlugin.debug())
		{
			bar.getMenuManager().prependToGroup("UML2SD_FILTERING",new OpenToolBox(this)); //$NON-NLS-1$
			bar.getToolBarManager().prependToGroup("UML2SD_FILTERING",new OpenToolBox(this)); //$NON-NLS-1$
		}
		
		if (sdExFilterProvider != null)
		{
			Action action = sdExFilterProvider.getFilterAction();
			if (action != null)
			{
				if (action.getId()==null)
					action.setId("org.eclipse.hyades.uml2sd.ui.extendedFilter"); //$NON-NLS-1$
				extFilterId = action.getId();
				if (action.getImageDescriptor()==null)
					action.setImageDescriptor(SDViewerPlugin.getResourceImage("filter_obj.gif")); //$NON-NLS-1$
				if (action.getText()==null)
					action.setText(SDViewerPlugin.getResourceString("POPUP_MENU_FILTER"));//$NON-NLS-1$			
				bar.getMenuManager().appendToGroup("UML2SD_FILTERING",action); //$NON-NLS-1$
				bar.getToolBarManager().appendToGroup("UML2SD_FILTERING",action); //$NON-NLS-1$
			}
		}
		else if (sdFilterProvider != null)
		{
			bar.getMenuManager().appendToGroup("UML2SD_FILTERING",new OpenSDFiltersDialog(this, sdFilterProvider));	//$NON-NLS-1$	
			bar.getToolBarManager().appendToGroup("UML2SD_FILTERING",new OpenSDFiltersDialog(this, sdFilterProvider));	//$NON-NLS-1$	
		}
		if (sdPagingProvider != null &&
			sdPagingProvider instanceof ISDAdvancedPagingProvider) {
			IContributionItem sdPaging = bar.getMenuManager().find("org.eclipse.hyades.uml2sd.ui.sdPaging"); //$NON-NLS-1$
			if (sdPaging != null)
			{
				bar.getMenuManager().remove(sdPaging);
				sdPaging = null;
			}
			bar.getMenuManager().appendToGroup("UML2SD_OTHER_COMMANDS",new OpenSDPagesDialog(this, (ISDAdvancedPagingProvider)sdPagingProvider)); //$NON-NLS-1$
			updatePagesMenuItem(bar);
		}
		
		if (sdExFindProvider != null)
		{
			Action action = sdExFindProvider.getFindAction();
			if (action != null)
			{
				if (action.getId()==null)
					action.setId("org.eclipse.hyades.uml2sd.ui.extendedFind"); //$NON-NLS-1$
				extFindId = action.getId();;
				if (action.getImageDescriptor()==null)
					action.setImageDescriptor(SDViewerPlugin.getResourceImage("search_seqdiag_co.gif")); //$NON-NLS-1$
				if (action.getText()==null)
					action.setText(SDViewerPlugin.getResourceString("POPUP_MENU_FIND"));//$NON-NLS-1$	
				bar.getMenuManager().appendToGroup("UML2SD_OTHER_COMMANDS",action); //$NON-NLS-1$
				bar.getToolBarManager().appendToGroup("UML2SD_OTHER_COMMANDS",action); //$NON-NLS-1$
			}
		}
		else if (sdFindProvider != null)
		{
			bar.getMenuManager().appendToGroup("UML2SD_OTHER_COMMANDS",new OpenSDFindDialog(this)); //$NON-NLS-1$
			bar.getToolBarManager().appendToGroup("UML2SD_OTHER_COMMANDS",new OpenSDFindDialog(this)); //$NON-NLS-1$
		}
		
		if (sdExtendedActionBarProvider != null) {
			sdExtendedActionBarProvider.supplementCoolbarContent(bar);
		}
		
		bar.updateActionBars();
	}
	
	/**
	 * Updates the view coolbar buttons state according to the value return by: 
	 * - ISDExtendedActionBarProvider.hasNextPage()<br> 
	 * - ISDExtendedActionBarProvider.hasPrevPage()<br>
	 *
	 */
	public void updateCoolBar()
	{
		if (sdPagingProvider != null)
		{
			IActionBars bar = getViewSite().getActionBars();
			if (bar ==null)
				return;
			IToolBarManager barManager = bar.getToolBarManager();
			if (barManager == null)
				return;
			IContributionItem nextPage = barManager.find(ACTION_NEXTPAGE);
			if (nextPage!= null && nextPage instanceof ActionContributionItem)
			{
				IAction nextPageAction = ((ActionContributionItem)nextPage).getAction();
				if (nextPageAction !=null && nextPageAction instanceof NextPage)
					((NextPage)nextPageAction).setEnabled(sdPagingProvider.hasNextPage());
			}
			
			IContributionItem prevPage = barManager.find(ACTION_PREVPAGE);
			if (prevPage!= null && prevPage instanceof ActionContributionItem)
			{
				IAction prevPageAction = ((ActionContributionItem)prevPage).getAction();
				if (prevPageAction !=null && prevPageAction instanceof PrevPage)
					((PrevPage)prevPageAction).setEnabled(sdPagingProvider.hasPrevPage());
			}
			
			updatePagesMenuItem(bar);
		}
	}
	
	/**
	 * Enables or disables the Pages... menu item, depending on the number of pages
	 * @param bar the bar containing the action
	 */
	protected void updatePagesMenuItem(IActionBars bar) {
		if (sdPagingProvider instanceof ISDAdvancedPagingProvider) {
			IMenuManager menuManager = bar.getMenuManager();
			IAction openSDPagesDialog = ((ActionContributionItem)menuManager.find(OpenSDPagesDialog.ACTIONID_SDPAGING)).getAction();
			if (openSDPagesDialog != null && openSDPagesDialog instanceof OpenSDPagesDialog) {
				openSDPagesDialog.setEnabled(((ISDAdvancedPagingProvider)sdPagingProvider).pagesCount() > 1);
			}
		}
	}


	/**
	 * The frame to render (the sequence diagram)
	 * @param frame the frame to display
	 */
	public void setFrame(Frame frame)
	{
		setFrame(frame,true);
	}
	
	/**
	 * The frame to render (the sequence diagram)
	 * @param frame the frame to display
	 */
	protected void setFrame(Frame frame, boolean resetPosition)
	{
		if (sdWidget==null)
			return;
		
		IUml2SDLoader l = LoadersManager.getLoadersManager().getCurrentLoader(getViewSite().getId(), this);
		if (l != null) {
			setContentDescription(l.getTitleString());
		}
		if (getSDWidget() != null)
			getSDWidget().setFrame(frame, resetPosition);
				
		if (timeCompressionBar != null)	
			timeCompressionBar.setFrame(frame);
		updateCoolBar();
		if (!frame.hasTimeInfo())
		{
			Composite parent=timeCompressionBar.getParent();
			timeCompressionBar.setVisible(false);
			parent.layout(true);
		}
		else 
		{
			Composite parent=timeCompressionBar.getParent();
			timeCompressionBar.setVisible(true);
			parent.layout(true);
		}
		IContributionItem shortKeysMenu = getViewSite().getActionBars().getMenuManager().find("org.eclipse.hyades.uml2sd.ui.actions");//$NON-NLS-1$
		MenuManager shortKeys = (MenuManager)shortKeysMenu;
		if (shortKeys != null) {
			IContributionItem[] items= shortKeys.getItems();
			for (int i=0; i<items.length;i++)
			{
				if (items[i] instanceof PluginActionContributionItem)
				{
					IAction action = ((PluginActionContributionItem)items[i]).getAction();
					if (action !=null)
						action.setEnabled(true);
				}
			}
		}
		createCoolbarContent();
	}
	
	/**
	 * Activate or desactivate the short key command given in parameter (see plugin.xml)
	 * @param id the command id defined in the plugin.xml
	 * @param value the state value
	 */
	public void setEnableCommand(String id,boolean value)
	{
		IContributionItem shortKeysMenu = getViewSite().getActionBars().getMenuManager().find("org.eclipse.hyades.uml2sd.ui.actions");//$NON-NLS-1$
		MenuManager shortKeys = (MenuManager)shortKeysMenu;
		if (shortKeys == null)
			return;
		IContributionItem item = shortKeys.find(id);
		if ((item !=null)&&(item instanceof PluginActionContributionItem))
		{
			IAction action = ((PluginActionContributionItem)item).getAction();
			if (action != null)
				action.setEnabled(value);
		}
	}
	
	/**
	 * Set the frame from an other thread than the one executing the main loop
	 * @param frame
	 */
	public void setFrameSync(final Frame frame) {
		getSDWidget().getDisplay().syncExec(new Runnable() {
			public void run() {
          		setFrame(frame);
			}
		});
	}
	
	/**
	 * Ensure an object is visible from an other thread than the one executing the main loop
	 * @param sm
	 */
	public void ensureVisibleSync(final GraphNode sm) {
		getSDWidget().getDisplay().syncExec(new Runnable() {
			public void run() {
				getSDWidget().ensureVisible(sm);
			}
		});
	}

	/**
	 * Set the frame and ensure an object is visible from an other thread than the one executing the main loop
	 * @param sm
	 */
	public void setFrameAndEnsureVisibleSync(final Frame frame, final GraphNode sm) {
		getSDWidget().getDisplay().syncExec(new Runnable() {
			public void run() {
				setFrameAndEnsureVisible(frame,sm);
			}
		});
	}
	/**
	 * Set the frame and ensure an object is visible
	 * @param sm
	 */
	public void setFrameAndEnsureVisible(Frame frame, GraphNode sm) {
				getSDWidget().clearSelection();
				setFrame(frame,false);
				getSDWidget().ensureVisible(sm);
	}
	
	/**
	 * Set the frame and ensure an object is visible from an other thread than the one executing the main loop
	 * @param sm
	 */
	public void setFrameAndEnsureVisibleSync(final Frame frame, final int x, final int y) {
		getSDWidget().getDisplay().syncExec(new Runnable() {
			public void run() {
				setFrameAndEnsureVisible(frame,x,y);
			}
		});
	}
	/**
	 * Set the frame and ensure an object is visible
	 * @param sm
	 */
	public void setFrameAndEnsureVisible(Frame frame, int x, int y) {
				getSDWidget().clearSelection();
				setFrame(frame,false);
				getSDWidget().ensureVisible(x,y);
				getSDWidget().redraw();
	}

	/**
	 * waitCursor is the cursor to be displayed when long tasks are running
	 */
	private Cursor waitCursor;
	
	/**
	 * Toggle between default and wait cursors from an other thread than the one executing the main loop
	 */
	public void toggleWaitCursorSync(final boolean wait_) {
		getSDWidget().getDisplay().syncExec(new Runnable() {
			public void run() {
				if (getSDWidget() == null ||
						getSDWidget().isDisposed()) {
						return;
					}
				if (wait_) {
					if (waitCursor != null &&
						!waitCursor.isDisposed()) {
						waitCursor.dispose();
					}
					waitCursor = new Cursor(getSDWidget().getDisplay(), SWT.CURSOR_WAIT);
					getSDWidget().setCursor(waitCursor);
					getSDWidget().getDisplay().update();
				} else {
					if (waitCursor != null &&
						!waitCursor.isDisposed()) {
						waitCursor.dispose();
					}
					waitCursor = null;
					getSDWidget().setCursor(null);
					getSDWidget().getDisplay().update();
				}
			}
		});
	}
	
	/**
	 * Return the time compression bar widget
	 * @return the time compression bar
	 */
	public TimeCompressionBar getTimeCompressionBar()
	{
		return timeCompressionBar;
	}
	
	/**
	 * Returns the current Frame (the sequence diagram container)
	 * @return the frame
	 */
	public Frame getFrame()
	{
		if (getSDWidget() != null)
			return getSDWidget().getFrame();
		else return null;
	}
	
	private boolean restoreLoader()
	{
		String id = getViewSite().getId();
//System.err.println("restoreLoader() id="+id);
		if (id == null) {
			return true;
		}
		IUml2SDLoader l = LoadersManager.getLoadersManager().getCurrentLoader(id, this);
//System.err.println("restoreLoader() l="+l);
		if ((l != null)){//&&( LoadersManager.getLoadersManager().getViewer(l)==this)){
			l.setViewer(this);
			return false;
		} 
		else {
			loadBlank();
			return true;
		}
	}
	
	private boolean isViewReady()
	{
		IWorkbenchPage persp = SDViewerPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getActivePage();
		IViewReference[] ref=persp.getViewReferences();
		for (int i=0;i<ref.length;i++)
		{
			if (ref[i].getView(false)==this)
				return true;
		}
		return false;
	}
	
	protected void createMenuGroup()
	{
		IActionBars bar = getViewSite().getActionBars();
		if (bar==null)
			return;
		bar.getToolBarManager().add(new Separator("UML2SD_VIEW_MODES")); //$NON-NLS-1$
		bar.getToolBarManager().add(new Separator("UML2SD_WORKING_SET")); //$NON-NLS-1$
		bar.getToolBarManager().add(new Separator("UML2SD_SORTING")); //$NON-NLS-1$
		bar.getToolBarManager().add(new Separator("UML2SD_FILTERING")); //$NON-NLS-1$
		bar.getToolBarManager().add(new Separator("UML2SD_VIEW_LAYOUT")); //$NON-NLS-1$
		bar.getToolBarManager().add(new Separator("UML2SD_LINK_EDITOR")); //$NON-NLS-1$
		bar.getToolBarManager().add(new Separator("UML2SD_OTHER_COMMANDS")); //$NON-NLS-1$
		bar.getToolBarManager().add(new Separator("UML2SD_OTHER_PLUGINS_COMMANDS")); //$NON-NLS-1$
		bar.getMenuManager().add(new Separator("UML2SD_VIEW_MODES")); //$NON-NLS-1$
		bar.getMenuManager().add(new Separator("UML2SD_WORKING_SET")); //$NON-NLS-1$
		bar.getMenuManager().add(new Separator("UML2SD_SORTING")); //$NON-NLS-1$
		bar.getMenuManager().add(new Separator("UML2SD_FILTERING")); //$NON-NLS-1$
		bar.getMenuManager().add(new Separator("UML2SD_VIEW_LAYOUT")); //$NON-NLS-1$
		bar.getMenuManager().add(new Separator("UML2SD_LINK_EDITOR")); //$NON-NLS-1$
		bar.getMenuManager().add(new Separator("UML2SD_OTHER_COMMANDS")); //$NON-NLS-1$
		bar.getMenuManager().add(new Separator("UML2SD_OTHER_PLUGINS_COMMANDS")); //$NON-NLS-1$
	}
	
	public Object getAdapter(Class _adapter) {
		Object obj = super.getAdapter(_adapter);
		if (sdPropertiesProvider != null && _adapter.equals(IPropertySheetPage.class)) {
			return sdPropertiesProvider.getPropertySheetEntry();
		}

		return obj;
	}
}
