import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.CompletionException;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.EventFactoryHome;
import org.eclipse.hyades.logging.events.cbe.ReportSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext;
import org.eclipse.hyades.logging.events.cbe.util.EventFormatter;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/** 
 * CLASS: HyadesEMFCommonBaseEventv101Sample.java
 * 
 * 
 * DESCRIPTION:  Sample class to demonstrate creating and configuring Common Base Events.
 *               Users experience creating a Common Base Event using the Common Base Event 
 *               XML file template Event Factory and logs the newly created Common Base Event 
 *               to the console.
 * 
 * 
 * ASSUMPTIONS:  1) This class must be executed using JRE 1.4.0 or above.
 *               2) The required JAR files must be specified in the JVM's classpath system variable.
 *               3) JRE 1.4.0 or above must be specified in the PATH environment variable.
 *               4) The 'HyadesEMFCommonBaseEventv101Sample.event.xml.file' environment variable is set to the sample's configuration template's (HyadesEMFCommonBaseEventv101Sample.event.xml) absolute path and name.  For example, &lt;workspace&gt;/HyadesEMFCommonBaseEventv101Sample/config/HyadesEMFCommonBaseEventv101Sample.event.xml 
 *                  where &lt;workspace&gt; is the absolute path to the workbench's workspace. 
 *
 *                 
 * DEPENDENCIES: The following JAR files are required to compile and execute HyadesLoggingJava14Sample.java:
 * 
 * 		         hlcore.jar				-		org.eclipse.hyades.logging.core
 * 		         hlcb101.jar			-		org.eclipse.hyades.logging.core
 *               common.jar             -       org.eclipse.emf.common\runtime
 *               ecore.jar              -       org.eclipse.emf.ecore\runtime
 *
 * 
 * @author	Paul E. Slauenwhite
 * @version	May 6, 2004
 * @since   1.0.1
 * @see		org.eclipse.hyades.logging.events.cbe.CommonBaseEvent 
 * @see		org.eclipse.hyades.logging.events.cbe.EventFactory
 * @see     org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext
 */

public class HyadesEMFCommonBaseEventv101Sample {

	public static void main(String[] args) {

		try {
			
            //Retrieve an EventFactoryHome instance based on the parameter EventFactoryHome classname from the EventFactoryContext:
		    EventFactoryHome eventFactoryHome = EventFactoryContext.getInstance().getEventFactoryHome("RuntimeEventXMLFileEventFactoryHomeImpl");
		    
			//NOTE: Alternatively, create a typed EventFactoryHome instance from the constructor of RuntimeEventXMLFileEventFactoryHomeImpl:
			//Create a typed EventFactoryHome instance:
		    //EventFactoryHome eventFactoryHome = new RuntimeEventXMLFileEventFactoryHomeImpl();

            //Retrieve an EventFactory instance based on the parameter EventFactory name from the EventFactoryHome instance:
	        //NOTE: The 'HyadesEMFCommonBaseEventv101Sample' event XML configuration template file exisits on the classloader's or context classloader's classpath.
		    EventFactory eventFactory = eventFactoryHome.getEventFactory("HyadesEMFCommonBaseEventv101Sample");
		    
			//Create a new instance of a report situation:
			ReportSituation reportSituation = eventFactory.createReportSituation();
			reportSituation.setReasoningScope("INTERNAL");
			reportSituation.setReportCategory("LOG");

			//Create a new instance of a situation:
			Situation situation = eventFactory.createSituation();
			situation.setCategoryName("ReportSituation");
			situation.setSituationType(reportSituation);

			//Create a new instance of a Common Base Event:
			CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();
			commonBaseEvent.setSituation(situation);
			commonBaseEvent.setSeverity(((short) (60)));
			commonBaseEvent.setMsg("Hyades EMF Common Base Event v101 Sample log message");

			//OPTIONAL: Add some ExtendedDataElements:
			commonBaseEvent.addExtendedDataElement("Sample ExtendedDataElement name", "Sample ExtendedDataElement value");

			//Complete the event:
	        if (!eventFactory.getCompleteEvent()) {

	            try {
	            	commonBaseEvent.complete();
	            }
	            catch (CompletionException c) {
	            }
	        }

	        //Log the event:
	        log(commonBaseEvent);

			System.out.println("HyadesEMFCommonBaseEventv101Sample successfully ended!");
		}
		catch (Throwable t) {
			System.out.println("ERROR - HyadesEMFCommonBaseEventv101Sample unsuccessfully ended!");
			System.out.println("REASON: " + t.getMessage());
		}
	}
	
	private static void log(CommonBaseEvent event){
		System.out.println(EventFormatter.toCanonicalXMLString(event,true));
	}
}
