import java.net.InetAddress;
import java.net.UnknownHostException;
import org.eclipse.hyades.logging.core.Guid;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.CompletionException;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.impl.TemplateContentHandlerImpl;

/*******************************************************************************
 * Copyright (c) 2004 Hyades project. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Common
 * Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: IBM - Initial API and implementation
 ******************************************************************************/

/**
 * An extension to the <code>TemplateContentHandlerImpl</code> implementation 
 * to set runtime configuration data on created Common Base Events.
 * <p>
 * 
 * @author   Paul E. Slauenwhite
 * @version  May 7, 2004
 */
public class RuntimeTemplateContentHandlerImpl extends TemplateContentHandlerImpl {

    /**
     * IP address (IPv4) of the local host, otherwise "127.0.0.1". 
     */
    private static String localHostIP = null;

    static {
        try {
            localHostIP = InetAddress.getLocalHost().getHostAddress();
        }
        catch (UnknownHostException u) {
            localHostIP = "127.0.0.1";
        }
    }

    /**
     * @see org.eclipse.hyades.logging.events.cbe.ContentHandler#completeEvent(org.eclipse.hyades.logging.events.cbe.CommonBaseEvent)
     */
    public void completeEvent(CommonBaseEvent event) throws CompletionException {

        //Only attempt to complete events which exist and have not been previously completed:
        if ((event != null) && (!event.isComplete())) {

            //Call the super class' completeEvent() API to populate the event with pre-runtime configuration data (e.g. predefined configuration template):
            //NOTE:  This call MUST appear first in this over-loaded API to preserve precedence rules for merging event properties.
            super.completeEvent(event);

            //If the parameter event's globalInstanceId property is not set, assign a new GUID:
            if (event.getGlobalInstanceId() == null) {
                event.setGlobalInstanceId(Guid.generate());
            }
            
            //If the parameter event's creationTime property is not set, assign the current time:
            if (!event.isSetCreationTime()) {
                event.setCreationTimeAsLong(System.currentTimeMillis());
            }

            //Attempt to retrieve the event's sourceComponentId:
            ComponentIdentification sourceComponetId = event.getSourceComponentId();

            //If the event's sourceComponentId property is set, update several of its properties:
            if (sourceComponetId != null) {

                //If the event's sourceComponentId threadId property is not set, assign the current thread's name:
                if (sourceComponetId.getThreadId() == null) {
                    sourceComponetId.setThreadId(Thread.currentThread().getName());
                }

                //If the event's sourceComponentId location property is not set, assign the local host's IP address:
                if ((sourceComponetId.getLocation() == null) && (sourceComponetId.getLocationType() == null)) {

                    //Set the event's sourceComponentId location property to the local host's IP address:
                    sourceComponetId.setLocation(localHostIP);

                    //Set the event's sourceComponentId locationType property to 'IPV4':
                    sourceComponetId.setLocationType(ComponentIdentification.LOCATION_TYPE_IPV4);
                }
            }            
        }
    }
}
