/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.impl;

import org.eclipse.hyades.logging.adapter.IStatus;

/**
 * Implementation of the IStatus interface to represent the 
 * status of an adapter entity (eg. Component or Context)
 */
public class Status implements IStatus {

	private String name = null;
	private boolean active = false;
	private int itemsProcessedCount = 0;
	private IStatus[] children = null;
	private long startTime = 0;
	private long endTime = 0;
	
	/**
	 * Sets the name of the entity this status is for.
	 * @param name The name to set.
	 */
	public void setName(String name) {
		this.name = name;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.logging.adapter.IStatus#getName()
	 */
	public String getName() {
		return name;
	}

	/**
	 * Sets the active status of the entity
	 * @param active The active status to set.
	 */
	public void setActive(boolean active) {
		this.active = active;
		if (active && startTime == 0) {
			startTime = System.currentTimeMillis();
		}
		else if (!active && startTime != 0 && endTime == 0) {
			endTime = System.currentTimeMillis();
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.logging.adapter.IStatus#isActive()
	 */
	public boolean isActive() {
		if (hasChildren()) {
	    	// Check the children to see if any are active and set the active status accordingly.
	    	active = false;
	    	
	    	for (int i=0; i < children.length; i++) {
	    		if (children[i] != null && children[i].isActive()) {
	    			active = true;
	    			break;
	    		}
	    	}			
		}
		return active;
	}

	/**
	 * Sets the count of the number of items processed by this entity
	 * @param itemsProcessedCount The itemsProcessedCount to set.
	 */
	public void setItemsProcessedCount(int itemsProcessedCount) {
		this.itemsProcessedCount = itemsProcessedCount;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.logging.adapter.IStatus#getItemsProcessedCount()
	 */
	public int getItemsProcessedCount() {
		// If there are children then add up all of the children's processed counts
		if (hasChildren()) {
			int count = 0;
			for (int i=0; i < children.length; i++) {
				if (children[i] != null) {
					if (((Status)children[i]).hasChildren()) { 
						count += children[i].getItemsProcessedCount();
					}
					// For leaf children (ie adapter Components) we only want the count from the last child
					else if (i == children.length-1){
						count += children[i].getItemsProcessedCount();
					}
				}
			}
			itemsProcessedCount = count;
		}
		return itemsProcessedCount;
	}

	/**
	 * Sets the status of children entities
	 * @param children The status of children.
	 */
	public void setChildrenStatus(IStatus[] children) {
		this.children = children;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.logging.adapter.IStatus#getChildrenStatus()
	 */
	public IStatus[] getChildrenStatus() {
		return children;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.logging.adapter.IStatus#getChildStatus(java.lang.String)
	 */
	public IStatus getChildStatus(String name) {
		// If there are children then look for the specified child
		for (int i=0; i < children.length; i++) {
			if (children[i] != null && children[i].getName().equals(name)) {
				return children[i];
			}
		}		
		return null;
	}

	/*
	 * Convenience method to indicate it the adapter entity has children.
	 */
	private boolean hasChildren() {
		if (children != null && children.length > 0) {
			return true;
		}
		return false;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.logging.adapter.IStatus#getElapsedTimeInMilliseconds()
	 */	
	public long getElapsedTimeInMilliseconds() {
		if (startTime != 0 && endTime != 0) {
			return endTime - startTime;
		}
		else if (startTime != 0) {
			return System.currentTimeMillis() - startTime;
		}
		
		return -1;
	}
	
	/**
	 * Get the elpased time since the entity started executing as a float in seconds
	 * @return elapsed time in seconds
	 */
	public float getElpasedTimeInSeconds() {
		long eTime = getElapsedTimeInMilliseconds();
		if (eTime != -1) {
			return (float)eTime/1000;
		}
		
		return -1;
	}
}
