/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.navigator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;
import java.net.MalformedURLException;
import java.net.URL;
import org.eclipse.core.runtime.*;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCOption;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.INavigatorContribution;
import org.eclipse.hyades.ui.extension.INavigatorItem;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.IStructuredSelection;

/**
 * Extends the Profiling Monitor, providing profile detail items under
 * every TRCAgentProxy object if it has profiling detail. 
 */
public class ProfileDetailNavigatorExtension implements INavigatorContribution {

    protected static final String getProfilingTypesExtPoint = "profilingTypes";
    private static Map profilingTypes;
    private Map typeItems = new HashMap();
    
	public boolean hasChildren(Object item) {
		return (item instanceof TRCAgentProxy);
	}
	
	public List getChildren(Object item) {
		//get the profiling details of this agent proxy
		if (item instanceof TRCAgentProxy)
			return getProfileDetailItems((TRCAgentProxy)item);
		return  new ArrayList(0);
	}
	
	// non-persistent items
	public Collection getModifiedItems() {
		return new ArrayList(0);
	}

	public void fillContextMenu(IMenuManager menu, IStructuredSelection selection) {
	}
	
	/**
	 * @param agentProxy
	 * @return a list of ProfileDetailItems for this agent proxy
	 * one Profile Detail Item per profiling type in the configuration
	 */
	private List getProfileDetailItems(TRCAgentProxy agentProxy)
	{
		Vector details = new Vector();
		//if we haven't read the types from the ext points yet, 
		//do that now
		if (profilingTypes == null) initializeProfilingTypes();
		
		List configs = agentProxy.getConfigurations();
		for (int i = 0; i < configs.size(); i++)
		{
			//need to loop through the configurations to find the correct one
			TRCConfiguration config = (TRCConfiguration)configs.get(i);
			List options = config.getOptions();
			for (int opt = 0; opt < options.size(); opt++)
			{
				//need to loop through all the options to find ones 
				//relating to the profiling type
				TRCOption option = (TRCOption)options.get(opt);
				if (option.getKey().startsWith(UIPlugin.getPluginId() + ".type"))
				{
					if (option.getValue().length() >0)
					{
						/* 
						 * Found a profiling type, add it to the list
						 * if it is a known type. If not, ignore it.
						 */
						INavigatorItem profilingType = getProfileDetailItem(agentProxy, option.getValue()); 
						if (profilingType != null) {
							details.add(profilingType);
						}
					}
				}
			}
			//if details size > 0, we found the right config, and can break
			if (details.size() > 0) break;
		}
		return details;
	}
	
	/**
	 * 
	 * @param agentProxy
	 * @param id - the id of the profiling type read from the configuration
	 * @return an instance of a ProfileDetailItem associatied with the
	 * agent proxy sent in, with it's corresponding
	 *  name and icon read from the extension point
	 */
	private ProfileDetailItem createProfileDetailItem(TRCAgentProxy agentProxy, String id)
	{
		IConfigurationElement elem = (IConfigurationElement)profilingTypes.get(id);

		if (elem != null) {
			URL pluginInstallUrl = Platform.getBundle(elem.getDeclaringExtension().getNamespace()).getEntry("/"); //$NON-NLS-1$			
			try {
				URL url = new URL(pluginInstallUrl + elem.getAttribute("icon"));
				return new ProfileDetailItem(agentProxy, elem.getAttribute("name"), elem.getAttribute("id"), url);
			}
			catch(MalformedURLException e) {
				HyadesUIPlugin.logError(e);
			}
		}
		
		return null;
	}
	
	private void initializeProfilingTypes()
	{
	   	//reads the extension points, and creates instances of the GetControlType objects
	   	//specified by the class attribute
	   	profilingTypes = new HashMap();
			
		//find the point
		IExtensionPoint point = Platform.getExtensionRegistry().getExtensionPoint(UIPlugin.PLUGIN_ID, getProfilingTypesExtPoint);

		if (point != null) {
			//find all elements for this extension point
			IConfigurationElement[] elements = point.getConfigurationElements();

			for (int i = 0; i < elements.length; i++) {
					
				IConfigurationElement elem = elements[i];
				String id = elem.getAttribute("id");								
				
				//just in case, make sure the id is not empty
				//puts the id, and configuration element in the map
				if (id.length() != 0)
					profilingTypes.put(id, elem);
			}
		}
	}

	public void dispose() {
	}
	
	private INavigatorItem getProfileDetailItem(TRCAgentProxy agent, String typeID) {
		Object obj = typeItems.get(agent);
		if (obj == null) {
			obj = new HashMap(5);
			typeItems.put(agent, obj);
		}
		
		Map agentTypeItems = (Map)obj;
		Object typeItem = agentTypeItems.get(typeID);
		if (typeItem == null) {
			INavigatorItem item = createProfileDetailItem(agent, typeID);
			if (item != null) {
				agentTypeItems.put(typeID, item);
				return item;
			}
			return null;
		}
		return (INavigatorItem)typeItem;
	}
}
