/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.util;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceProfileTypeGroupUI;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetsManager;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSet;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetTypeGroup;
import org.eclipse.hyades.trace.ui.launcher.IProfilingType;
import org.eclipse.hyades.trace.ui.launcher.ProfilingAttribute;
import org.eclipse.hyades.trace.ui.launcher.ProfilingSetsManagerCopy;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

//This class is responsible for showing the appropriate control when the user clicks on
// an entry in the TreeViewer.
public class ProfilingTypesPageContent extends Composite {

	private StackLayout detailsLayout;
	private Control[] controls;
	private IProfilingSetType[] profilingTypes;
	private Control currentControl;
	private TraceProfileTypeGroupUI _traceProfileTypeGroupUI;
	private Map groupControls;	//a map of the profiling group id, and it's control

	// Constructor.
	public ProfilingTypesPageContent(Composite composite, int style) {
		super(composite, style);
		currentControl = null;
		groupControls = new HashMap();
	}

	public Composite createContent(IProfilingSetType[] types, ProfilingSetsManagerCopy copy) {

		// Create a new stack layout.
		detailsLayout = new StackLayout();
	    this.setLayout(detailsLayout);
    	this.setLayoutData(new GridData(GridData.FILL_BOTH));

		controls = new Control[types.length + 1];
		profilingTypes = types;

		for(int i = 0; i < types.length; i ++) {
			controls[i] = types[i].getProfilingType().createControl(this, copy);
			controls[i].setVisible(false);
		}

		controls[controls.length - 1] = new Composite(this, SWT.NONE);
		controls[controls.length - 1].setVisible(false);

		Map groups = copy.getProfilingTypesGroups();
		Object[] keySet = groups.keySet().toArray();
		_traceProfileTypeGroupUI = new TraceProfileTypeGroupUI();
		
		for (int j = 0; j < keySet.length; j++)
		{
			IProfilingSetTypeGroup pTypeGroup = (IProfilingSetTypeGroup)groups.get(keySet[j]);
			groupControls.put(pTypeGroup.getId(), _traceProfileTypeGroupUI.createControl(this, pTypeGroup.getName(), pTypeGroup.getDescription()));
			
		}
		
		return null;
	}

	public boolean performApply(ProfilingSetsManagerCopy copy)
	{
		IProfilingSet set = copy.getDefaultSet();
		if(set == null)
			return true;
		
		StringBuffer errorMsg = new StringBuffer();			
		List types = set.getProfilingTypes();
		for(int i = 0; i < profilingTypes.length; i++) {
			
			if(types.contains(profilingTypes[i].getId()))
			{
				IProfilingType type = profilingTypes[i].getProfilingType();
				String msg = type.validateConfiguration(copy);
				if(msg != null)
				{
					errorMsg.append(msg).append("\n");
				}
				
			}
		}
		
		if(errorMsg.length() > 0)// conflicts for the current configuration
		{
			String msg = UIPlugin.getResourceString("ERROR_DLG_CONFLICTS_MSG");												
			if(!MessageDialog.openQuestion(getShell(), UIPlugin.getResourceString("ERROR_DLG_CONFLICTS_TITLE"), msg+errorMsg.toString()))
			{
				return false;
			}										
		}
		
		List optionsList = ProfilingSetsManager.getDefaultProfilingOptions();
		for(int i = 0; i < profilingTypes.length; i++) {
			
			if(types.contains(profilingTypes[i].getId()))
			{
				IProfilingType type = profilingTypes[i].getProfilingType();
				ProfilingAttribute[] options = type.getAttributes();
				for(int idx=0; idx<options.length; idx++) {
					optionsList.add(options[idx]);
				}
			}
		}
		
		// remove the duplicate entries (with precedence)
		optionsList = ProfilingSetsManager.filterDuplicateOptions(optionsList);
		
		//apply attributes to the manager copy
		set.getAttributes().clear();
		Map attrs = set.getAttributes();
		for(int idx=0; idx<optionsList.size(); idx++)
		{
			ProfilingAttribute option = (ProfilingAttribute) optionsList.get(idx);
			attrs.put(option.getName(), option);
		}
		
		return true;
	}
			
	// Show the control specified on the screen.
	public void showDetailsFor(Control detail) {
		detailsLayout.topControl = detail;
		currentControl = detail;
		detail.setVisible(true);
			layout();
	}
	
	//show the UI for the profiling group
	public void showGroupDetails(IProfilingSetTypeGroup pTypeGroup) {
		Control newControl = (Control)groupControls.get(pTypeGroup.getId());
		Control currentControl = getCurrentlyDisplayingControl();
		if (newControl.equals(currentControl) == false) {
			showDetailsFor(newControl);
		}
		layout();
	}

	public void showEmptyDetails() {
		detailsLayout.topControl = controls[controls.length - 1];
		currentControl = controls[controls.length - 1];
		controls[controls.length - 1].setVisible(true);
		layout();
	}

	// Dispose all of the controls.
	public void dispose() {
		currentControl = null;
		profilingTypes = null;
		for(int i = 0; i < controls.length; i++) {
			if (controls[i] != null) {
				controls[i].dispose();
			}
		}
		
		Object[] groupControlsArray = groupControls.values().toArray();
		
		for (int j = 0; j < groupControlsArray.length; j++)	{
			if (((Control)groupControlsArray[j]) != null) {
				((Control)groupControlsArray[j]).dispose();
			}
		}
		groupControls.clear();
		groupControls = null;
		controls = null;
		
	}
	
	/**
	 * Returns the control that is currently being displayed in the StackLayout.
	 * Returns null if no control has been displayed.
	 * 
	 * @return the control object currently being displayed.
	 */
	public Control getCurrentlyDisplayingControl() {
		return currentControl;
	}

	public void selectionChanged(SelectionChangedEvent event) {
	}
}

