/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.launch.extensions;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.configuration.CFGArtifactLocationPair;
import org.eclipse.hyades.models.common.configuration.CFGClass;
import org.eclipse.hyades.models.common.configuration.util.ConfigurationUtil;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.launch.extensions.IDeploymentsProvider;

/**
 * This is the default Deployment Provider used by the Test&Deployment tab and by
 * the TestComponent&Deployment tab, when there is no specific extension for the test type
 * that provides a DeploymentsProvider.
 * This provider lists all applicable Deployments to a given Test found in the Workspace.
 * @author jcanches
 */
public class DefaultDeploymentsProvider implements IDeploymentsProvider {
	
	private CFGClass theTestElement;
	private List deploymentsList;
	
	private void findDeployments(Object resource) {
		if (resource instanceof IContainer) {
			IContainer container = (IContainer) resource;
			IResource[] members;
			try {
				members = container.members();
				for (int i = 0; i < members.length; i++) {
					findDeployments(members[i]);
				}
			} catch (CoreException e) {
				TestUIPlugin.logError(e);
			}
		} else if (resource instanceof IFile) {
			IFile file = (IFile) resource;
			if (ICommonConstants.DEPLOYMENT_FILE_EXTENSION.equals(file.getFileExtension())) {
				EObject[] eobjs = EMFUtil.load(null, file);
				for (int i = 0; i < eobjs.length; i++) {
					if (eobjs[i] instanceof TPFDeployment) {
						TPFDeployment deployment = (TPFDeployment)eobjs[i];	
						
						// reload if it's been loaded and modified
						// bugzilla #65607
						if(deployment.eResource().isModified())	{	
							deployment = (TPFDeployment)EMFUtil.reload(deployment);
						}						
						
						if (isDeploymentApplicable(deployment)) {
							deploymentsList.add(deployment);
						}
					}
				}
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.launch.extensions.IDeploymentsProvider#getDeployments(java.lang.Object)
	 */
	public TPFDeployment[] getDeployments(Object testElement) {
		if (testElement == null) return new TPFDeployment[0];
		this.deploymentsList = new ArrayList();
		
		// Add the default deployment (this deployment is not stored in a file)
		this.deploymentsList.add(ConfigurationUtil.createDefaultDeployment());
		
		if (testElement instanceof TPFTestSuite || testElement instanceof TPFTestComponent) {
			theTestElement = (CFGClass)testElement;
		} else if (testElement instanceof TPFTestCase) {
			theTestElement = ((TPFTestCase)testElement).getTestSuite();
		}
		findDeployments(ResourcesPlugin.getWorkspace().getRoot());
		TPFDeployment[] result = new TPFDeployment[deploymentsList.size()];
		return (TPFDeployment[]) deploymentsList.toArray(result);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.launch.extensions.IDeploymentsProvider#getDefaultDeployment(java.lang.Object)
	 */
	public TPFDeployment getDefaultDeployment(Object testElement) {
		// Return the default deployment
		return ConfigurationUtil.createDefaultDeployment();
	}
	
	/**
	 * Return whether the specified deployment is applicable for this.theTestElement.
	 * Applicable means that the Deployment specifies where the Test Element is to be deployed. 
	 * @param deployment
	 * @return
	 */
	private boolean isDeploymentApplicable(TPFDeployment deployment) {
		if (ConfigurationUtil.isDefaultLocation(deployment)) return true;
		CFGArtifactLocationPair pair = ConfigurationUtil.searchPairWithTestAsset(theTestElement, deployment);
		if (pair != null) {
			return pair.getLocation() != null;
		}
		return false;
	}
}
