/*******************************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import org.eclipse.core.resources.IFile;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.navigator.DefaultHyadesProxyNodeFactory;
import org.eclipse.hyades.test.ui.navigator.IFileProxyFactory;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;

/** Generic Proxy Factory for EMF files.
 * It opens the model stored inside the file. 
 * If the model contents only one object, the factory won't create a file level node.
 * An exception is made for execution file which always show the file level even if the file content only one TPFExecutionResult. 
 * Delegation is made for typed object.
 *  
 * @author jgout
 * @since 3.2
 */
public class EMFResourceProxyFactory implements IFileProxyFactory{
		
	/** This method is called to build a node that contains multiple EMF objects as children.
	 *  The object created depends of the extension of the file (.datapool -> DatapoolFileProxyNode)
	 *  using as name the name of the file since if there are multiple objects inside, we can't
	 *  know which one should be used to name the node.
	 * @param file
	 * @return file level proxy node
	 */
	private FileProxyNode createFileProxy(IFile file, Resource resource) {
		String extension = file.getFileExtension();
		String fileName = file.getName();
		
		//- remove the extension from the file name 
		fileName = fileName.substring(0, fileName.length() - extension.length() - 1);
		if(extension != null) {
			if(extension.equals(ICommonConstants.ARTIFACT_FILE_EXTENSION)) {
				return new ArtifactFileProxyNode(file, resource);
			} else if(extension.equals(ICommonConstants.DATAPOOL_FILE_EXTENSION)) {
				return new DatapoolFileProxyNode(file, resource);
			} else if(extension.equals(ICommonConstants.LOCATION_FILE_EXTENSION)) {
				return new LocationFileProxyNode(file, resource);
			} else if(extension.equals(ICommonConstants.DEPLOYMENT_FILE_EXTENSION)) {
				return new DeploymentFileProxyNode(file, resource);
			} else if(extension.equals(ICommonConstants.TEST_SUITE_FILE_EXTENSION)) {
				return new TestSuiteFileProxyNode(file, resource);
			}  else if(extension.equals(ICommonConstants.TEST_COMPONENT_EXTENSION)) {
				return new TestComponentFileProxyNode(file, resource);
			}  else if(extension.equals(ICommonConstants.EXECUTION_FILE_EXTENSION)) {
				return new ExecutionFileProxyNode(file, resource);
			} else return null;
		} else return null;
	}

	//- this method load the resource associated to the given file using the test navigator resource set
	private Resource getEMFResource(IFile file) {
		URI uri = URI.createPlatformResourceURI(file.getFullPath().toString());
		ResourceSet rs = EMFUtil.getResourceSet();
		return rs.getResource(uri, true);
	}
	
	/** Create a proxy node (which can be a full tree of proxies) from the given file.
	 *  The file is supposed to be a EMF file.
	 *  This method loads the content of the file and unload the resource at the end.
	 * @param file the file to convert
	 * @retrun the root of the proxy node tree
	 */
	public IProxyNode create(IFile file) {
		IProxyNode res = null;
		Resource resource = getEMFResource(file);
		EObject[] eObjects;
		if(resource != null) {
			eObjects = (EObject[]) resource.getContents().toArray(new EObject[resource.getContents().size()]); 
			if(eObjects.length == 0) {
				//- nothing in this file, just unload
			} else if (eObjects.length == 1 && !ICommonConstants.EXECUTION_FILE_EXTENSION.equals(file.getFileExtension())) {
				//- only one root object, so create proxy for this object 
				res = DefaultHyadesProxyNodeFactory.getInstance().create(eObjects[0], file.getParent());
			} else {
				//- there are multiple root objects, create a file proxy level and a content level
				//- create the root proxy (file node)
				res = createFileProxy(file, resource);
			}
			resource.unload();
		}
		return res;
	}
}
