/*******************************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.util.ArrayList;
import java.util.HashMap;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.test.ui.TestUIPlugin;

/**
 * @author jgout
 * @since 3.2
 */
public class FileFactoryManager {
	
	private static FileFactoryManager instance;
	private HashMap factories;
	
	public static FileFactoryManager getInstance() {
		if(instance == null) {
			instance = new FileFactoryManager();
		}
		return instance;
	}
	
	private FileFactoryManager() {
		factories = new HashMap();
		IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(TestUIPlugin.getID() + ".testNavigatorFileFactory"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] members = extPoint.getConfigurationElements();
			for (int i = 0; i < members.length; i++) {
				if (members[i].getName() != null) {
					//- get extension of files we want to make visible
					IConfigurationElement element = members[i];
					String extension = element.getAttribute("extension"); //$NON-NLS-1$
					if(extension != null && extension.length() != 0) {
						addFactory(extension, element);
					}
				}
			}
		}		
	}
	
	private void addFactory(String extension, IConfigurationElement element) {
		if(!factories.containsKey(extension)) {
			//- if the given extension is not yet in the internal map, add it
			ArrayList l = new ArrayList();
			l.add(element);
			factories.put(extension, l);
		} else {
			//- extension already registered
			ArrayList fac = (ArrayList)factories.get(extension);
			if(!fac.contains(element)) {
				fac.add(element);
			}
		}
	}
	
	/** Returns the list of factories registered to the given file extension
	 *  Caller should manage the choice of the factory to use.
	 *  FileProxyManager is the main caller of this method.
	 *  The rule is that if a factory is not the right one for a file it should return null.
	 *  That way, caller can iterate until it finds a not null result of the create method.
	 * 
	 * @param extension the file name extension 
	 * @return the list of factories, this list can be empty if extension is null or empty
	 */
	public ArrayList getFactories(String extension) {
		if(extension == null || extension.length() == 0) return new ArrayList();
		if(factories.containsKey(extension)) {
			ArrayList l = (ArrayList)factories.get(extension);
			//- check if all factories have been loaded
			for (int i = 0; i<l.size(); i++) {
				Object value = l.get(i);
				if (value instanceof IConfigurationElement) {
					//- if the factory has not yet been loaded
					IConfigurationElement element = (IConfigurationElement) value;
					try {
						//- replace the configurationElement by the loaded factory
						l.set(i,element.createExecutableExtension("factory")); //$NON-NLS-1$
					} catch (CoreException e) {
						//- remove this element since we can not create a factory from it
						l.remove(i);
					} 
				}
			}
			//- return factories
			return l;
		} else {
			return new ArrayList();
		}
	}

	/** Return whether the given file extension has been registered 
	 * @param fileExtension
	 * @return true if the given file extension has been registered yet and false otherwise
	 */
	public boolean isRegistered(String fileExtension) {
		return factories.containsKey(fileExtension);
	}
}
