/*******************************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.util.HashMap;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.navigator.DefaultHyadesProxyNodeFactory;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITypedElementProxyFactory;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.internal.extension.AssociationMappingRegistry;

/**
 * @author jgout
 * @since 3.2
 */
public class TypedElementFactoryManager {
	
	private static TypedElementFactoryManager instance;
	private HashMap factories;
	
	/**
	 * Factory to used to create proxy from unknown type element.
	 * Methods use the default public hyades factory
	 * @author jgout
	 */
	class DefaultTypedElementFactory implements ITypedElementProxyFactory {
		
		public IProxyNode create(TPFExecutionResult result, Object parent) {
			return DefaultHyadesProxyNodeFactory.getInstance().create(result, parent);
		}
		
		public IProxyNode create(TPFTest test, Object parent) {
			return DefaultHyadesProxyNodeFactory.getInstance().create(test, parent);
		}
		
		public IProxyNode create(TPFTestComponent component, Object parent) {
			return DefaultHyadesProxyNodeFactory.getInstance().create(component, parent);
		}
	}
	
	public static TypedElementFactoryManager getInstance() {
		if(instance == null) {
			instance = new TypedElementFactoryManager();
		}
		return instance;
	}
	
	private TypedElementFactoryManager() {
		factories = new HashMap();
		IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(TestUIPlugin.getID() + ".testNavigatorTypedElementFactory"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] members = extPoint.getConfigurationElements();
			for (int i = 0; i < members.length; i++) {
				if (members[i].getName() != null) {
					//- get extension of files we want to make visible
					IConfigurationElement element = members[i];
					String typeName = element.getAttribute("type"); //$NON-NLS-1$
					if(typeName != null && typeName.length() != 0) {
						addFactory(typeName, element);
					}
				}
			}
		}		
	}
	
	private void addFactory(String typeName, IConfigurationElement element) {
		if(factories.containsKey(typeName)) {
			//- if the given type is not yet in the internal map, add it
			factories.put(typeName, element);
		}
	}
	
	public ITypedElementProxyFactory getFactory(String typeName) {
		if(factories.containsKey(typeName)) {
			Object value = factories.get(typeName);
			if (value instanceof IConfigurationElement) {
				//- if the factory has not yet been loaded
				IConfigurationElement element = (IConfigurationElement) value;
				try {
					ITypedElementProxyFactory factory = (ITypedElementProxyFactory)element.createExecutableExtension("factory"); //$NON-NLS-1$
					//- store this new factory for the next times
					factories.put(typeName, factory);
					return factory;
				} catch (CoreException e) {
					TestUIPlugin.logError(e);
					//- problem building the instance of factory using reflection
					return new DefaultTypedElementFactory();
				} 
			} else {
				//- the factory has been loaded yet
				if (value instanceof ITypedElementProxyFactory) {
					return (ITypedElementProxyFactory) value;
				} else {
					TestUIPlugin.logError("internal error"); //$NON-NLS-1$
					//- problem with stored factory (this should never appear)
					return null;
				}
			}
		} else {
			//- no factory registered for this type
			//- look for this type as a test suite type first (declared by hyades.ui.typeDescriprions extension point)
			AssociationMappingRegistry registry = (AssociationMappingRegistry)TestUIExtension.getTestSuiteMappingRegistry();
			IAssociationDescriptor descriptor = registry.getAssociationMapping(HyadesUIPlugin.EP_TYPE_DESCRIPTIONS).getDefaultAssociationDescriptor(typeName);
			if(descriptor == null) {
				//- not a test suite type, is it a test case type ?
				registry = (AssociationMappingRegistry)TestUIExtension.getTestCaseMappingRegistry();
				descriptor = registry.getAssociationMapping(HyadesUIPlugin.EP_TYPE_DESCRIPTIONS).getDefaultAssociationDescriptor(typeName);
			}
			if(descriptor != null) {	
				//- but typeDescriptions extension found, use default factory
				return new DefaultTypedElementFactory();
			} else {
				//- unknown type
				return null;
			}
		}
	}
}
