/*******************************************************************************
 * Copyright (c) 2003, 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.wizard.selection;

import java.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.test.ui.dialog.EObjectResourceSelectionDialog;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.IWizard;
import org.eclipse.jface.wizard.IWizardNode;
import org.eclipse.jface.wizard.WizardSelectionPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWizard;
import org.eclipse.hyades.test.ui.TestUIPlugin;

/**
 * This is a {@link org.eclipse.jface.wizard.WizardSelectionPage} that selects one or more resources
 * with a certain file extension in the workspace.
 * 
 * @author bjiang
 * @since  1.3
 */
public class WizardResourceSelectionPage extends WizardSelectionPage implements SelectionListener, ModifyListener
{
	private IWorkbench workbench;
	private IStructuredSelection structuredSelection;
	private IWizardNode newFileWizardNode;
	private String fileExtension;
	private Image fileImage;
	private Button useExistButton;
	private Button createNewButton;
	private Button browseButton;
	private Text existFileText;
	private boolean isValid;
	private EObject[] selectedEObjects;
	
	/**
	 * @param pageName
	 */
	public WizardResourceSelectionPage(String pageName, IWorkbench workbench, IStructuredSelection structuredSelection, String fileExtension, Image fileImage, IWizardNode newFileWizardNode)
	{
		super(pageName);
		this.workbench = workbench;
		this.structuredSelection = structuredSelection;
		this.fileExtension = fileExtension;
		this.fileImage = fileImage;
		this.newFileWizardNode = newFileWizardNode;
		isValid = false;
		setTitle(TestUIPlugin.getString("DLG_RESOURCE_SEL")); //$NON-NLS-1$
		setMessage(TestUIPlugin.getString("DLG_RESOURCE_DSC")); //$NON-NLS-1$
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent)
	{
		Group group = new Group(parent, SWT.SHADOW_NONE);
		group.setLayout( new GridLayout(1, false) );
		
		useExistButton = new Button(group, SWT.RADIO);
		useExistButton.setText(TestUIPlugin.getString("DLG_RESOURCE_USE_EXST")); //$NON-NLS-1$
		useExistButton.setSelection(true);
		useExistButton.addSelectionListener(this);

		Composite browseContainer = new Composite(group, SWT.NONE);
		browseContainer.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		GridLayout browseLayout = new GridLayout();
		browseLayout.numColumns = 3;
		browseContainer.setLayout(browseLayout);
				
		Label label = new Label(browseContainer, SWT.NONE);
		label.setText(TestUIPlugin.getString("DLG_RESOURCE_EXST")); //$NON-NLS-1$
		existFileText = new Text(browseContainer, SWT.BORDER);
		existFileText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		//existFileText.setEditable(false);
		existFileText.addModifyListener(this);
		browseButton = new Button(browseContainer, SWT.PUSH);
		browseButton.setText(TestUIPlugin.getString("BTN_BROWSE")); //$NON-NLS-1$
		browseButton.addSelectionListener(this);
		
		createNewButton = new Button(group, SWT.RADIO);
		createNewButton.setText(TestUIPlugin.getString("DLG_RESOURCE_NEW")); //$NON-NLS-1$
		createNewButton.setLayoutData(new GridData(GridData.GRAB_HORIZONTAL));
		createNewButton.addSelectionListener(this);
		
		setControl(group);
	}
	
	protected IWizardNode getNewFileWizardNode()
	{
		if((newFileWizardNode != null) && (!newFileWizardNode.isContentCreated()))
		{
			IWizard wizard = newFileWizardNode.getWizard();
			if(wizard instanceof IWorkbenchWizard)
			{
				((IWorkbenchWizard)wizard).init(workbench, structuredSelection);
				wizard.addPages();
			}
		}
		return newFileWizardNode;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e)
	{
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e)
	{
		if(e.widget == useExistButton)
		{
			if(isValid() == true)
				setPageComplete(true);
			existFileText.setEnabled(true);
			browseButton.setEnabled(true);
			browseButton.setFocus();
			super.setSelectedNode(null);
		}
		else if(e.widget == createNewButton)
		{
			setPageComplete(false);
			existFileText.setEnabled(false);
			browseButton.setEnabled(false);
			super.setSelectedNode(getNewFileWizardNode());
		}
		else if(e.widget == browseButton)
		{
		    EObjectResourceSelectionDialog fileDialog = new EObjectResourceSelectionDialog(Display.getCurrent().getActiveShell(), TestUIPlugin.getString("DLG_RESOURCE_SEL"), new String[]{fileExtension}, fileImage); //$NON-NLS-1$
			int ret = fileDialog.open();
			if(ret == Window.OK)
			{
				this.selectedEObjects = fileDialog.getSelectedObjects();
				
				IFile[] files = fileDialog.getSelectedFiles();			
				if(files != null && files.length > 0)
				{
					String selectedFiles = ""; //$NON-NLS-1$
					for(int i = 0; i < files.length; i++)
					{
						selectedFiles += files[i].getFullPath().toString();
						if(i < (files.length - 1))
							selectedFiles += ", "; //$NON-NLS-1$
					}
					existFileText.removeModifyListener(this);
					existFileText.setText(selectedFiles);
					isValid = true;
					setPageComplete(true);
					existFileText.addModifyListener(this);
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
	 */
	public void modifyText(ModifyEvent e)
	{
		if(e.widget == existFileText)
		{
//			validate
			String existFile = existFileText.getText();			
			if(existFile != null && existFile.length() > 0)
			{			
				IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
				Vector objects = new Vector(); 				
				StringTokenizer tokenizer = new StringTokenizer(existFile, ","); //$NON-NLS-1$
				String path;
				while(tokenizer.hasMoreTokens())
				{
					path = tokenizer.nextToken();
					IResource resource = root.findMember(path);
					if(resource == null || !resource.exists() || !fileExtension.equals(resource.getFileExtension()))
					{
						setErrorMessage(path + TestUIPlugin.getString("_ERROR_DLG_RESOURCE_INVLD")); //$NON-NLS-1$
						break;
					}
					else			
						setErrorMessage(null);
					
					URI uri = URI.createPlatformResourceURI(((IFile)resource).getFullPath().toString());
					EObject[] eObjects = EMFUtil.getEObjects(uri, true);
					if(eObjects != null || eObjects.length > 0)
					{
						for(int i = 0; i < eObjects.length; i++)
						{
							objects.add(eObjects[i]);
						}
					}					
				}
				
				if(objects.size() > 0)
				{
					selectedEObjects = new EObject[objects.size()];
					for(int i = 0; i < objects.size(); i++)
						selectedEObjects[i] = (EObject)objects.get(i);
					isValid = true;
					setPageComplete(true);
					return;
				}
			}

			selectedEObjects = null;
			isValid = false;
			setPageComplete(false);
		}
	}
	
	private boolean isValid()
	{
		return isValid;
	}

	public EObject[] getSelectedEObjects()
	{
		if(selectedEObjects == null)
			selectedEObjects = new EObject[0];
		return this.selectedEObjects;
	}
}
