/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.launch.tabGroups;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTabGroup;
import org.eclipse.debug.ui.CommonTab;
import org.eclipse.debug.ui.ILaunchConfigurationDialog;
import org.eclipse.debug.ui.ILaunchConfigurationTab;
import org.eclipse.hyades.test.ui.internal.launch.tabs.ExecutionHistoryTab;
import org.eclipse.hyades.test.ui.internal.launch.tabs.TestAndDeploymentTab;
import org.eclipse.hyades.test.ui.internal.launch.tabs.TestComponentAndDeploymentTab;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfileTab;

/**
 * <p>This class provides an abstract TabGroup for the all Hyades Test Launch Configurations.
 * A Hyades Tab Group is organized in three to five parts:<br>
 * 1/ A Runnable Item Selection Tab. This tab allows to select which is to be run.*<br>
 * 2/ [optional] Any additional tabs providing options specifying how the runnable item will be run<br>
 * 3/ [optional] The Hyades Profiling Tab (if the selected run mode is Profile)
 * 4/ The Execution History Tab, that specifies where the Execution History model will
 * be produced.<br>
 * 5/ The Eclipse Common Tab.<br></p>
 * <p>This class is an abstract implementation that provides parts 3, 4 and 5.<br>
 * Subclasses have to provide part 1 by implementing <code>addRunnableItemSelectionTab(List tabList)</code>.
 * They also have to provide Part 2 by implementing <code>addAdditionalTabs(List tabList)</code>.<br></p>
 * <p>If the common Tab layout provided above is not desired, subclasses may override the
 * <code>createTabs(ILaunchConfigurationDialog, String)</code> in order to create tabs at their
 * convenience. This class provides methods for adding part 3, 4 and 5 to a list.</p>
 * @see AbstractLaunchConfigurationTabGroup
 * @author jcanches
 */
public abstract class AbstractTabGroup extends AbstractLaunchConfigurationTabGroup {

	/**
	 * Creates the tabs. This method can be redefined to create a customized set of tabs.
	 * @see AbstractLaunchConfigurationTabGroup#createTabs
	 */
	public void createTabs(ILaunchConfigurationDialog dialog, String mode) {
		List tabs = new ArrayList(5);
		
		this.addRunnableItemSelectionTab(tabs, mode);
		this.addExecutionHistoryTab(tabs);
		this.addAdditionalTabs(tabs, mode);
		if (this.isProfilingMode(mode)) {
			this.addProfilingTab(tabs, getProfilingAttachOption());
		}
		this.addCommonTab(tabs);
		
		ILaunchConfigurationTab[] array = new ILaunchConfigurationTab[tabs.size()];
		tabs.toArray(array);
		setTabs(array);
	}
	
	/**
	 * Is the supplied mode the Profiling mode.
	 * @param mode
	 * @return
	 */
	protected boolean isProfilingMode(String mode) {
		return mode.equals(ILaunchManager.PROFILE_MODE);
	}
	
	/**
	 * Is the supplied mode the Debug mode.
	 * @param mode
	 * @return
	 */
	protected boolean isDebugMode(String mode) {
		return mode.equals(ILaunchManager.DEBUG_MODE);
	}
	
	/**
	 * Add to the tab list the Hyades Execution History Tab.
	 * @param tabList
	 */
	protected void addExecutionHistoryTab(List tabList) {
		tabList.add(new ExecutionHistoryTab(getLaunchConfigurationType()));
	}
	
	/**
	 * Add to the tab list the Hyades Profiling Tab.
	 * The tab is added with attaching option enabled.
	 * @param tabList The list the tab is to be added to.
	 * @param attach Provides the attach option of the Profiling tab.
	 * @see org.eclipse.hyades.trace.ui.internal.launcher.ProfileTab#ProfileTab(boolean)
	 */
	protected void addProfilingTab(List tabList, boolean attach) {
		tabList.add(new ProfileTab(attach));
	}
	
	/**
	 * Add to the tab list the Eclipse Common Tab.
	 * @param tabList
	 */
	protected void addCommonTab(List tabList) {
		tabList.add(new CommonTab());
	}
	
	/**
	 * Add to the tab list the Test & Deployment Selection Tab.
	 * @param tabList
	 * @param mode The run mode (this is used for proposing test elements that support
	 * this mode only).
	 * @param launchConfigurationType The name of the Launch Configuration Type (this is used
	 * for proposing test elemnts that support this launch configuration type only). <code>null</code>
	 * is accepted and denotes the default Hyades Launch Configuration.
	 * @deprecated Use addTestAndDeploymentTab(List, String) instead and implement 
	 * getLaunchConfigurationType().
	 */
	protected void addTestAndDeploymentTab(List tabList, String mode, String launchConfigurationType) {
		tabList.add(new TestAndDeploymentTab(mode, launchConfigurationType));
	}
	
	/**
	 * Add to the tab list the Test & Deployment Selection Tab.
	 * @param tabList
	 * @param mode The run mode (this is used for proposing test elements that support
	 * this mode only).
	 */
	protected void addTestAndDeploymentTab(List tabList, String mode) {
		tabList.add(new TestAndDeploymentTab(mode, getLaunchConfigurationType()));
	}
	
	/**
	 * Add to the tab list the Test Component & Deployment Selection Tab.
	 * @param tabList
	 * @param modeThe run mode (this is used for proposing test elements that support
	 * this mode only).
	 * @param launchConfigurationTypeThe name of the Launch Configuration Type (this is used
	 * for proposing test elemnts that support this launch configuration type only). <code>null</code>
	 * is accepted and denotes the default Hyades Launch Configuration.
	 * @deprecated Use addTestComponentAndDeploymentTab(List, String) instead and implement
	 * getLaunchConfigurationType().
	 */
	protected void addTestComponentAndDeploymentTab(List tabList, String mode, String launchConfigurationType) {
		tabList.add(new TestComponentAndDeploymentTab(mode, launchConfigurationType));
	}
	
	/**
	 * Add to the tab list the Test Component & Deployment Selection Tab.
	 * @param tabList
	 * @param modeThe run mode (this is used for proposing test elements that support
	 * this mode only).
	 * @param launchConfigurationTypeThe name of the Launch Configuration Type (this is used
	 * for proposing test elemnts that support this launch configuration type only). <code>null</code>
	 * is accepted and denotes the default Hyades Launch Configuration.
	 */
	protected void addTestComponentAndDeploymentTab(List tabList, String mode) {
		tabList.add(new TestComponentAndDeploymentTab(mode, getLaunchConfigurationType()));
	}
	
	/**
	 * Return the attach option for constructing the Profile tab (if applicable).
	 * By default, return true. This method may be redefined by subclasses.
	 * @param launchConfigurationType
	 * @return
	 */
	protected boolean getProfilingAttachOption() {
		return true;
	}
	
	/**
	 * Returns the Launch Configuration Type this tab group applies for.
	 * null is valid and means "Basic Hyades Test Launch Configuration Type".
	 * Subclasses should redefine this method (unless no particular type-based
	 * extension is used).
	 * @return
	 */
	protected String getLaunchConfigurationType() {
		return null;
	}
	
	/**
	 * Add to the tab list the Runnable Item selection Tab (Runnable Item
	 * is typically what is to be run, e.g. a Test or a Test Component)
	 * @param tabList
	 */
	abstract protected void addRunnableItemSelectionTab(List tabList, String mode);
	
	/**
	 * Add to the tab list any additional tab.
	 * This method should be sub-classed to provide additional tabs. These tabs
	 * are inserted before the Profiling tab (if any) or the Common tab.
	 * @param tabList
	 */
	abstract protected void addAdditionalTabs(List tabList, String mode);
	
}
