/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.local.control;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import org.eclipse.hyades.execution.security.ISecureClientParameters;
import org.eclipse.hyades.internal.execution.local.common.ActiveAgentListCommand;
import org.eclipse.hyades.internal.execution.local.common.AgentDetailsCommand;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.ControlMessage;
import org.eclipse.hyades.internal.execution.local.common.GetPropertyListCommand;
import org.eclipse.hyades.internal.execution.local.common.KillProcessCommand;
import org.eclipse.hyades.internal.execution.local.common.ProcessExitedCommand;
import org.eclipse.hyades.internal.execution.local.common.PropertyListCommand;
import org.eclipse.hyades.internal.execution.local.common.QueryAgentDetailsCommand;
import org.eclipse.hyades.internal.execution.local.common.QueryAgentListCommand;
import org.eclipse.hyades.internal.execution.local.common.QueryProcessListCommand;
import org.eclipse.hyades.internal.execution.local.common.RegisteredProcessListCommand;
import org.eclipse.hyades.internal.execution.local.common.SetNVPairCommand;
import org.eclipse.hyades.internal.execution.security.LoginFailedException;
import org.eclipse.hyades.internal.execution.security.SecureConnectionRequiredException;
import org.eclipse.hyades.internal.execution.security.UntrustedAgentControllerException;
import org.eclipse.hyades.internal.execution.security.User;

class NodeImpl implements Node {
    
    private static final int DEFAULT_LIST_PROCESS_TIMEOUT = 7000;
    
	protected String _name;
	protected InetAddress[] _addr;
	protected Vector _listeners=new Vector(10);
	protected Connection _connection;
	protected Application _application;
	private User _user;
	private Object propLock = new Object();
	
	private ISecureClientParameters _securityParms=null;
	
	/* The process table for this node.  The processes will be indexed based upon their PID */
	protected Hashtable _processes=new Hashtable(300);

	/* The properties table for this node. */
	protected SetNVPairCommand[] props = null;
	
	/* The listener for all our processes events.  Do not implement
	   the interface on the instance as this could cause deadlocks
	 */
	 private ProcessListener _processListener=null;
	
	/* These two variables are used to keep track of where we are
	   when listProcesses() is invoked.  The count keeps track
	   of how many responses we are waiting for and the lock
	   controls access to itc change.  The listProcesses() is
	   a synchronized method so this is safe
	*/
	private long _listProcessesWaitingCommandCount=0;
	private Object _listProcessesLock=new Object();
	
	public NodeImpl(String name, InetAddress addr) {
		super();
		_name=name;
		try {
			_addr=InetAddress.getAllByName(addr.getHostName());
		}
		catch(UnknownHostException e) {
			
		}
			
		
		/* Create a process listener for this node. */
		_processListener=new ProcessListener() {
			public synchronized void processLaunched(Process process) {
				try {
					Long key=new Long(process.getProcessId());
					
					/* If there is a process with this exact PID in the processList
					 * then it must have exited when this client was disconnected
					 * from the remote machine.  Remove this process and place the
					 * currently executing one in the list.
					 */
					_processes.remove(key);
					_processes.put(key, process);
				}
				catch(InactiveProcessException e) {
					/* We can ignore this */
				}
			}
			
			public synchronized void processExited(Process process) {
				try {
					Long key=new Long(process.getProcessId());
					_processes.remove(key);
				}
				catch(InactiveProcessException e) {
					/* We can ignore this */
				}
			}
		};
	}


	public NodeImpl(String name, InetAddress[] addr) {
		super();
		_name=name;
		_addr=addr;
		
		/* Create a process listener for this node. */
		_processListener=new ProcessListener() {
			public synchronized void processLaunched(Process process) {
				try {
					Long key=new Long(process.getProcessId());
					if(!_processes.containsKey(key)) {
						_processes.put(key, process);
					}
				}
				catch(InactiveProcessException e) {
					/* We can ignore this */
				}
			}
			
			public synchronized void processExited(Process process) {
				try {
					Long key=new Long(process.getProcessId());
					_processes.remove(key);
				}
				catch(InactiveProcessException e) {
					/* We can ignore this */
				}
			}
		};
	}


	/**
	 * @see Node#killProcess(Process)
	 */
	public void killProcess(Process process) throws InactiveProcessException, NotConnectedException{
		/* Determine if we have a connection */
		if(_connection==null) {
			throw new NotConnectedException();
		}
		
		/* Determine if the process is active */
		if(!process.isActive()) {
			throw new InactiveProcessException();
		}
		
		/* Kill the process */
		ControlMessage message=new ControlMessage();
		KillProcessCommand command=new KillProcessCommand();
		command.setProcessId(Long.parseLong(process.getProcessId()));
		message.appendCommand(command);
		try {
			_connection.sendMessage(message, new CommandHandler() {
				public void incommingCommand(Node node, CommandElement command) {
					/* Ignore the result of the call */	
				}
			});
		}
		catch(IOException e) {
			/* Should probably handle this one */
		}
	}

	/**
	 * @see Node#shutdown(long)
	 */
	public void shutdown(long delay) throws NotConnectedException {
		throw new NotImplementedException();
	}

	/**
	 * @see Node#reboot(long)
	 */
	public void reboot(long delay) throws NotConnectedException {
		throw new NotImplementedException();
	}

	/**
	 * @see Node#listMonitors()
	 */
	public Enumeration listMonitors() throws NotConnectedException {
		throw new NotImplementedException();
	}

	/**
	 * @see Node#listProcesses()
	 */
	public synchronized Enumeration listProcesses() throws NotConnectedException {
	    return this.listProcesses(NodeImpl.DEFAULT_LIST_PROCESS_TIMEOUT);
	}

	/**
	 * Query the process list and blocks the calling thread until the process list is
	 * completely retrieved or the timeout period has been exceeded.
	 */
	private synchronized Enumeration listProcesses(int timeout) throws NotConnectedException {
		/* Determine if we have a connection */
		if(_connection==null) {
			throw new NotConnectedException();
		}
		
		/* Save the "this" reference as the below anonymous classes need to reference
		   the containing node.
		*/
		final Node node=this;
		
		/* Request the process list from the client engine */
		ControlMessage message=new ControlMessage();
		QueryProcessListCommand command=new QueryProcessListCommand();
		message.appendCommand(command);
		
		/* Before we build the model we need to add a connection listener so that if the
		  connection is dropped we don't deadlock.  This must be removed when we are done so
		  we don't leak memory.
		*/
		final ConnectionListener connectionListener=new ConnectionListener() {
				public void connectionClosed(Connection connection) {	
					/* If there is a pending list processes lock we need to release it as our connection has been lost */
					synchronized(_listProcessesLock) {
						_listProcessesLock.notifyAll();
					}
				}
			};
		
		/* Wait until all of the below anonymous classes are done processing the complete
		   process list and then return the elements that comprise the list.
		*/
		synchronized(_listProcessesLock) {
			_connection.addConnectionListener(connectionListener);
			
				
		try {
			_connection.sendMessage(message, new CommandHandler() {
				/* This is the anonymous handler that will take the process list and
			 	  request the details for each process.
				*/
				public void incommingCommand(Node node, CommandElement command) {
					switch((int)command.getTag()) {
					case (int)org.eclipse.hyades.internal.execution.local.common.Constants.RA_PROCESS_LIST:
					{	
						RegisteredProcessListCommand qs=(RegisteredProcessListCommand)command;
						long[] list=qs.getProcessList();
						final long entries=list.length;
						
						
						/* Take this opportunity to scrub the process list */
						Enumeration enum=_processes.elements();
						while(enum.hasMoreElements()) {
							long pid=0;
							Process current=(Process)enum.nextElement();
							try {
								pid=Long.parseLong(current.getProcessId());
							}
							catch(InactiveProcessException e) {
								/* Skip this process */
								continue;
							}
							long offset;
							for(offset=0; offset<entries; offset++) {
								if(pid==list[(int)offset]) {
									
									/* In case this process has died we need to clean it from the active list as
									 * we are recycling PID's here.
									 */
									if(current.isActive()) {
										break;
									}
									else {
										offset=entries;
									}
								}
							}
							if(offset==entries) {
								/* This process needs to be informed it has died */
								_processes.remove(new Long(pid));
								try {
									ProcessExitedCommand pec=new ProcessExitedCommand();
									pec.setProcessId(pid);
									((ProcessImpl)current).handleCommand(pec);
								}
								catch(ClassCastException e) {
									/* This can be ignored as it isn't the right type */
								}		
							}
						}
						
						/* If the list is empty we notify the other thread and return */
						if(entries==0) {
							synchronized(_listProcessesLock) {
								_listProcessesLock.notify();
							}
							break;
						}
						

						/* Iterate through the process list and request the details for each process */
						ControlMessage msg=new ControlMessage();
						for(int i=0; i<entries; i++) {
							QueryAgentListCommand qal=new QueryAgentListCommand();
							qal.setProcessId(list[i]);
							msg.appendCommand(qal);
						}
						try {
							/* We are going to request process details for each of the processes.  We need to
							   wait until we get all the responses.
							*/
							synchronized(_listProcessesLock) {
								_listProcessesWaitingCommandCount=entries;
							}
							_connection.sendMessage(msg, new CommandHandler() {
								/* This is the anonymous handler that takes the process details and populates the model */
								public void incommingCommand(Node node, CommandElement command) {
									switch((int)command.getTag()) {
									case (int)org.eclipse.hyades.internal.execution.local.common.Constants.RA_AGENT_LIST:
									{
										ActiveAgentListCommand aal=(ActiveAgentListCommand)(command);
										
										/* Look in the process table for this process */
										Long processId=new Long(aal.getProcessId());
										Process process=(Process)_processes.get(processId);
										if(process==null) {
											process=new ProcessImpl(node, aal.getProcessName(), aal.getProcessId());
											try {
												process.addProcessListener(_processListener);
											}
											catch(ClassCastException e) {
												/* We can ignore this */
											}
											_processes.put(processId, process);
										}
										else {
											/* Ensure the proper process information.  This is
											   neccessary as the original process with this pid may have exited
											*/
											if(!process.getExecutable().equals(aal.getProcessName()) && !aal.getProcessName().equals("unknown")) {
												ProcessExitedCommand pec=new ProcessExitedCommand();
												pec.setProcessId(aal.getProcessId());
												((ProcessImpl)process).handleCommand(pec);
												_processes.remove(processId);
												process=new ProcessImpl(node, aal.getProcessName(), aal.getProcessId());
												try {
													process.addProcessListener(_processListener);
												}
												catch(ClassCastException e) {
													/* We can ignore this */
												}
												_processes.put(processId, process);	
											}	
										}
										
										/* Iterate through the agents */
										String[] agents=aal.getAgents();
										((ProcessImpl)process).removeInactiveAgents(agents); // Bug 71586

										for(int j=0; j<agents.length; j++) {
											ControlMessage msg2=new ControlMessage();
											QueryAgentDetailsCommand cmd=new QueryAgentDetailsCommand();
											cmd.setProcessId(aal.getProcessId());
											cmd.setAgentName(agents[j]);
											msg2.appendCommand(cmd);
											try {
												synchronized(_listProcessesLock) {
													_listProcessesWaitingCommandCount++;
												}
												_connection.sendMessage(msg2, new CommandHandler() {
													/* This is the anonymous handler that will take the process list and
			 	  								   	   request the details for each process.
													*/
													public void incommingCommand(Node node, CommandElement command) {
														switch((int)command.getTag()) {
														case (int)org.eclipse.hyades.internal.execution.local.common.Constants.RA_AGENT_DETAILS:
															AgentDetailsCommand adc=(AgentDetailsCommand)command;
															
															/* Lookup the process in the process table */
															Process processLookup=(Process)_processes.get(new Long(adc.getProcessId()));
															/* This should never happen as we query as they are added to the table, but just to be safe... */
															if(processLookup==null) {
																break;
															}
															
															((ProcessImpl)processLookup).setUUID(adc.getProcessUUID());
															
															/* Does this agent already exist? */
															Agent agent=processLookup.getAgent(adc.getAgentName());
															if(agent==null) {
																agent=new AgentImpl(processLookup, adc.getAgentName(), adc.getAgentType(), true);
																((AgentImpl)agent).setUUID(adc.getAgentUUID());
																try {
																	((ProcessImpl)processLookup).addAgent(agent);
																	agent.addAgentListener((ProcessImpl)processLookup);
																}
																catch(ClassCastException e) {
																	/* Cannot set relationship unless process is a ProcessImpl */
																}
															}
														}
														synchronized(_listProcessesLock) {
															_listProcessesWaitingCommandCount--;
															if(_listProcessesWaitingCommandCount==0) {
																_listProcessesLock.notify();
															}
														}
													}
												});
											}
											catch(IOException e) {
												/* We should probably handle this one */
											}			
										}
									}
									}
									/* If this is the last process details allow "listProcesses()" to return */
									synchronized(_listProcessesLock) {
										_listProcessesWaitingCommandCount--;
										if(_listProcessesWaitingCommandCount==0) {
											_listProcessesLock.notify();
										}
									}	
								}
							});
						}
						catch(IOException e) {
							synchronized(_listProcessesLock) {
								_listProcessesWaitingCommandCount=0;
								_listProcessesLock.notify();
							}
						}
					}  /* end case */
					}  /* end switch */
				}
			
			});
		}
		catch(IOException e) {
			/* We should probably handle this one */
			return null;
		}
		
		try {
			/* Wait up to the timeout value for all the process information then exit the lock */
			_listProcessesLock.wait(timeout);
			_connection.removeConnectionListener(connectionListener);
		}
		catch(InterruptedException e) {
		}
	
		} /* synchronized */
		
		return ((Hashtable)_processes.clone()).elements();
	}
	
	/**
	  * @see Node#getProcess()
	  */
	public synchronized Process getProcess(String processId) {
		return (Process)_processes.get(new Long(processId));
	}

	/**
	 * @see Node#getInetAddress()
	 */
	public InetAddress getInetAddress() {
		if(_addr!=null) {
			return _addr[0];
		}
		
		return null;
	}
	
	/**
	 * @see Node#getAllInetAddresses()
	 */
	public InetAddress[] getAllInetAddresses() {
		return _addr;	
	}

	/**
	 * @see Node#getName()
	 */
	public String getName() {
		return _name;
	}
	
	/**
	 * @see Node#isConnected()
	 */
	public boolean isConnected() {
		if(_connection!=null) {
			return _connection.isActive();
		}
		return false;
	}
	
	/**
	 * @see Node#connect()
	 */
	public synchronized Connection connect(int port) throws AgentControllerUnavailableException, SecureConnectionRequiredException, UntrustedAgentControllerException, LoginFailedException {
		
		
		if(_connection==null || !_connection.isActive()) {
			try {
				
				/* If the keystore is specified then we will try a secure connection */
				if(_securityParms==null) {
					_connection=new ConnectionImpl();
					((ConnectionImpl)_connection).connect(this, port);
				}
				else {
					/* Look through our extension points for a security provider */
					_connection=new SecureConnectionImpl();
					if(_connection!=null) {
						_connection.connect(this, port);
					}
				}

				//
				// Bug 71980
				// If we do a connect(), we have to make sure we get all the process registered in this node.
				// There might have been some processes launched by other Node instances and thus the process
				// list is just local to that instance.
				//
				Enumeration processes = listProcesses(); // This will get the processes as well as the agents
				while(processes.hasMoreElements()) {
					Process process = (Process)processes.nextElement();
					Long key;
					try {
						key = new Long(process.getProcessId());
						if(!_processes.containsKey(key)) {
							_processes.put(key, process);
						}
					} catch (NumberFormatException e) {
						// Cannot resolve the process ID, do nothing
					} catch (InactiveProcessException e) {
						// Process is not active, do nothing
					}
				}
			}
			catch(NotConnectedException nce) {
				// Do nothing... since we should be connected already
			}
			catch(IOException e) {
				_connection=null;
				throw new AgentControllerUnavailableException();
			}
		}


		return _connection;
		
	}
	
	
	/**
	 * @see Node#getConnection()
	 */
	public Connection getConnection() {
		return _connection;
	}
	
	public ProcessListener getProcessListener() {
		return _processListener;	
	}
	
	/**
	 * @see Node#setUser()
	 */
	public void setUser(User user) {
		_user=user;	
	} 
	
	/**
	 * @see Node#getUser()
	 */
	public User getUser() {
		return _user;
	}

	
	
	
	public void setApplication(Application app) {
		_application=app;
	}
	
	public Application getApplication() {
		return _application;	
	}
	
	public void setSecurityParameters(ISecureClientParameters manager) {
		
		_securityParms=manager;	
	}
	
	public ISecureClientParameters getSecurityParameters() {
		return _securityParms;
	}

	/**
	 * Bug 53938
	 * 
	 * Query the options specified in the Agent Controller's
	 * configuration file.
	 * @param name The name of the option
	 * @param type The type of the option
	 * @return An array of SetNVPairCommand objects
	 */
	public SetNVPairCommand[] getPropertyValues(String name, String type) throws NotConnectedException {
		/* Determine if we have a connection */
		if(_connection==null) {
			throw new NotConnectedException();
		}
		
		/* Save the "this" reference as the below anonymous classes need to reference
		   the containing node.
		*/
		final Node node=this;
		
		/* Request the process list from the client engine */
		ControlMessage message=new ControlMessage();
		GetPropertyListCommand command=new GetPropertyListCommand();
		command.setName(name);
		command.setType(type);
		message.appendCommand(command);

		synchronized(propLock) {
			try {
				_connection.sendMessage(message, new CommandHandler() {
					/* This is the anonymous handler that will take the property list */
					public void incommingCommand(Node node, CommandElement command) {
						switch((int)command.getTag()) {
							case (int)org.eclipse.hyades.internal.execution.local.common.Constants.RA_PROPERTY_LIST:
							{	
								PropertyListCommand qs = (PropertyListCommand)command;
								props = qs.getPropertyListValues();
								synchronized(propLock) {
									propLock.notify();
								}
							}
						}
					}
				});
				propLock.wait(10000); // 10 sec
			}
			catch(Exception e) {
				return null;
			}
		} // synchronized

		return props;
	}
	
	/* 
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.local.control.Node
	 * 		#waitForProcess(java.lang.String, int)
	 * 
	 * Waits for a particular process to be present or the timeout value to be reached; after the 
	 * process list returns (the internal process list in this node instance is updated) the
	 * process is retrieved via the get process method.  The timeout value is approximate since
	 * there is some time needed to execute the various operations within the loop and each wait
	 * is done timeout divided by 2000 times.  For example, a 60000 (60 second) timeout would cause
	 * the loop to execute 30 times and wait for 2 seconds at the end of each process list query.
	 * 
	 * The listProcesses(...) method default wait time itself is up to 7 seconds to receive the
	 * process list.
	 * 
	 */
	public synchronized Process waitForProcess(String processIdentity, int timeout) throws NotConnectedException, InactiveProcessException {
	    
	    // Change this from a polling model into a notify model (this is good enough for now)
	    try {
	        for (int i=0, j=timeout/2000; i<j; i++) {
	            for (Enumeration processes = this.listProcesses() ; processes.hasMoreElements(); ) {
	                Process process = (Process) processes.nextElement();
	                if (process.getProcessId().equals(processIdentity)) {
	                    return process;
	                }
	            }
	            this.wait(2000);
	        }
	    } catch (InterruptedException e) {
	        // Do not need to handle this
	        e.printStackTrace();
	    }
	    
	    return null; // no process was found in the allowed time
	}

}

