/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.editor;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.PartInitException;

import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.hyades.ui.util.IRefreshable;

/**
 * Instances of this class should be used when creating an editor extension association
 * descriptor.
 * 
 * <p>The editor extension is an object that is responsible to add the pages to be 
 * presented in a {@link IHyadesEditorPart} and to implement some typical editor 
 * actions such as the saving and refresh methods. 
 * 
 * <p>The list of adapter classes used by the Hyades framework is:<OL>
 * <LI>{@link org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter - invoked when 
 * external changes have happend on files associated with this editor extension.</LI>
 * </OL> 
 * 
 * @author marcelop
 * @since 0.0.1
 */
public interface IEditorExtension
extends IDisposable, IRefreshable, ISelectionListener, IAdaptable
{
	/**
	 * Initializes this editor extension with the given IHyadesEditorPart.
	 * 
	 * <p>This method is automatically called shortly after part construction; it marks
	 * the start of the part's life cycle. The 
	 * {@link IDisposable#dispose() dispose} method will be called 
	 * automatically at the end of the life cycle. Clients must not call this method.
	 * 
	 * <p>Implementors of this method must examine the edited object available at 
	 * {@link IHyadesEditorPart#getEditorObject()} and the 
	 * {@link IHyadesEditorPart#getEditorInput() editor input} to
	 * determine if one of them is understood.  If not, the implementor must throw
	 * a <code>PartInitException</code>
	 * 
	 * @param hyadesEditorPart
	 * @exception PartInitException if this editor extension was not initialized successfully
	 */
	public void init(IHyadesEditorPart hyadesEditorPart)
	throws PartInitException;

	/**
	 * Returns the hyades editor part in which this extension is presented.
	 * @return IHyadesEditorPart
	 */
	public IHyadesEditorPart getHyadesEditorPart();

	/**
	 * Creates the pages of this editor extension.  Each page should be added 
	 * to the specified <code>hyadesEditor</code> using one of the 
	 * <code>addPage(...)</code> methods.
	 * 
	 * <p>The <code>getContainer()</code> method of the IHyadesEditorPart 
	 * provides the parent for the controls to be added as pages.
	 * 
	 * @param hyadesEditor
	 */
	public void createPages();
		
	/**
	 * Sets the current selection for this selection provider.
	 * @param selection the new selection
	 */
	public void setSelection(IStructuredSelection structuredSelection);

	/**
	 * Returns the current selection for this editor extension.
	 * @return the current selection
	 */
	public IStructuredSelection getSelection();
	
	/**
	 * Returns the message to show in the status line.  If this method
	 * returns null the status line won't present any message.
	 * 
	 * <p>If a subclass requires a more sophisticated use of the 
	 * status line, it should manipulate the status line direct.  The
	 * status line can be obtained by 
	 * <code>getHyadesEditorPart().getEditorSite().getActionBars().getStatusLineManager()</code>.
	 *
	 * @param selection the current selection
	 * @return the status line message
	 */
	public String getStatusLineMessage(IStructuredSelection structuredSelection);
	

	/**
	 * Sets the cursor and selection state for this editor extension to the passage 
	 * defined by the given marker.
	 * @param marker the marker
	 */
	public void gotoMarker(IMarker marker);	

	/**
	 * Returns true when the IHyadesEditorPart should check the pages
	 * when calculating its dirty state or false when it should only relay on the
	 * value returned by this implementor's <code>isDirty()</code> method.
	 * @return boolean
	 * @see #isDirty()
	 */
	public boolean checkPagesDirtyState();
	
	/**
	 * Returns true if the IHyadesEditorPart should refresh the pages
	 * after invoking this implementor's <code>refreshContent(Object)</code> 
	 * method or false otherwise.
	 * @return boolean
	 * @see IRefreshable#refreshContent(Object)
	 */
	public boolean refreshPages();
	
	/**
	 * Saves the contents of this editor extension.
	 * 
	 * <p>If the save is successful, the editor extension should fire a property 
	 * changed event reflecting the new dirty state (<code>PROP_DIRTY</code> property).
	 * 
	 * <p>If the save is canceled through user action, or for any other reason, the
	 * part should invoke <code>setCancelled</code> on the <code>IProgressMonitor</code>
	 * to inform the caller.
	 * 
	 * <p>This method is long-running; progress and cancellation are provided
	 * by the given progress monitor. 
	 *
	 * @param monitor the progress monitor
	 */
	public void doSave(IProgressMonitor monitor);
	
	/**
	 * Saves the contents of this editor extension to another object.
	 * 
	 * <p>Implementors are expected to open a "Save As" dialog where the user will
	 * be able to select a new name for the contents. After the selection is made,
	 * the contents should be saved to that new name.  During this operation a
	 * <code>IProgressMonitor</code> should be used to indicate progress.
	 * 
	 * <p>If the save is successful, the editor extension fires a property changed 
	 * event reflecting the new dirty state (<code>PROP_DIRTY</code> property).
	 */
	public void doSaveAs();

	/**
	 * Returns whether the contents of this editor extension have changed since the 
	 * last save operation. If this value changes the editor extension must fire a property 
	 * listener event with <code>PROP_DIRTY</code>.
	 * @return <code>true</code> if the contents have been modified and need
	 *   saving, and <code>false</code> if they have not changed since the last
	 *   save
	 */
	public boolean isDirty();

	/**
	 * Returns whether the "Save As" operation is supported by this part.
	 * @return <code>true</code> if "Save As" is supported, and <code>false</code>
	 *  if not supported
	 */
	public boolean isSaveAsAllowed();

	/**
	 * Returns whether the contents of this editor extension should be saved when the 
	 * editor extension is closed.
	 * @return <code>true</code> if the contents of the editor extension should be saved 
	 * on close, and <code>false</code> if the contents are expendable
	 */
	public boolean isSaveOnCloseNeeded();
	
	/**
	 * Notifies this editor extension that the page with the given index has been
	 * activated. This method is called when the user selects a different tab.
	 * @param index
	 * @return boolean indicating that the Hyades Editor Part selection listeners 
	 * should be notified. 
	 */
	public boolean pageActivated(int index);	
}
