/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.util;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaConventions;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.ui.actions.WorkspaceModifyOperation;

/**
 * Contains utility methods to work with the Eclipse's JDT.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class JavaUtil
{
	/**
	 * Sets the value of a classpath variable.
	 * @param variable
	 * @param value
	 * @param progressMonitor
	 * @throws JavaModelException
	 * @throws InterruptedException
	 * @throws InvocationTargetException
	 */
	public static void setVariableValue(final String variable, final String value, IProgressMonitor progressMonitor)
	throws JavaModelException, InterruptedException, InvocationTargetException
	{
		if((variable == null) || (value == null))
			return;
			
		String currentValue = getVariableValue(variable);
		if(currentValue != null)
		{
			if(value.equals(currentValue))
				return;
				
			if(new File(value).equals(new File(currentValue)))
				return;
		}
			
		IRunnableWithProgress operation = new WorkspaceModifyOperation()
		{
			public void execute(IProgressMonitor monitor)
			throws CoreException, InvocationTargetException, InterruptedException
			{
				JavaCore.setClasspathVariable(variable, new Path(value), monitor);
			}
		};
		
		operation.run(progressMonitor);
	}
	
	/**
	 * Returns the value of the classpath variable specified in the
	 * argument.
	 * @param variable
	 * @return String
	 */
	public static String getVariableValue(String variable)
	{
		IPath path = JavaCore.getClasspathVariable(variable);		
		if(path != null)
		{
			String value = path.makeAbsolute().toString();
			if(!value.equals(""))
				return value;
		}
			
		return null;
	}
	
	/**
	 * Adds a resource pattern to the list of resources that should not be
	 * copied to the output folder.
	 * 
	 * @param resourcePattern 	 Possible values:	
	 * {<name>[,<name>]*} where <name> is a file name pattern (only * wild-
	 * cards allowed) or the name of a folder which ends with '/'
	 */
	public static void addResourcePatternToNotCopyList(String resourcePattern)
	{
		String value = JavaCore.getOption("org.eclipse.jdt.core.builder.resourceCopyExclusionFilter");
		if(value == null)
			value = "";
			
		if(value.indexOf(resourcePattern) >= 0)
			return;
			
		if(!"".equals(value))
			value = value + ",";
			
		value = value + resourcePattern;
		Hashtable options = new Hashtable(3);
		options.put("org.eclipse.jdt.core.builder.resourceCopyExclusionFilter", value);
		JavaCore.setOptions(options);
	}
	
	/**
	 * Remmoves a resource pattern to the list of resources that should not be
	 * copied to the output folder.
	 * 
	 * @param resourcePattern 	 Possible values:	
	 * {<name>[,<name>]*} where <name> is a file name pattern (only * wild-
	 * cards allowed) or the name of a folder which ends with '/'
	 */
	public static void removeResourcePatternToNotCopyList(String resourcePattern)
	{
		String value = JavaCore.getOption("org.eclipse.jdt.core.builder.resourceCopyExclusionFilter");
		if(value == null)
			value = "";

		if(value.indexOf(resourcePattern) < 0)
			return;
		
		value = StringUtil.replace(value, resourcePattern, "");
		value = StringUtil.replace(value, ",,", "");
			
		try
		{
			if(value.startsWith(","))
				value = value.substring(",".length());	
			if(value.endsWith(","))
				value = value.substring(0, value.length()-1);
		}
		catch(RuntimeException re)
		{
		}
			
		Hashtable options = new Hashtable(3);
		options.put("org.eclipse.jdt.core.builder.resourceCopyExclusionFilter", value);
		JavaCore.setOptions(options);
	}
	
	/**
	 * Returns a Project's classpath.  If the project has a dependency on another 
	 * project the exported entries of the last one will be added to the classpath
	 * of the former. 
	 * 
	 * @param project
	 * @param exportedOnly
	 * @return String[]
	 */
	public static String[] getProjectClasspath(IProject project, boolean exportedOnly)
	{
		if(project == null)
			return new String[0];
			
		IJavaProject javaProject = JavaCore.create(project);
		if(javaProject == null)
			return new String[0];

		List cpEntries = null;
		try
		{
			cpEntries = new ArrayList(Arrays.asList(javaProject.getRawClasspath()));
		}
		catch(JavaModelException e)
		{
		}
		if((cpEntries == null) || cpEntries.isEmpty())
			return new String[0];
		
		Set classpath = new HashSet(cpEntries.size());
		try
		{	
			IPath path = javaProject.getOutputLocation();
			if(path != null)
				classpath.add(ResourceUtil.getFullPath(path));
		}
		catch(JavaModelException e)
		{
		}
				
		for (Iterator i = cpEntries.iterator(); i.hasNext();)
		{
			IClasspathEntry entry = JavaCore.getResolvedClasspathEntry((IClasspathEntry)i.next());
			if(exportedOnly && (!entry.isExported()))
				continue;
			
			String strcp = null;
			switch(entry.getEntryKind())
			{
				case IClasspathEntry.CPE_LIBRARY:
				case IClasspathEntry.CPE_VARIABLE:
					strcp = ResourceUtil.getFullPath(entry.getPath());
					if(strcp != null)
						classpath.add(strcp);
					break;

				case IClasspathEntry.CPE_PROJECT:
					IPath projectPath = entry.getPath().makeAbsolute();
					IProject parentProject = ResourcesPlugin.getWorkspace().getRoot().getProject(projectPath.toString());				
					classpath.addAll(Arrays.asList(getProjectClasspath(parentProject, true)));
					break;
				
				case IClasspathEntry.CPE_CONTAINER:
					break;
					
				case IClasspathEntry.CPE_SOURCE:
					strcp = ResourceUtil.getFullPath(entry.getOutputLocation());
					if(strcp != null)
						classpath.add(strcp);
			}
		}
				
		return (String[])classpath.toArray(new String[classpath.size()]);
	}
	
	/**
	 * Returns a valid Java class based on the specified name.  The name argument
	 * should not contain the package information.
	 *    
	 * @param name
	 * @param generate.  Generates a new name even if the name argument is a valid
	 * identifier. 
	 * @return the class name or <code>null</code> if the name is not valid. 
	 */
	public static String getValidClassName(String name, boolean generate)
	{
		if(name == null)
			return null;

		name = name.trim();
		if((!generate) && JavaConventions.validateIdentifier(name).isOK())
			return name;
		
		name = StringUtil.toProperCase(name.trim());
		name = StringUtil.replace(name, " ", "");
		if(!Character.isLetter(name.charAt(0)))
			name = "C" + name;

		if(!JavaConventions.validateIdentifier(name).isOK())
			return null;
			
		return name;
	}
	
	/**
	 * Returns a valid Java identifier based on the specified name.  The name argument
	 * should not contain the package information.  The identifier can be a method
	 * or an attribute.
	 *    
	 * @param name
	 * @param generate.  Generates a new name even if the name argument is a valid
	 * identifier. 
	 * @return the identifier name or <code>null</code> if the name is not valid. 
	 */
	public static String getValidIdentifierName(String name)
	{
		if(name == null) {
			return null;
		} else if("".equals(name)) {
			return name;
		}
		
		name = name.trim();
		if(Character.isLetter(name.charAt(0)))
		{
			String auxName = name.substring(0, 1).toLowerCase();
			if(name.length() > 1)
				auxName = auxName + name.substring(1);
			name = auxName;
		}

		if(JavaConventions.validateIdentifier(name).isOK())
			return name;

		name = StringUtil.toProperCase(name.trim());
		name = StringUtil.replace(name, " ", "");

		if(Character.isLetter(name.charAt(0)))
		{
			String auxName = name.substring(0, 1).toLowerCase();
			if(name.length() > 1)
				auxName = auxName + name.substring(1);
			name = auxName;
		}
		
		if(!Character.isJavaIdentifierStart(name.charAt(0)))
			name = "c" + name;

		if(!JavaConventions.validateIdentifier(name).isOK())
		{
			StringBuffer sb = new StringBuffer(name);
			int length = name.length();
			for(int i = 0; i < length; i++){
			   if(!Character.isJavaIdentifierPart(sb.charAt(i)))
				  sb.replace(i,i+1,"_");	
			}
			return sb.toString();
		}
		return name;
	}	
}
