package org.eclipse.hyades.logging.java;

import java.util.logging.Formatter;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import org.eclipse.hyades.logging.core.IExternalizableToXml;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.MsgCatalogToken;
import org.eclipse.hyades.logging.events.cbe.MsgDataElement;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext;
import org.eclipse.hyades.logging.events.cbe.util.EventHelpers;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Extension of the Java Logging <code>java.util.logging.Formatter</code>
 * class used by <code>java.util.logging.Handler</code> instances to format
 * <code>java.util.logging.LogRecord</code> instances to XML.
 * <p>
 * If the <code>java.util.logging.LogRecord</code> instance is an
 * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code> (e.g.
 * instance of the
 * <code>org.eclipse.hyades.logging.core.IExternalizableToXml</code>
 * interface), the encapsulated Common Base Event is serialized to XML based on
 * the Common Base Event v.1.0.1 specification. Otherwise, the
 * <code>java.util.logging.LogRecord</code> is first converted to a Common
 * Base Event (see below for mapping details) and serialized to XML based on the 
 * Common Base Event v.1.0.1 specification.
 * <p>
 * The <code>java.util.logging.LogRecord</code> instance is converted to a Common
 * Base Event by mapping the <code>java.util.logging.LogRecord</code> properties 
 * to Common Base Event properties using the following mapping:
 * <p>
 * <table border="1" cellpadding="10">
 * <tr>
 * <th>Log Record Property</th>
 * <th>Common Base Event Property</th>
 * </tr>
 * <tr>
 * <td>level</td>
 * <td>ExtendedDataElement[0...n].Name = "CommonBaseEventLogRecord:level" <br>
 * ExtendedDataElement[0...n].Type = "noValue" <br>
 * ExtendedDataElement[0...n].Children[0].Name = "name" <br>
 * ExtendedDataElement[0...n].Children[0].Type = "string" <br>
 * ExtendedDataElement[0...n].Children[0].Values[0] = &lt;level's name&gt; <br>
 * ExtendedDataElement[0...n].Children[1].Name = "value" <br>
 * ExtendedDataElement[0...n].Children[0].Type = "int" <br>
 * ExtendedDataElement[0...n].Children[0].Values[0] = &lt;level's numerical
 * value&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>loggerName</td>
 * <td>ExtendedDataElement[0...n].Name = "CommonBaseEventLogRecord:loggerName"
 * <br>
 * ExtendedDataElement[0...n].Type = "string" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;logger's name&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>msg</td>
 * <td>MsgDataElement.MsgCatalogId = &lt;msg&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>millis</td>
 * <td>creationTime = &lt;millis&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>parameters*</td>
 * <td>MsgDataElement.MsgCatalogTokens = &lt;string representation of
 * parameters&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>resourceBundle*</td>
 * <td>[not mapped] <br>
 * </td>
 * </tr>
 * <tr>
 * <td>resourceBundleName</td>
 * <td>MsgDataElement.MsgCatalog = &lt;resource bundle name&gt; <br>
 * <td>MsgDataElement.MsgCatalogType = "Java" <br>
 * </td>
 * </tr>
 * <tr>
 * <td>sequenceNumber</td>
 * <td>ExtendedDataElement[0...n].Name =
 * "CommonBaseEventLogRecord:sequenceNumber" <br>
 * ExtendedDataElement[0...n].Type = "long" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;sequence number&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>sourceClassName</td>
 * <td>ExtendedDataElement[0...n].Name =
 * "CommonBaseEventLogRecord:sourceClassName" <br>
 * ExtendedDataElement[0...n].Type = "string" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;source class name&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>sourceMethodName</td>
 * <td>ExtendedDataElement[0...n].Name =
 * "CommonBaseEventLogRecord:sourceMethodName" <br>
 * ExtendedDataElement[0...n].Type = "string" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;source method name&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>threadID</td>
 * <td>ExtendedDataElement[0...n].Name = "CommonBaseEventLogRecord:threadID"
 * <br>
 * ExtendedDataElement[0...n].Type = "int" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;thread ID&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>thrown*</td>
 * <td>ExtendedDataElement[0...n].Name = "CommonBaseEventLogRecord:stackTrace" <br>
 * ExtendedDataElement[0...n].Type = "stringArray" <br>
 * ExtendedDataElement[0...n].Values[0...n] =
 * [&lt; <code>java.lang.Throwable</code>'s class name&gt;[: &lt;
 * <code>java.lang.Throwable</code>'s localized message&gt;]]<br>
 * ExtendedDataElement[0...n].Values[(n + 1)...m] =
 * &lt; <code>java.lang.Throwable</code>'s stackTraceElement[0...(m - (n + 1))]&gt;]<br>
 * <br>
 * ExtendedDataElement[0...n].Children[0...m].Name = "Cause" <br>
 * ExtendedDataElement[0...n].Children[0...n].Type = "stringArray" <br>
 * ExtendedDataElement[0...n].Children[0...m].Values[0...n] =
 * &lt;cause's class name&gt;[: &lt;cause's localized message&gt;]<br>
 * ExtendedDataElement[0...n].Children[0...m].Values[(n + 1)...m] =
 * &lt;cause's stackTraceElement[0...(m - (n + 1))]&gt;</li>
 * <li>[children[0] = &lt;cause's cause&gt;]<br>
 * </td>
 * </tr>
 * </table>
 * <p>
 * <b>* </b> <code>java.util.logging.LogRecord</code> property is stored in
 * <code>java.util.logging.LogRecord</code> super class to eliminate loss of
 * data.
 * <p>
 * If the Extended Data Element's <code>values</code> property is
 * larger than 1024 characters, the string is segmented into a String array of
 * 1024-character elements and set on the Extended Data Element. However, if the
 * Extended Data Element's <code>values</code> property is 1024 or less
 * characters, the string is set directly on the first element a String array
 * and set on the Extended Data Element. As such, the <code>type</code>
 * property is set to <code>stringArray</code>.
 * <p>
 * Names identifying the associated
 * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
 * properties in the encapsulated Common Base Event for the
 * <code>java.util.logging.LogRecord</code> properties are provided as static
 * string constants in the <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code> 
 * class. These name properties are prefixed by the
 * <code>CommonBaseEventLogRecord:</code> namespace (see
 * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
 * <p>
 * An instance of this class will be returned from the
 * <code>getFormatter()</code> API on <code>java.util.logging.Handler</code>
 * instances.
 * <p>
 * The default <code>java.util.logging.LogManager</code> implementation uses
 * the following configuration variable in the default
 * &lt;JRE&gt;/lib/logging.properties configuration file to load configuration
 * for <code>java.util.logging.Logger</code> instances:
 * <p>
 * <handler name>.formatter = org.eclipse.hyades.logging.java.XmlFormatter
 * <p>
 * Alternatively, an instantiation of this formatter class may be set directly
 * on <code>java.util.logging.Handler</code> instances by using the
 * <code>setFormatter()</code> API.
 * <p>
 * NOTE: The Java Logging classes must be on the CLASSPATH at run-time to
 * utilize this formatter class.
 * <p>
 * 
 * @author   Paul E. Slauenwhite
 * @version  October 27, 2004
 * @since    April 1, 2003
 * @see      java.util.logging.Formatter
 * @see      java.util.logging.LogRecord
 * @see      org.eclipse.hyades.logging.java.CommonBaseEventLogRecord
 * @see      org.eclipse.hyades.internal.logging.core.XmlGenerator
 * @see      org.eclipse.hyades.logging.core.IExternalizableToXml
 */
public class XmlFormatter extends Formatter {

    /**
     * Instance variable which holds a reference to a
     * <code>org.eclipse.hyades.logging.events.cbe.EventFactory</code> for
     * generating event instances.
     */
    private final static EventFactory EVENT_FACTORY = EventFactoryContext.getInstance().getSimpleEventFactoryHome().getAnonymousEventFactory();

    /**
     * Private static reference to the platform-dependent line separator character.
     */
    private static final String LINE_SEPARATOR = System.getProperty("line.separator");
    
    /**
     * Generates the XML representation of the parameter
     * <code>java.util.logging.LogRecord</code> instance.
     * <p>
     * If the parameter <code>java.util.logging.LogRecord</code> instance is
     * an <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
     * (e.g. instance of the
     * <code>org.eclipse.hyades.logging.core.IExternalizableToXml</code>
     * interface), the encapsulated Common Base Event is serialized to XML based
     * on the Common Base Event v.1.0.1 specification. Otherwise, the
     * <code>java.util.logging.LogRecord</code> is first converted to a Common
     * Base Event (see class comment header for mapping details) and serialized to 
     * XML based on the Common Base Event v.1.0.1 specification.
     * <p>
     * 
     * @param logRecord
     *            The <code>java.util.logging.LogRecord</code> instance to be
     *            formatted to XML.
     * @return The XML representation of the parameter
     *         <code>java.util.logging.LogRecord</code> instance.
     * @see java.util.logging.Formatter#format(LogRecord)
     */
    public String format(LogRecord logRecord) {

        //Determine if the parameter LogRecord subclass implements IExternalizableToXml (e.g. CommonBaseEventLogRecord):
        try {
            return (((IExternalizableToXml)(logRecord)).externalizeCanonicalXmlString().concat(LINE_SEPARATOR));
        } 
        catch (Exception e) {
            //Ignore since LogRecord subclass does not implement IExternalizableToXml or a serialization error.
        }

        //Map the parameter LogRecord's properties to Common Base Event properties:
        CommonBaseEvent commonBaseEvent = EVENT_FACTORY.createCommonBaseEvent();
        
        //Map the sequence number property:
        ExtendedDataElement sequenceNumberExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
        sequenceNumberExtendedDataElement.setName(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER);
        sequenceNumberExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_LONG_VALUE);
        sequenceNumberExtendedDataElement.setValues(new String[] { Long.toString(logRecord.getSequenceNumber())});

        commonBaseEvent.addExtendedDataElement(sequenceNumberExtendedDataElement);

        //Map the thread ID property:
        ExtendedDataElement threadIDExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
        threadIDExtendedDataElement.setName(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID);
        threadIDExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_INT_VALUE);
        threadIDExtendedDataElement.setValues(new String[] { Integer.toString(logRecord.getThreadID())});

        commonBaseEvent.addExtendedDataElement(threadIDExtendedDataElement);

        //Map the millis property:
        commonBaseEvent.setCreationTimeAsLong(logRecord.getMillis());

        //Map the level property:
        Level level = logRecord.getLevel();
        
        if(level != null){
            
	        ExtendedDataElement levelNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
	        levelNameExtendedDataElement.setName(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL_NAME);
	        levelNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
	        levelNameExtendedDataElement.setValues(new String[] { level.getName()});
	
	        ExtendedDataElement levelValueExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
	        levelValueExtendedDataElement.setName(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL_VALUE);
	        levelValueExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_INT_VALUE);
	        levelValueExtendedDataElement.setValues(new String[] { Integer.toString(level.intValue())});
	
	        ExtendedDataElement levelExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
	        levelExtendedDataElement.setName(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL);
	        levelExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_NO_VALUE_VALUE);
	        levelExtendedDataElement.addChild(levelNameExtendedDataElement);
	        levelExtendedDataElement.addChild(levelValueExtendedDataElement);
	
	        commonBaseEvent.addExtendedDataElement(levelExtendedDataElement);
        }

        //Map the logger name property:
        String loggerName = logRecord.getLoggerName();
        
        if(loggerName != null){
	        
            ExtendedDataElement loggerNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
	        loggerNameExtendedDataElement.setName(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME);
	        loggerNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
	        loggerNameExtendedDataElement.setValues(new String[] {loggerName});
	
	        commonBaseEvent.addExtendedDataElement(loggerNameExtendedDataElement);
	    }
        
        //Map the message property:
        String message = logRecord.getMessage();
        
        if(message != null){
        
            MsgDataElement msgDataElement = EVENT_FACTORY.createMsgDataElement();
            msgDataElement.setMsgCatalogId(message);

            commonBaseEvent.setMsgDataElement(msgDataElement);
        }
        
        //Map the parameters property:
        Object[] parameters = logRecord.getParameters();
        
        if (parameters != null) {

            MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();

            if (msgDataElement == null) {

                msgDataElement = EVENT_FACTORY.createMsgDataElement();

                commonBaseEvent.setMsgDataElement(msgDataElement);
            } 

            MsgCatalogToken msgCatalogToken = null;

            for (int counter = 0; counter < parameters.length; counter++) {

                msgCatalogToken = EVENT_FACTORY.createMsgCatalogToken();
                msgCatalogToken.setValue(String.valueOf(parameters[counter]));

                msgDataElement.addMsgCatalogToken(msgCatalogToken);
            }
        }

        //Map the resource bundle name property:
        String resourceBundleName = logRecord.getResourceBundleName();
        
        if(resourceBundleName != null){
            
            MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();

            if (msgDataElement == null) {

                msgDataElement = EVENT_FACTORY.createMsgDataElement();

                commonBaseEvent.setMsgDataElement(msgDataElement);
            }

            msgDataElement.setMsgCatalogType("Java");
            msgDataElement.setMsgCatalog(resourceBundleName);
        }
                
        //Map the source class name property:
        String sourceClassName = logRecord.getSourceClassName();
        
        if(sourceClassName != null){
            
            ExtendedDataElement sourceClassNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
            sourceClassNameExtendedDataElement.setName(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME);
            sourceClassNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
            sourceClassNameExtendedDataElement.setValues(new String[] { sourceClassName});

            commonBaseEvent.addExtendedDataElement(sourceClassNameExtendedDataElement);
        }

        //Map the source method name property:
        String sourceMethodName = logRecord.getSourceMethodName();
        
        if(sourceMethodName != null){
            
            ExtendedDataElement sourceMethodNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
            sourceMethodNameExtendedDataElement.setName(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME);
            sourceMethodNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
            sourceMethodNameExtendedDataElement.setValues(new String[] { sourceMethodName});

            commonBaseEvent.addExtendedDataElement(sourceMethodNameExtendedDataElement);
        }

        //Map the thrown property:
        Throwable thrown = logRecord.getThrown();
        
        if(thrown != null){
            commonBaseEvent.addExtendedDataElement(EventHelpers.convertToExtendedDataElement(thrown,CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THROWN));
        }
        
        return (((IExternalizableToXml)(commonBaseEvent)).externalizeCanonicalXmlString().concat(LINE_SEPARATOR));
    }
}