/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.trace.util;


/**
 * This is the record for accessing a method.  For up-to-date descriptions of the tag attributes,
 * see Event Specification for the Java Profiler at <code>http://www.eclipse.org/hyades</code>.
 * 
 * @author Richard Duggan, Qiyan Li
 */
public class RecordMethodAccess {

    /**
     * the constant event type indicating an entry event
     */
    public static final String METHOD_ENTRY = "methodEntry";

    /**
     * the constant event type indicating a call event
     */
    public static final String METHOD_CALL = "methodCall";

    /**
     * the constant event type indicating a return event
     */
    public static final String METHOD_RECEIVE = "methodReceive";

    /**
     * the constant event type indicating a call event
     */
    public static final String METHOD_REPLY = "methodReply";

    /**
     * the constant event type indicating a return event
     */
    public static final String METHOD_RETURN = "methodReturn";

    /**
     * the constant event type indicating an exit event
     */
    public static final String METHOD_EXIT = "methodExit";

    private final String event;
    private int transientThreadIdRef = 0;
    private int threadIdRef = 0;
    private String time = null;
    private final int methodIdRef;
    private String ticket = null;
    private int transientObjIdRef = 0;
    private final int objIdRef;
    private int transientClassIdRef = 0;
    private int classIdRef;
    private long threadCpuTime = 0;
    private long sequenceCounter = 0;
    private int stackDepth = 0;
    private String collationValue = null;
    private String traceIdRef = null;
    private RecordArray parameterList = null;
    private RecordVariable returnValue = null;
    private RecordInvocationContext invocationContext = null;
    private String remoteNodeIdRef = null;
    private String remoteProcessIdRef = null;
    private String remoteAgentIdRef = null;
    private int remoteThreadIdRef = 0;
    private long remoteTicket = 0;
    private long remoteSequenceCounter = 0;


    /**
     * Creates a record for the method access event.
     * 
     * @param event         the type of the event
     * @param methodIdRef   the unique ID distinguishing the method
     * @param objIdRef      the unique ID identifying the object on which the event happened
     */
    public RecordMethodAccess(String event, int methodIdRef, int objIdRef) {
        this.event = event;
        this.methodIdRef = methodIdRef;
        this.objIdRef = objIdRef;
    }


    /**
     * Converts this record into a string.
     * 
     * @return  this record as a string.
     */
    public String toString() {

        return "<" + event + " objIdRef=\"" + objIdRef + "\" methodIdRef=\"" + methodIdRef + "\"" +
            (classIdRef == 0 ? "" : " classIdRef=\"" + classIdRef + "\"") +
            (transientThreadIdRef == 0 ? "" : " transientThreadIdRef=\"" + transientThreadIdRef + "\"") +
            (threadIdRef == 0 ? "" : " threadIdRef=\"" + threadIdRef + "\"") +
            (time == null ? "" : " time=\"" + time + "\"") +
            (ticket == null ? "" : " ticket=\"" + ticket + "\"") +
            (transientObjIdRef == 0 ? "" : " transcientObjIdRef=\"" + transientObjIdRef + "\"") +
            (transientClassIdRef == 0 ? "" : " transientClassIdRef=\"" + transientClassIdRef + "\"") +
            (threadCpuTime == 0 ? "" : " threadCpuTime=\"" + threadCpuTime + "\"") +
            (sequenceCounter == 0 ? "" : " sequenceCounter=\"" + sequenceCounter + "\"") +
            (stackDepth == 0 ? "" : " stackDepth=\"" + stackDepth + "\"") +
            (collationValue == null ? "" : " collationValue=\"" + collationValue + "\"") +
            (traceIdRef == null ? "" : " traceIdRef=\"" + traceIdRef + "\"") +
            (remoteNodeIdRef == null ? "" : " remoteNodeIdRef=\"" + remoteNodeIdRef + "\"") +
            (remoteProcessIdRef == null ? "" : " remoteProcessIdRef=\"" + remoteProcessIdRef + "\"") +
            (remoteAgentIdRef == null ? "" : " remoteAgentIdRef=\""+ remoteAgentIdRef + "\"") +
            (remoteThreadIdRef == 0 ? "" : " remoteThreadIdRef=\"" + remoteThreadIdRef + "\"") +
            (remoteTicket == 0 ? "" : " remoteTicket=\"" + remoteTicket + "\"") +
            (remoteSequenceCounter == 0 ? "" : " remoteSequenceCounter=\"" + remoteSequenceCounter + "\"") +
            (parameterList == null && returnValue == null && invocationContext == null ? "/>" : ">\n") +
            (parameterList == null ? "" : parameterList.toString()) +
            (returnValue == null ? "" : returnValue.toString() + "\n") +
            (invocationContext == null ? "" : invocationContext.toString() + "\n") +
            (parameterList == null && returnValue == null && invocationContext == null ? "" : "</" + event + ">");
    }


    /**
     * Saves a parameter list using the <code>parameterNames</code>, <code>parameterClasses</code>,
     * and <code>parameterValues</code>.  All three parameters must be of the same length, or all
     * parameters will be ignored.
     * 
     * @param parameterIds      an ordered list of the parameter IDs
     * @param parameterClasses  an ordered list of the parameter classes
     * @param parameterValues   an ordered list of the parameter values
     */
    public void setParameterList(int[] parameterIds, Class[] parameterClasses, Object[] parameterValues) {
        
        parameterList = (parameterIds == null || parameterClasses == null || parameterValues == null ||
            parameterIds.length == 0 || parameterClasses.length == 0 || parameterValues.length == 0) ?
            null : new RecordArray(RecordVariable.PARAMETER, parameterIds, parameterClasses, parameterValues);
    }


    public void setRemoteContext(String remoteNodeIdRef, String remoteProcessIdRef, String remoteAgentIdRef,
        int remoteThreadIdRef, long remoteTicket, long remoteSequenceCounter) {

        this.remoteNodeIdRef = remoteNodeIdRef;
        this.remoteProcessIdRef = remoteProcessIdRef;
        this.remoteAgentIdRef = remoteAgentIdRef;
        this.remoteThreadIdRef = remoteThreadIdRef;
        this.remoteTicket = remoteTicket;
        this.remoteSequenceCounter = remoteSequenceCounter;
    }


    public int getClassIdRef() {
        return classIdRef;
    }


    public String getCollationValue() {
        return collationValue;
    }


    public String getEvent() {
        return event;
    }


    public RecordInvocationContext getInvocationContext() {
        return invocationContext;
    }


    public int getMethodIdRef() {
        return methodIdRef;
    }


    public int getObjIdRef() {
        return objIdRef;
    }


    public long getSequenceCounter() {
        return sequenceCounter;
    }


    public int getStackDepth() {
        return stackDepth;
    }


    public long getThreadCpuTime() {
        return threadCpuTime;
    }


    public int getThreadIdRef() {
        return threadIdRef;
    }


    public String getTicket() {
        return ticket;
    }


    public String getTime() {
        return time;
    }


    public String getTraceIdRef() {
        return traceIdRef;
    }


    public int getTransientClassIdRef() {
        return transientClassIdRef;
    }


    public int getTransientObjIdRef() {
        return transientObjIdRef;
    }


    public int getTransientThreadIdRef() {
        return transientThreadIdRef;
    }


    public void setCollationValue(String collationValue) {
        this.collationValue = collationValue;
    }


    public void setInvocationContext(RecordInvocationContext invocationContext) {
        this.invocationContext = invocationContext;
    }


    public void setSequenceCounter(long sequenceCounter) {
        this.sequenceCounter = sequenceCounter;
    }


    public void setStackDepth(int stackDepth) {
        this.stackDepth = stackDepth;
    }


    public void setThreadCpuTime(long threadCpuTime) {
        this.threadCpuTime = threadCpuTime;
    }


    public void setThreadIdRef(int threadIdRef) {
        this.threadIdRef = threadIdRef;
    }


    public void setTicket(String ticket) {
        this.ticket = ticket;
    }


    public void setTime(String time) {
        this.time = time;
    }


    public void setTraceIdRef(String traceIdRef) {
        this.traceIdRef = traceIdRef;
    }


    public void setTransientClassIdRef(int transientClassIdRef) {
        this.transientClassIdRef = transientClassIdRef;
    }


    public void setTransientObjIdRef(int transientObjIdRef) {
        this.transientObjIdRef = transientObjIdRef;
    }


    public void setTransientThreadIdRef(int transientThreadIdRef) {
        this.transientThreadIdRef = transientThreadIdRef;
    }


    public RecordVariable getReturnValue() {
        return returnValue;
    }


    public void setReturnValue(RecordVariable returnValue) {
        this.returnValue = returnValue;
    }


    public RecordArray getParameterList() {
        return parameterList;
    }


    public void setClassIdRef(int classIdRef) {
        this.classIdRef = classIdRef;
    }


    public void setParameterList(RecordArray parameterList) {
        this.parameterList = parameterList;
    }


    public String getRemoteNodeIdRef() {
        return remoteNodeIdRef;
    }


    public String getRemoteProcessIdRef() {
        return remoteProcessIdRef;
    }


    public long getRemoteSequenceCounter() {
        return remoteSequenceCounter;
    }


    public int getRemoteThreadIdRef() {
        return remoteThreadIdRef;
    }


    public long getRemoteTicket() {
        return remoteTicket;
    }


    public void setRemoteNodeIdRef(String remoteNodeIdRef) {
        this.remoteNodeIdRef = remoteNodeIdRef;
    }


    public void setRemoteProcessIdRef(String remoteProcessIdRef) {
        this.remoteProcessIdRef = remoteProcessIdRef;
    }


    public void setRemoteSequenceCounter(long remoteSequenceCounter) {
        this.remoteSequenceCounter = remoteSequenceCounter;
    }


    public void setRemoteThreadIdRef(int remoteThreadIdRef) {
        this.remoteThreadIdRef = remoteThreadIdRef;
    }


    public void setRemoteTicket(long remoteTicket) {
        this.remoteTicket = remoteTicket;
    }
}
