package org.eclipse.hyades.logging.parsers;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.lang.reflect.InvocationTargetException;
import java.util.Hashtable;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.hyades.logging.commons.Logger;
import org.eclipse.hyades.logging.core.IPreemptedLogger;
import org.eclipse.hyades.logging.core.LoggingAgent;

/**
 * This class handles initialization (e.g. calling the setUserInput() API) 
 * and invocation (e.g. calling the parse() API) of a remote log
 * file parser based on the parameters passed to the main() API and 
 * logs all parsed log record(s) to a Logging Agent.
 * 
 * Generally, this class is invoked by a local instance of the Workbench 
 * via the Agent Controller running on a remote host.
 * This class is invoked by a local instance of the Workbench by executing 
 * the Agent Controller's RemoteLogParserLoader executable based on the following 
 * Agent Controller configuration entry:
 * 
 * 	 <Application configuration="default" executable="RemoteLogParserLoader" path="%JAVA_PATH%\java.exe" location="%JAVA_PATH%">
 * 	 	<Parameter value="org.eclipse.hyades.logging.parsers.RemoteLogParserLoader" position="prepend"/>
 * 		<Variable name="CLASSPATH" value="%RASERVER_HOME%\lib\logutil.jar" position="prepend"/>
 * 	 </Application>
 * 
 * The possible parameters which may be passed to the main() API are as follows:
 * 
 * [org.eclipse.hyades.logging.parsers.RemoteLogParserLoader] <parser package name>.<parser class name> [<key 1>=<value 1> ... <key n>=<value n>] 
 * 
 * where:
 *     <parser class name> the class name of the remote log file parser
 *     <key n>             the name of the nth configuration variable required by the remote log file parser such as 'file_path'
 *     <value n>           the value of the nth configuration variable required by the remote log file parser such as 'C:\logs\access.log'
 * 
 * The parsed log record(s) are returned to the local instance of the Workbench via a Logging Agent and the Agent Controller.
 * 
 * All errors and/or exceptions are caught and returned to the local instance of the Workbench via a Logging Agent and the Agent Controller. 
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @version	August 27, 2004
 * @since   April 17, 2003
 */
public class RemoteLogParserLoader {

    /**
     * Handles the initialization (e.g. calling the setUserInput() API) and invocation 
     * (e.g. calling the parse() API) of a log file parser.  
     * 
     * The possible parameters which may be passed to the main() API are as follows:
     * 
     * [org.eclipse.hyades.logging.parsers.RemoteLogParserLoader] <parser package name>.<parser class name> [<key 1>=<value 1> ... <key n>=<value n>] 
     * 
     * where:
     *     <parser class name> the class name of the remote log file parser
     *     <key n>             the name of the nth configuration variable required by the remote log file parser such as 'file_path'
     *     <value n>           the value of the nth configuration variable required by the remote log file parser such as 'C:\logs\access.log'
     * 
     * The parsed log record(s) are returned to the local instance of the Workbench via a Logging Agent and the Agent Controller.
     * 
     * All errors and/or exceptions are caught and returned to the local instance of the Workbench via a Logging Agent and the Agent Controller. 
     * 
     * @param args the array of non-null String parameters for the initialization of a log file parser
     */
    public static void main(String[] args) {
    
    	//Set Apache logger:
    	System.setProperty("org.apache.commons.logging.Log", "org.eclipse.hyades.logging.commons.Logger");

        Log logger = null;
        long maxWaitTimeInMillis = 300000;

        try {
            maxWaitTimeInMillis = Long.parseLong(System.getProperty("org.eclipse.hyades.logging.parsers.maxWaitTimeInMillis"));
        }
        catch (Exception e) {
        }

        try {

            logger = LogFactory.getLog(ParserConstants.REMOTE_LOG_LOADER_AGENT_NAME);

            int parameterNum = args.length;

            //Throw an exception if no parameters have been passed:
            if (parameterNum == 0)
                throw new LogParserException(ParserUtilities.getResourceString("REMOTE_LOG_PARSER_PARAMETER_ERROR_"));

            //NOTE: The first parameter could be 'org.eclipse.hyades.logging.parsers.RemoteLogParserLoader' if specified in the serviceconfig.xml file:
            int parameterIndex = 0;

            //If the first parameter is 'org.eclipse.hyades.logging.parsers.RemoteLogParserLoader', use the next parameter:
            if (args[parameterIndex].trim().equals(ParserConstants.REMOTE_LOG_LOADER_CLASS))
                parameterIndex++;

            String parserClassName = null;

            //Retrieve and set the name of the log parser class or throw an exception if no class name parameter has been passed:
            //NOTE:  The parser class name MUST be of the format 'org.eclipse.hyades.logging.parsers.<parser class name>':
            if (parameterNum > parameterIndex)
                parserClassName = args[parameterIndex];
            else
                throw new LogParserException(ParserUtilities.getResourceString("REMOTE_LOG_PARSER_CLASSNAME_PARAMETER_ERROR_"));
           
            //Use the next parameter:
            parameterIndex++;

            //Throw an exception if no initialization parameter(s) for the log parser class have been passed:
            if (parameterNum <= parameterIndex)
                throw new LogParserException(ParserUtilities.getResourceString("REMOTE_LOG_PARSER_CONFIG_PARAMETER_ERROR_"));

            Hashtable parserParameters = new Hashtable();
            String keyValuePair = null;
            int equalsSignIndex = -1;

            //Retrieve and set the parameter(s) for the log parser class:
            while (parameterNum > parameterIndex) {

                keyValuePair = args[parameterIndex++];
                equalsSignIndex = keyValuePair.indexOf('=');

                if (equalsSignIndex != -1) {
                    parserParameters.put(keyValuePair.substring(0, equalsSignIndex), keyValuePair.substring(equalsSignIndex + 1));
                }
            }

            if (parserParameters.isEmpty())
                throw new LogParserException(ParserUtilities.getResourceString("REMOTE_LOG_PARSER_CONFIG_PARAMETER_ERROR_"));

            //Resolve the client's language and country codes for locale-specific message resolution:
            Locale clientLocale = null;
            
            if(parserParameters.containsKey(ParserConstants.CLIENT_LOCALE_LANGUAGE_KEY)){
                
                String clientLanguage = ((String)(parserParameters.get(ParserConstants.CLIENT_LOCALE_LANGUAGE_KEY)));
                
                if(parserParameters.containsKey(ParserConstants.CLIENT_LOCALE_COUNTRY_KEY)){

                    String clientCountry = ((String)(parserParameters.get(ParserConstants.CLIENT_LOCALE_LANGUAGE_KEY)));

                    if(parserParameters.containsKey(ParserConstants.CLIENT_LOCALE_VARIANT_KEY)){
                        clientLocale = new Locale(clientLanguage,clientCountry,((String)(parserParameters.get(ParserConstants.CLIENT_LOCALE_VARIANT_KEY))));
                    }
                    else{
                        clientLocale = new Locale(clientLanguage,clientCountry);
                    }
                }
                else{
                    clientLocale = new Locale(clientLanguage);
                }
            }
            
            
            //Assume this is a PreemptedLogger and recover if necessary:
            if (logger instanceof IPreemptedLogger) {

                ((IPreemptedLogger) (logger)).waitUntilLogging(maxWaitTimeInMillis);

                if (!((IPreemptedLogger) (logger)).isLogging()) {
                    throw new LogParserException(ParserUtilities.getResourceString("REMOTE_LOG_PARSER_MONITORING_ERROR_", clientLocale,parserClassName, String.valueOf(((double) (maxWaitTimeInMillis)) / 1000.0)));
                }
            }

            try {

                if (logger instanceof Logger) {
                    ((Logger) logger).setLevel(Logger.TRACE_LEVEL);
                }
               
                Class parserClass = Class.forName(parserClassName);
                Object parser = parserClass.newInstance();
                               
                parserClass.getMethod("setUserInput", new Class[] { parserParameters.getClass()}).invoke(parser, new Object[] { parserParameters });
               
                parserClass.getMethod("parse", new Class[] { Class.forName("org.apache.commons.logging.Log")}).invoke(parser, new Object[] { logger });
            }
            //If the called method throws a LogParserException:
            catch (InvocationTargetException i) {
                throw new LogParserException(ParserUtilities.getResourceString("REMOTE_LOG_PARSER_EXECUTION_ERROR_", clientLocale, parserClassName, i.getTargetException().getLocalizedMessage()));
            }
            catch (Throwable t) {               
                throw new LogParserException(ParserUtilities.getResourceString("REMOTE_LOG_PARSER_INITIALIZATION_ERROR_", clientLocale, parserClassName));
            }
        }
        catch (Throwable t) {
        	       
            LoggingAgent errorLogger = new LoggingAgent(ParserConstants.REMOTE_LOG_LOADER_ERROR_AGENT_NAME);

            errorLogger.waitUntilLogging(maxWaitTimeInMillis);

            if (errorLogger.isLogging())
                errorLogger.write(t.getMessage());
            else
                System.err.println(t.getMessage());

            errorLogger.deregister();
        }

        if ((logger != null) && (logger instanceof IPreemptedLogger)) {
            ((IPreemptedLogger) logger).finalize();
        }
    }
}
