/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.collection.correlation;

/**
 * <p>This interface captures two different kinds of relationships in the process of a remote invocation:
 * the relationship between a local caller (parent) and a local callee (child), and the relationship between
 * a sender and a receiver over a communication channel.  Examples of the former are local entry and exit events
 * in a local method invokation, and the examples of the latter are remote send, receive, reply, and return events
 * in a remote method invokation.  These two relationships are distinct in the sense that the former always inserts
 * or removes the active stack frame, with the parent frame always sitting below the active frame (exception for
 * the boundary case), whereas the latter may or may not modifies the stack.  In the case where the modification
 * does indeed take place, as in the case of receive and reply, the context data is not complete, i.e., there is
 * only a parent correlator without a child.  Similarly, call and return do not modify the stack.</p>
 * 
 * <p>Note that the role of each entity does not change over time.  In particular, this is true when a receiver
 * sends reply to a sender.  That is, the receiver will still be the receiver, even though it is acting as
 * a sender of the reply.</p>
 * 
 * @author Qiyan Li
 */
public class CorrelatorAssociation {
    private ICorrelatorData parent; // the parent correlator
    private ICorrelatorData child; // the child correlator

    /**
     * Set the parent (caller) context data of a local method invokation.
     * 
     * @param parent    the parent (caller) context data of a local method invokation
     */
    public void setParent(ICorrelatorData parentCorrelator) {
        parent = parentCorrelator;
    }

    /**
     * Get the parent (caller) context data of a local method invokation.
     * 
     * @return the parent (caller) context data of a local method invokation.
     */
    public ICorrelatorData getParent() {
        return parent;
    }

    /**
     * Set the child (callee) context data of a local method invokation.
     * 
     * @param child the child (callee) context data of a local method invokation
     */
    public void setChild(ICorrelatorData childCorrelator) {
        child = childCorrelator;
    }

    /**
     * Get the child (callee) context data of a local method invokation.
     * 
     * @return the child (callee) context data of a local method invokation.
     */
    public ICorrelatorData getChild() {
        return child;
    }

    /**
     * Re-initializes this object for later re-use.
     */
    public void initialize() {
        parent = child = null;
    }
}
