/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.core;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.uml2sd.ui.SDViewerPlugin;
import org.eclipse.hyades.uml2sd.ui.drawings.IGC;
import org.eclipse.hyades.uml2sd.ui.drawings.ISDPreferences;
import org.eclipse.hyades.uml2sd.ui.preferences.SDViewPref;

/**
 * The Frame class is the base sequence diagram graph nodes container.<br>
 * For instance, only one frame can be drawn in the View.<br>
 * Lifelines, Messages and Stop which are supposed to represent a Sequence diagram are
 * drawn in a Frame.<br>
 * Only the graph node added to their representing list will be drawn.
 * 
 * The lifelines are appended along the X axsis when added in a frame.<br>
 * The syncMessages are ordered along the Y axsis depending on the event occurrence they are attached to.<br>
 * 
 * @see org.eclipse.hyades.uml2sd.ui.core.Lifeline Lifeline for more event occurence details
 * @author sveyrier
 * @version 1.0
 */
public class BasicFrame extends GraphNode{
	
	/**
	 * Contains the max enlapsed time between two consecutive messages in the whole frame
	 */
	protected double maxTime = -50;
	/**
	 * Contains the min enlapsed time between two consecutive messages in the whole frame
	 */
	protected double minTime = -50;
	
	/**
	 * Indicate if the min and max enlapsed time between two consecutive 
	 * messages in the whole frame need to be computed
	 */
	protected boolean computeMinMax = true;
	
	/**
	 * Store the preference set by the user regarding the 
	 * external time. This flag is used determine if the
	 * min and max need to be recomputed in case this preference is changed.
	 */
	protected boolean lastExternalTimePref=SDViewPref.getInstance().excludeExternalTime();
	
	/**
	 * The greater event occurence created 
	 * on graph nodes drawn in this Frame
	 * This directly impact the Frame height
	 */
	protected int verticalIndex = 0;

	/**
	 * The index along the x axis where the next lifeline will is drawn
	 * This directly impact the Frame width
	 */
	protected int horizontalIndex = 0;

	protected boolean timeInfo = false;
	
	
	/**
	 * The current Frame visible area
	 */
	private int visibleAreaX;
	private int visibleAreaY;
	private int visibleAreaWidth;
	private int visibleAreaHeight;
	
	protected static ISDPreferences userPref = null;
	
	
	protected String unitName = null;
	
	protected int forceEventOccurrenceSpacing = -1;
	
	public HashMap nodes;
	public HashMap fnodes;
	public HashMap bnodes;
	
	public HashMap indexes;	
	public HashMap fSort;
	public HashMap bSort;
	
	
	/**
	 * Creates an empty frame.
	 */
	public BasicFrame()
	{
		nodes=new HashMap();
		fnodes=new HashMap();
		bnodes=new HashMap();
		indexes=new HashMap();
		bSort=new HashMap();
		fSort=new HashMap();
		
		Metrics.forcedEventSpacing = forceEventOccurrenceSpacing;
	}
	

	
	/**
	 * Reset the internal index of the first visible GraphNode for
	 * each ordered GraphNode lists (lifelines, and
	 * message lists)
	 *
	 */
	public void resetIndex()
	{
		Iterator it=indexes.keySet().iterator();
		while (it.hasNext())
		{
			Object nodeType=it.next();
			indexes.put(nodeType,new Integer(0));
		}
	}

	/**
	 * 
	 * Returns the greater event occurence 
	 * known by the Frame
	 * @return the greater event occurrence
	 */	
	protected int getMaxEventOccurrence()
	{
		return verticalIndex;
	}
	
	/**
	 * Set the greater event occurrence created
	 * in GraphNodes included in the frame
	 * @param eventOccurrence the new greater event occurrence
	 */
	protected void setMaxEventOccurrence(int eventOccurrence)
	{
			verticalIndex=eventOccurrence;
	}
	
	/**
	 * This method increase the lifeline place holder
	 * The return value is usually assign to a lifeline. This can be used to set the lifelines drawing order.
	 * Also, calling this method two times and assigning only the last given index to a lifeline will increase
	 * this lifeline draw spacing (2 times the default spacing) from the last added lifeline.
	 * @return a new lifeline index
	 */
	protected int getNewHorizontalIndex ()
	{
		return ++horizontalIndex;
	}
	
	/**
	 * Returns the current horizontal index
	 * @return the current horizontal index
	 * @see Frame#getNewHorizontalIndex() for horizontal index description
	 */
	protected int getHorizontalIndex ()
	{
		return horizontalIndex;
	}
	
	/**
	 * Add a GraphNode into the frame
	 * @param nodeToAdd the node to add
	 */
	public void addNode (GraphNode nodeToAdd)
	{
		computeMinMax=true;
		//Nothing to add
		if (nodeToAdd == null)
			return;
		
		if (nodes.get(nodeToAdd.getArrayId())==null)
		{
			nodes.put(nodeToAdd.getArrayId(),new ArrayList());
			indexes.put(nodeToAdd.getArrayId(),new Integer(0));
			fnodes.put(nodeToAdd.getArrayId(),new ArrayList());
			fSort.put(nodeToAdd.getArrayId(),new Boolean(false));
			if (nodeToAdd.getBackComparator()!=null)
			{
				bnodes.put(nodeToAdd.getArrayId(),new ArrayList());
				bSort.put(nodeToAdd.getArrayId(),new Boolean(false));
			}
		}
		
		List fNodeList=(List)fnodes.get(nodeToAdd.getArrayId());
		List bNodeList=(List)bnodes.get(nodeToAdd.getArrayId());
		if (fNodeList!=null&&fNodeList.size()>0)
		{
			//check if the nodes are added y ordered
			//if not, tag the list to sort it later (during draw)
			GraphNode node=(GraphNode)fNodeList.get(fNodeList.size()-1);
			Comparator fcomp = nodeToAdd.getComparator();
			Comparator bcomp = nodeToAdd.getBackComparator();
			if ((fcomp!=null)&&(fcomp.compare(node,nodeToAdd)==1))
			{
				fSort.put(nodeToAdd.getArrayId(),new Boolean(true));
			}
			if ((bcomp!=null)&&(bcomp.compare(node,nodeToAdd)==1))
			{
				bSort.put(nodeToAdd.getArrayId(),new Boolean(true));
			}
		}
		fNodeList.add(nodeToAdd);
		nodes.put(nodeToAdd.getArrayId(),fNodeList);
		fnodes.put(nodeToAdd.getArrayId(),fNodeList);
		if (nodeToAdd.getBackComparator()!=null)
		{	
			bNodeList.add(nodeToAdd);
			bnodes.put(nodeToAdd.getArrayId(),bNodeList);
		}
	}
	
	
	/**
	 * Computes the index of the first visible GraphNode for
	 * each ordered graph node lists depending on the visible 
	 * area given in parameter
	 *
	 * @param x visible area top left corner x coordinate
	 * @param y visible area top left corner y coordinate
	 * @param width visible area width
	 * @param height visible area height
	 */
	public void updateIndex(int x, int y, int width, int height)
	{
		visibleAreaX = x;
		visibleAreaY = y;
		visibleAreaWidth = width;
		visibleAreaHeight = height;
		
		if (SDViewerPlugin.debugIndex())
		{
			System.out.print("*****************************\n"); //$NON-NLS-1$
			System.out.print("Visible area position in virtual screen (x,y)= "+x+" "+y+"\n\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
		
		Iterator it=nodes.keySet().iterator();
		while (it.hasNext())
		{
			Object nodeType=it.next();
			int direction = 1;
			int drawIndex = ((Integer)indexes.get(nodeType)).intValue();
/*			if (x==0)
			{
				drawIndex = 0;
				indexes.put(nodeType,new Integer(drawIndex));
			}*/
			if ((nodes.get(nodeType) != null)&&(((List)nodes.get(nodeType)).size()>1))
			{
				if(((GraphNode)((List)nodes.get(nodeType)).get(drawIndex)).positiveDistanceToPoint(x,y))
					direction=-1;
				
				if (drawIndex==0)
					direction = 1;
			
				if ((direction == -1) && (bnodes.get(nodeType) != null))
				{
					GraphNode currentNode= (GraphNode)((List)nodes.get(nodeType)).get(drawIndex);
					drawIndex = Arrays.binarySearch(((List)bnodes.get(nodeType)).toArray(),
							((List)nodes.get(nodeType)).get(drawIndex),
							currentNode.getBackComparator());
					nodes.put(nodeType,bnodes.get(nodeType));
					if (drawIndex<0)
					{
						drawIndex=0;
						direction = 1;
					}
					else nodes.put(nodeType,bnodes.get(nodeType));
				}
				GraphNode prev=null;
				
				for (int i=drawIndex; i<((List)nodes.get(nodeType)).size()&&i>=0;i=i+direction)
				{
					drawIndex = i;
					indexes.put(nodeType,new Integer(i));
					
					GraphNode currentNode= (GraphNode)((List)nodes.get(nodeType)).get(i);
					
					if (prev==null)
						prev = currentNode;
					
					Comparator comp=currentNode.getComparator();
					HashMap sort=fSort;
					
					if (direction==-1)
					{
						if (currentNode.getBackComparator()!=null)
						{
							comp=currentNode.getBackComparator();
							sort=bSort;
						}
					}
					
					if (i<((List)nodes.get(nodeType)).size()-1)
					{
						GraphNode next= (GraphNode)((List)nodes.get(nodeType)).get(i+1);
						
						if ((comp!=null)&&(comp.compare(currentNode,next)==1))
							sort.put(nodeType,new Boolean(true));
					}
					if (direction ==1)
					{
						if(((GraphNode)((List)nodes.get(nodeType)).get(i)).positiveDistanceToPoint(x,y))
							break;
					}
					else 
					{
						if (currentNode.getBackComparator()==null)
						{
							if (currentNode.isVisible(x,y,width,height)
									&&!currentNode.positiveDistanceToPoint(x,y))
								break;
						}
						else 
						{
							if (currentNode.isVisible(x,y,width,height)
									&&!currentNode.positiveDistanceToPoint(x,y))
							{
								if ((comp!=null)&&(comp.compare(currentNode,prev)<=0))
									break;
							}
							else if ((comp!=null)&&(comp.compare(currentNode,prev)<=0))
								prev = currentNode;
						}
					}
				}
			
				nodes.put(nodeType,fnodes.get(nodeType));
				if ((bnodes.get(nodeType)!=null)&&(direction ==-1))
				{
//					nodes.put(nodeType,fnodes.get(nodeType));
					int index=((Integer)indexes.get(nodeType)).intValue();
					List list=(List)nodes.get(nodeType);
					List backList=(List)bnodes.get(nodeType);
					GraphNode currentNode = (GraphNode)(backList.get(index));
					if (index>0)
					{
						index = Arrays.binarySearch(list.toArray(),
								backList.get(index),
							currentNode.getComparator());
							if (index<0)
								index=0;
						indexes.put(nodeType,new Integer(index));
					}
				}
				
				for (int i=drawIndex; i<((List)nodes.get(nodeType)).size()&&i>=0;i++)
				{
					GraphNode toDraw = (GraphNode)((List)nodes.get(nodeType)).get(i);
					toDraw.updateInternalIndex(x, y, width, height);
					if (!toDraw.isVisible(x,y,width,height))
						break;
				}
			}
			if (SDViewerPlugin.debugIndex())
			{
				System.out.print("First drawn "+nodeType+" index = "+drawIndex+"\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$		
				System.out.print(nodeType + " found in "+0+" iterations\n"); //$NON-NLS-1$ //$NON-NLS-2$
			}
		}

		if (SDViewerPlugin.debugIndex())
		{
			System.out.print("*****************************\n"); //$NON-NLS-1$
		}
	}

	
	/**
	 * @return the frame x axis value in the containing view
	 * @see org.eclipse.hyades.uml2sd.ui.core.GraphNode#getX()
	 */
	public int getX()
	{
		return Metrics.FRAME_H_MARGIN;
	}
	
	/**
	 * @return the frame y axis value in the containing view
	 * @see org.eclipse.hyades.uml2sd.ui.core.GraphNode#getX()
	 */
	public int getY()
	{
		return Metrics.FRAME_V_MARGIN;
	}

	/**
	 * The frame width depends on the number of lifeline added in the frame
	 * @return the frame width  
	 * @see org.eclipse.hyades.uml2sd.ui.core.GraphNode#getWidth()
	 */
	public int getWidth()
	{
		if (horizontalIndex == 0)
			return Metrics.swimmingLaneWidth() + Metrics.LIFELINE_H_MAGIN*2- Metrics.FRAME_H_MARGIN -
				Metrics.LIFELINE_SPACING/2;
		else return horizontalIndex * Metrics.swimmingLaneWidth() + Metrics.LIFELINE_H_MAGIN*2 -	Metrics.LIFELINE_SPACING;
	}
	
	/**
	 * The Frame height depends on the maximum number of messsages added to a lifeline( Taking all lifelines into account)
	 * @return the frame height  
	 * @see org.eclipse.hyades.uml2sd.ui.core.GraphNode#getHeight()
	 */
	public int getHeight()
	{
		if (forceEventOccurrenceSpacing>=0)
			Metrics.forcedEventSpacing = forceEventOccurrenceSpacing;
		return  verticalIndex * (Metrics.getMessagesSpacing() + Metrics.getMessageFontHeigth()) +
				Metrics.LIFELINE_NAME_H_MARGIN +Metrics.FRAME_NAME_H_MARGIN + Metrics.getFrameFontHeigth() + Metrics.LIFELINE_VT_MAGIN +
				Metrics.LIFELINE_VB_MAGIN + Metrics.LIFELINE_NAME_H_MARGIN + Metrics.FRAME_NAME_H_MARGIN +	Metrics.getLifelineFontHeigth()*2;
	}
	
	
	/**
	 * Returns the graph node which contains the point given in parameter
	 * for the given graph node list and starting the iteration at the given index<br>
	 * WARNING: Only graph nodes with smaller coordinates
	 * than the current visible area can be returned.<br>
	 * 
	 * @param x the x coordinate of the point to test
	 * @param y the y coordinate of the point to test
	 * @param list the list to search in
	 * @param fromIndex list browsing starting point
	 * @return the graph node containing the point given in parameter, null otherwise
	 */ 
	protected GraphNode getNodeFromListAt(int x, int y, List list, int fromIndex)
	{
		if (list==null)
			return null;
		for (int i=fromIndex; i<list.size();i++)
		{
			GraphNode node = (GraphNode)list.get(i);
			//only lifeline list is x ordered
			//Stop browsing the list if the node is outside  the visible area
			//all others nodes will be not visibles			
			if ((node instanceof Lifeline)&&
				(node.getX()>visibleAreaX+visibleAreaWidth))
				break;
			if (node.getHeight()<0)
			{
				if (node.getY()+node.getHeight()>visibleAreaY+visibleAreaHeight)
					break;
			}
			else 
			{
				if (node.getY()>visibleAreaY+visibleAreaHeight)
					break;
			}
			if (node.contains(x, y))
				return node;
		}
		return null;
	}
	
	
	/**
	 * Returns the graph node which contains the point given in parameter
	 * WARNING: Only graph nodes in the current visible area can be returned
	 * @param x the x coordinate of the point to test
	 * @param y the y coordinate of the point to test
	 * @return the graph node containing the point given in parameter, null otherwise
	 */
	public GraphNode getNodeAt(int x, int y)
	{
		
		Iterator it=nodes.keySet().iterator();
		GraphNode node=null;
		while (it.hasNext())
		{
			Object nodeType=it.next();
			List list=(List)nodes.get(nodeType);
			int index=((Integer)indexes.get(nodeType)).intValue();
			node = getNodeFromListAt(x, y, list, index);
			if (node != null)
			{
				GraphNode internalNode=node.getInternalNodeAt(x,y,visibleAreaX,visibleAreaY,
						visibleAreaWidth,visibleAreaHeight);
				if (internalNode!=null)
					return internalNode;
				else return node;
			}
		}
		return null;
	}
	
	/**
	 * Draw the Frame rectangle
	 * @param context the context to draw to
	 */
	protected void drawFrame(IGC context)
	{
		context.setBackground(Frame.getUserPref().getBackGroundColor(ISDPreferences.PREF_FRAME));
		context.setForeground(Frame.getUserPref().getForeGroundColor(ISDPreferences.PREF_FRAME));
		
		int x=getX();
		int y=getY();
		int w=getWidth();
		int h=getHeight();

		//Draw the frame main rectangle		
		context.fillRectangle(x, y, w, h);
		context.drawRectangle(x, y, w, h);
		
		context.setBackground(Frame.getUserPref().getBackGroundColor(ISDPreferences.PREF_FRAME_NAME));
		context.setForeground(Frame.getUserPref().getForeGroundColor(ISDPreferences.PREF_FRAME_NAME));
		context.setFont(Frame.getUserPref().getFont(ISDPreferences.PREF_FRAME_NAME));
		
		int nameWidth = context.textExtent(getName()) + 2*Metrics.FRAME_NAME_V_MARGIN;
		int nameHeight = Metrics.getFrameFontHeigth()+
						+ Metrics.FRAME_NAME_H_MARGIN*2;
		
		//Draw the frame name area	
		if (nameWidth>w)
			nameWidth = w;
		
		int[] points={x, y, x + nameWidth, y, x + nameWidth, y -11 + nameHeight,
			x - 11 + nameWidth, y + nameHeight,	x, y + nameHeight, x, y + nameHeight};
		context.fillPolygon(points);
		context.drawPolygon(points);
		context.drawLine(x, y, x, y + nameHeight);

		context.setForeground(Frame.getUserPref().getFontColor(ISDPreferences.PREF_FRAME_NAME));
		context.drawTextTruncatedCentred(getName(),x, y,nameWidth-11 , nameHeight ,false);
		
		context.setBackground(Frame.getUserPref().getBackGroundColor(ISDPreferences.PREF_FRAME));
		context.setForeground(Frame.getUserPref().getForeGroundColor(ISDPreferences.PREF_FRAME));
	}

	/**
	 * Draws the Frame on the given context.<br>
	 * This method start width GraphNodes ordering if neeeded.<br>
	 * After, depending on the visible area, only visible GraphNodes are drawn.<br>
	 * @param context the context to draw to
	 * @see org.eclipse.hyades.uml2sd.ui.core.GraphNode#draw(IGC)
	 */
	public void draw (IGC context)
	{
		draw(context,true);
	}

	/**
	 * Draws the Frame on the given context.<br>
	 * This method start width GraphNodes ordering if neeeded.<br>
	 * After, depending on the visible area, only visible GraphNodes are drawn.<br>
	 * @param context the context to draw to
	 * @param drawFrame indicate if the frame rectangle need to be redrawn
	 * @see org.eclipse.hyades.uml2sd.ui.core.GraphNode#draw(IGC)
	 */
	protected void draw (IGC context, boolean drawFrame)
	{
		visibleAreaHeight = context.getVisibleHeight();
		visibleAreaWidth = context.getVisibleWidth();
		visibleAreaX= context.getContentsX();
		visibleAreaY= context.getContentsY();
		
		if (forceEventOccurrenceSpacing>=0)
			Metrics.forcedEventSpacing = forceEventOccurrenceSpacing;
		else Metrics.forcedEventSpacing = -1;
		if (userPref == null)
			return;
		//If the nodes have not been added ordered, the array is ordered
		Iterator it=fSort.keySet().iterator();
		while (it.hasNext())
		{
			Object nodeType=it.next();
			boolean sort = ((Boolean)fSort.get(nodeType)).booleanValue();
			if (sort)
			{
				Object[] temp=((List)fnodes.get(nodeType)).toArray();
				GraphNode node=(GraphNode)((List)nodes.get(nodeType)).get(0);
				Arrays.sort(temp,node.getComparator());
				fSort.put(nodeType,new Boolean(false));
				nodes.put(nodeType, Arrays.asList(temp));
				fnodes.put(nodeType, Arrays.asList(temp));
				if (SDViewerPlugin.debugSorting())
					System.out.print(nodeType+" array sorted\n"); //$NON-NLS-1$
			}
		}
		
		Iterator it2=bSort.keySet().iterator();
		while (it2.hasNext())
		{
			Object nodeType=it2.next();
			boolean sort = ((Boolean)bSort.get(nodeType)).booleanValue();
			if (sort)
			{
				Object[] temp=((List)bnodes.get(nodeType)).toArray();
				GraphNode node=(GraphNode)((List)nodes.get(nodeType)).get(0);
				Arrays.sort(temp,node.getBackComparator());
				bSort.put(nodeType,new Boolean(false));
				bnodes.put(nodeType, Arrays.asList(temp));
				if (SDViewerPlugin.debugSorting())
					System.out.print(nodeType+" back array sorted\n"); //$NON-NLS-1$
			}
		}
		
		if (SDViewerPlugin.debugDisplay())
		{
			System.out.print("*****************************\n"); //$NON-NLS-1$
		}
		if (drawFrame)
			drawFrame(context);
		
		int arrayStep =1;
		if ((Metrics.getMessageFontHeigth()+Metrics.MESSAGES_NAME_SPACING*2)*context.getZoom()<Metrics.MESSAGE_SIGNIFICANT_VSPACING)
			arrayStep = Math.round(Metrics.MESSAGE_SIGNIFICANT_VSPACING/((Metrics.getMessageFontHeigth()
				+Metrics.MESSAGES_NAME_SPACING*2)*context.getZoom()));
		
		int count=0;
		Iterator it3=fSort.keySet().iterator();
		while (it3.hasNext())
		{
			count=0;
			Object nodeType=it3.next();
			GraphNode node=(GraphNode)((List)nodes.get(nodeType)).get(0);
			context.setFont(Frame.getUserPref().getFont(node.prefId));
			int index=((Integer)indexes.get(nodeType)).intValue();
			count=drawNodes(context,(List)nodes.get(nodeType),index,arrayStep);
			if (SDViewerPlugin.debugDisplay())
				System.out.print(count +" "+nodeType+" drawn, starting from index "+index+"\r\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
		if (SDViewerPlugin.debugDisplay())
		{
			System.out.print("*****************************\n"); //$NON-NLS-1$
		}
	}
	
	/**
	 * Draw the GraphNode stored in the given list, starting at
	 * index startIndex with the given step
	 * @param context the context to draw to
	 * @param list the GraphNodes list
	 * @param startIndex the start index
	 * @param step the step to browse the list
	 * @return the number of GraphNodes drawn
	 */
	private int drawNodes(IGC context, List list, int startIndex, int step)
	{
		GraphNode last=null;
		int nodesCount=0;
		if (list.size()<0)
			return 0;
		
		GraphNode node=(GraphNode)list.get(0);
		context.setFont(Frame.getUserPref().getFont(node.prefId));
		Comparator comparator=node.getComparator();
		for (int i=startIndex; i<list.size(); i=i+step)
		{
			GraphNode toDraw= (GraphNode)list.get(i);
			if (i<list.size()-1)
			{
				GraphNode next= (GraphNode)list.get(i+1);
				if ((comparator!=null)&&(comparator.compare(toDraw,next)==1))
				{
					fSort.put(next.getArrayId(),new Boolean(true));
				}
			}
			int cx=context.getContentsX();
			int cy=context.getContentsY();
			int cw=context.getVisibleWidth();
			int ch=context.getVisibleHeight();
			//The arrays should be ordered, no needs to continue for this one
			if (!toDraw.isVisible(cx,cy,cw,ch)
				&&toDraw.positiveDistanceToPoint(cx+cw,cy+ch))
					break;
			//***Common*** nodes visibility
			if ((!toDraw.isSameAs(last)||toDraw.isSelected())&&
					(toDraw.isVisible(context.getContentsX(),context.getContentsY(),
					context.getVisibleWidth(),context.getVisibleHeight())))
			{
				nodesCount++;
				toDraw.draw(context);
			}
			last=toDraw;
		}
		return nodesCount;
	}
	
	/**
	 * Determine if the given point (px,py) is contained in the rectangle
	 * (x,y,width,height)
	 * @param x the rectangle x coordinate
	 * @param y the rectangle y coordinate
	 * @param width the rectangle width
	 * @param height the rectangle height
	 * @param px the x coordinate of the point to test 
	 * @param py the y coordinate of the point to test 
	 * @return true if contained false otherwise
	 */
	public static boolean contains(int x,int y, int width, int height, int px, int py)
	{
		int locX = x;
		int locY = y;
		int locWidth = width;
		int locHeight = height;
		
		if (width<0)
		{
			locX = locX + width;
			locWidth = -locWidth;
		}
		
		if (height<0)
		{
			locY = locY + height;
			locHeight = -locHeight;
		}
		return (px >= locX) && (py >= locY) && ((px - locX) < locWidth) && ((py - locY) < locHeight);
	}
	
	public static void setUserPref(ISDPreferences pref)
	{
		userPref = pref;
	}
	
	public static ISDPreferences getUserPref()
	{
		return userPref;
	}
	
	public void setTimeUnitName(String name)
	{
		unitName = name;
	}
	
	public String getTimeUnitName()
	{
		return unitName;
	}
	
	public void forceEventOccurrenceSpacing(int space)
	{
		forceEventOccurrenceSpacing = space;
	}
	
	/**
	 * Return the X coordinates of the frame visible area
	 * @return the X coordinates of the frame visible area
	 */
	public int getVisibleAreaX()
	{
		return visibleAreaX;
	}
	
	/**
	 * Return the X coordinates of the frame visible area
	 * @return the X coordinates of the frame visible area
	 */
	public int getVisibleAreaY()
	{
		return visibleAreaY;
	}
	
	/**
	 * Return the minimum time stored in the frame taking all GraphNodes into account
	 * @return the minimum GraphNode time 
	 */
	public double getMinTime()
	{
		if (lastExternalTimePref != SDViewPref.getInstance().excludeExternalTime())
		{
			lastExternalTimePref = SDViewPref.getInstance().excludeExternalTime();
			computeMinMax=true;
		}
		if (computeMinMax)
		{
			computeMinMax();
			computeMinMax = false;
		}
		return minTime;
	}
	
	/**
	 * Return the maximum time stored in the frame taking all GraphNodes into account
	 * @return the maximum GraphNode time 
	 */
	public double getMaxTime()
	{
		if (lastExternalTimePref != SDViewPref.getInstance().excludeExternalTime())
		{
			lastExternalTimePref = SDViewPref.getInstance().excludeExternalTime();
			computeMinMax=true;
		}
		if (computeMinMax)
		{
			computeMinMax();
			computeMinMax = false;
		}
		return maxTime;
	}
	
	/**
	 * Browse all the GraphNode to compute the mim and max times
	 * store in the Frame
	 */
	protected void computeMinMax()
	{
		List timeArray = buildTimeArray();
		for (int i=0; i<timeArray.size()-1;i++)
		{
			TimeEvent m1 = (TimeEvent)timeArray.get(i);
			TimeEvent m2 = (TimeEvent)timeArray.get(i+1);
			if (computeMinMax)
			{
				minTime = m2.getTime()-m1.getTime();
				maxTime=minTime;
				computeMinMax = false;
			}
			if ((m2.getTime()-m1.getTime())<minTime)
				minTime = m2.getTime()-m1.getTime();
			if ((m2.getTime()-m1.getTime())>maxTime)
				maxTime = m2.getTime()-m1.getTime();
		}
	}
	
	protected List buildTimeArray() 
	{

	    Iterator it=fSort.keySet().iterator();
	    List timeArray = new ArrayList();
		while (it.hasNext())
		{
			Object nodeType=it.next();
			GraphNode node=(GraphNode)((List)nodes.get(nodeType)).get(0);
			List list=(List)nodes.get(nodeType);
			for (int i=0;i<list.size();i++)
			{
				Object timedNode=list.get(i);
				if ((timedNode instanceof ITimeRange)&&((ITimeRange)timedNode).hasTimeInfo())
				{
					int event = ((GraphNode)list.get(i)).getStartOccurrence();
					double time = ((ITimeRange)list.get(i)).getFirstTime();
					TimeEvent f = new TimeEvent(time, event, (ITimeRange)list.get(i));
					timeArray.add(f);
					if (event!=((GraphNode)list.get(i)).getEndOccurrence())
					{
						event = ((AsyncMessage)list.get(i)).getEndOccurrence();
						time = ((ITimeRange)list.get(i)).getLastTime();
						f = new TimeEvent(time, event, (ITimeRange)list.get(i));
						timeArray.add(f);
					}
				}
			}
		}
		return timeArray;
	}
}
