/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.core;

import java.util.Comparator;

import org.eclipse.hyades.uml2sd.ui.drawings.IGC;
import org.eclipse.hyades.uml2sd.ui.drawings.ISDPreferences;



/**
 * The base class used for all UML2 graph nodes displayed in the Sequence Diagram SDWidget.
 * 
 * @author sveyrier
 * @version 1.0
 */

public abstract class GraphNode {
	
	
	protected int startEventOccurrence = 0;
	
	protected int endEventOccurrence = 0;
	
	/**
	 * Preference ColorId to use to draw font
	 */
	public String prefId = ISDPreferences.PREF_SYNC_MESS;
	
	/**
	 *  The selection state of the graph node.
	 */
	private boolean selected = false;
	
	/**
	 * The graph node name used to label the graph node in the View.
	 */
	private String 	name	 = ""; //$NON-NLS-1$
	
	/**
	 * Set the graph node name.<br>
	 * It is the name display in the view to label the graph node.
	 * @param the name to set
	 */
	public void setName(String nodeName)
	{
		name = nodeName;
	}

	/**
	 * Returns the graph node name.<br>
	 * It is the name display in the view to label the graph node.
	 * @return the graph node name
	 */
	public String getName()
	{
		return name;
	}
	
	
	/**
	 * Tags the the graph node has selected.<br>
	 * WARNING: This method is only used to draw the 
	 * graph node using the system selection colors. <br>
	 * To use the complete
	 * SDViewer selection mechanism (selection management, notification, etc..)
	 * see SDWidget class
	 * @see org.eclipse.hyades.uml2sd.ui.view.SDWidget#addSelection(GraphNode)
	 * @see org.eclipse.hyades.uml2sd.ui.view.SDWidget#removeSelection(GraphNode)
	 * @see org.eclipse.hyades.uml2sd.ui.view.SDWidget#clearSelection()
	 * @param  selection - true to set selected, false to set unselected
	 */
	public void setSelected(boolean selection)
	{
		selected = selection;
	}
	
	/**
	 * Returns true if the graph node is selected, false otherwise.<br>
	 * The returned value is used to highlight the graph node in the View.
	 * @return true if selected, false otherwise
	 * 
	 */
	public boolean isSelected()
	{
		return selected;
	}
	
	/**
	 * Returns true if the graph node contains the point given in parameter, return false otherwise.
	 * @param	x  the x coordinate of the point to test containment <br>
	 * 			y  the y coordinate of the point to test containment
	 * @return true if contained, false otherwise
	 */
	public boolean contains(int x, int y)
	{
		return false;
	}
	
	/**
	 * Returns the x coordinate of the graph node
	 * @return the x coordinate
	 */
	abstract public int getX();
	
	/**
	 * Returns the y coordinate of the graph node
	 * @return the y coordinate
	 */
	abstract public int getY();
	
	/**
	 * Returns the graph node height
	 * @return the graph node height
	 */
	abstract public int getHeight();
	
	/**
	 * Returns the graph node width
	 * @return the graph node width
	 */
	abstract public int getWidth();
	
	/**
	 * Draws the graph node in the given context
	 * @param context the graphical context to draw in
	 */
	abstract protected void draw(IGC context);
	
	
	/**
	 * Returns the GraphNode visibility for the given visible area.
	 * Wrong visibity calculation, may strongly impact drawing performance
	 * @param vx
	 * @param vy
	 * @param vwidth
	 * @param vheight
	 * @return true if visible false otherwise
	 */
	public boolean isVisible(int x, int y, int width, int height)
	{
		return false;
	}
	
	/**
	 * Return a comparator to sort the GraphNode of the same type
	 * This comparator is used to order the GraphNode array of the given node type.
	 * (see getArrayId).
	 * @return the comparator
	 */
	public Comparator getComparator()
	{
		return null;
	}
	
	/**
	 * If needed, return a different comparator to backward scan the GraphNode array
	 * @return the backward comparator or null if not needed
	 */
	public Comparator getBackComparator()
	{
		return null;
	}
	
	/**
	 * Compare two graphNodes
	 * @param node the node to compare to
	 * @return true if equal false otherwise
	 */
	public boolean isSameAs(GraphNode node)
	{
		return false;
	}
	
	/**
	 * Return the node type for all class instances.
	 * This id is used to store the same nodes kind in the same ordered array.
	 * @return the node type identifier
	 */
	public String getArrayId()
	{
		return null;
	}
	
	/**
	 * Return true if the distance from the GraphNode to the given point is positif
	 * @param x the point x coordinate
	 * @param y the point y coordinate
	 * @return true if positif false otherwise
	 */
	public boolean positiveDistanceToPoint(int x, int y)
	{
		return false;
	}
	
	/**
	 * Called back to perform extra work when the view content is moved and before drawing.
	 * Default implementation does nothing
	 * @param x the new visible content x coordinate
	 * @param y the new visible content y coordinate
	 * @param width the new visible content width
	 * @param height the new visible content height
	 */
	public void updateInternalIndex(int x, int y, int width, int height)
	{
	}
	
	/**
	 * If the GraphNode is composed of multiple GraphNode,
	 * return the one which contains the given point
	 * @param x the point to test x coordinate
	 * @param y the point to test y coordinate
	 * @param vx the visible area x coordinate
	 * @param vy the visible area y coordinate
	 * @param vw the visible area width
	 * @param vh the visible area height
	 * @return the GraphNode which contains this point if any, null otherwise
	 */
	public GraphNode getInternalNodeAt(int x, int y, int vx, int vy, int vw, int vh)
	{
		return null;
	}
	
	/**
	 * Returns the start event occurrence attached to this graphNode.
	 * @return the start event occurrence attached to the graphNode
	 */
	public int getStartOccurrence()
	{
		return startEventOccurrence;
	}
	
	/**
	 * Returns the end event occurrence attached to this graphNode
	 * @return the start event occurrence attached to the graphNode
	 */
	public int getEndOccurrence()
	{
		return endEventOccurrence;
	}
	
}
