/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.view;


import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

import org.eclipse.hyades.uml2sd.ui.SDViewerPlugin;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDCollapseProvider;
import org.eclipse.hyades.uml2sd.ui.core.BaseMessage;
import org.eclipse.hyades.uml2sd.ui.core.Frame;
import org.eclipse.hyades.uml2sd.ui.core.GraphNode;
import org.eclipse.hyades.uml2sd.ui.core.ITimeRange;
import org.eclipse.hyades.uml2sd.ui.core.Lifeline;
import org.eclipse.hyades.uml2sd.ui.core.Metrics;
import org.eclipse.hyades.uml2sd.ui.drawings.IColor;
import org.eclipse.hyades.uml2sd.ui.drawings.ISDPreferences;
import org.eclipse.hyades.uml2sd.ui.load.LoadersManager;
import org.eclipse.hyades.uml2sd.ui.preferences.SDViewPref;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.accessibility.ACC;
import org.eclipse.swt.accessibility.Accessible;
import org.eclipse.swt.accessibility.AccessibleAdapter;
import org.eclipse.swt.accessibility.AccessibleControlAdapter;
import org.eclipse.swt.accessibility.AccessibleControlEvent;
import org.eclipse.swt.accessibility.AccessibleEvent;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Caret;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.ui.part.ViewPart;

/**
 * 
 * @author sveyrier
 *
 */
public class SDWidget extends ScrollView implements SelectionListener, IPropertyChangeListener,
								DisposeListener,ITimeCompressionListener{
	
	private Frame frame;
	private Image overView = null;
	private static Color foreGroundColor 	= null;
	private static Color backGroundColor 	= null;
	private static Display display 			= null;
	private Menu   menu						= null;
	protected MenuItem	zoomIn				= null;
	protected MenuItem 	zoomOut				= null;
	protected SDWidgetSelectionProvider selProvider	=null;
	protected float  zoomValue				= 1;
	protected boolean zoomInMode			= false;
	protected boolean zoomOutMode			= false;
	
	protected List selectedNodeList = null;
	protected boolean multiSelection =false;
	
	private ViewPart site=null;
	
	public GraphNode currentGraphNode = null;
	
	private TimeCompressionBar timeBar = null;
	
	private DiagramToolTip toolTip = null;
	
	private Accessible accessible=null;
	
	protected GraphNode toolTipNode;
	
	protected GraphNode dragAndDrop=null;
	
	protected int focusedWidget = -1;
	
	private boolean getDragAndDrop=false;
	
	protected int dragX=0;
	protected int dragY=0;
	
	protected boolean reorderMode=false;
	
	Image collapaseCaretImg=null;
	Image arrowUpCaretImg=null;
	Image currentCaretImage=null;
	
	protected ISDCollapseProvider collapseProvider = null;
	
	protected Caret insertionCartet=null;
	
	protected ArrayList reorderList=null;
	
	public SDWidget(Composite c, int s)
	{
		super(c, s | SWT.NO_BACKGROUND,false);
		display = getDisplay();
		setOverviewEnabled(true);
		selectedNodeList = new ArrayList();
		selProvider=new SDWidgetSelectionProvider();
		SDViewPref.getInstance().addPropertyChangeListener(this);
		toolTip = new DiagramToolTip(getViewControl());
		super.addDisposeListener(this);
		
	    accessible =getViewControl().getAccessible(); 

        accessible.addAccessibleListener(new AccessibleAdapter() { 
                public void getName(AccessibleEvent e) { 
                        if (e.childID==ACC.CHILDID_SELF)
                        {
   //                     	e.result = "";
                        }
                        //Case toolTip
                        else if (e.childID==0)
                        {
                        	if (toolTipNode != null)
                			{
                				if (toolTipNode instanceof Lifeline)
                				{
                					Lifeline lifeline = (Lifeline)toolTipNode;
                					e.result =lifeline.getToolTipText();
                				}
                				else e.result =toolTipNode.getName();
                			}
                        }
                        else
                        {
                        	e.result =currentGraphNode.getName();
                        }
                } 
        }); 
	        
        accessible.addAccessibleControlListener(new AccessibleControlAdapter() { 
        	 public void getFocus(AccessibleControlEvent e) { 
                        if (focusedWidget == -1) e.childID = ACC.CHILDID_SELF; 
                        else e.childID = focusedWidget; 
        } 	
        	
                public void getRole(AccessibleControlEvent e) { 
                        switch (e.childID) { 
                                case ACC.CHILDID_SELF: e.detail = ACC.ROLE_CLIENT_AREA; break; 
                                case 0: e.detail = ACC.ROLE_TOOLTIP; break; 
                                case 1: e.detail = ACC.ROLE_LABEL; break; 
                        } 
                } 

                public void getState(AccessibleControlEvent e) { 
                        e.detail = ACC.STATE_FOCUSABLE; 
                        if (e.childID == ACC.CHILDID_SELF) { 
                                e.detail |= ACC.STATE_FOCUSED; 
                        } else { 
                                e.detail |= ACC.STATE_SELECTABLE; 
                                if (e.childID == focusedWidget) e.detail |= ACC.STATE_FOCUSED | ACC.STATE_SELECTED| ACC.STATE_CHECKED; 
                        } 
                }
        });

        insertionCartet=new Caret((Canvas)getViewControl(),SWT.NONE);
        insertionCartet.setVisible(false);
        
        collapaseCaretImg=SDViewerPlugin.getResourceImage("arrow_colapse.bmp").createImage();//$NON-NLS-1$
        arrowUpCaretImg=SDViewerPlugin.getResourceImage("arrow_up.bmp").createImage();//$NON-NLS-1$
        reorderList = new ArrayList();
	}
	
	public void setTimeBar (TimeCompressionBar bar)
	{
		if (bar != null)
		{
			timeBar = bar;
			timeBar.addTimeCompressionListener(this);
		}
	}
	
	protected void setCollapseProvider(ISDCollapseProvider provider)
	{
		collapseProvider = provider;
	}
	
	protected void keyPressedEvent(KeyEvent event) 
	{
		if (event.keyCode == SWT.CTRL)
			multiSelection =true;
		if ((event.character==' ')&&((zoomInMode)||(zoomOutMode)))
		{
			int cx = Math.round((getContentsX()+getVisibleWidth()/2)/zoomValue);
			int cy = Math.round((getContentsY()+getVisibleHeight()/2)/zoomValue);
			if (zoomInMode)
			{
				if (zoomValue<64)
				zoomValue = zoomValue*(float)1.25;				
			}
			else zoomValue = zoomValue/(float)1.25;
			int x = Math.round(cx*zoomValue-getVisibleWidth()/2);
			int y = Math.round(cy*zoomValue-getVisibleHeight()/2);
			setContentsPos(x ,y);
			if (timeBar != null)
				timeBar.setZoom(zoomValue);
			//redraw also resize the scrollView content
			redraw();
		}
	}
					
	protected void keyReleasedEvent(KeyEvent event) 
	{
		setFocus(-1);
		multiSelection = false;
		super.keyReleasedEvent(event);
		setFocus(1);
	}
	
	/**
	 * Resize the contents to insure the frame fit into the view
	 * @param frame the frame which will be drawn int the view
	 */
	public void resizeContents(Frame frame)
	{
		int width = Math.round((frame.getWidth()+2*Metrics.FRAME_H_MARGIN)* zoomValue);
		int height = Math.round((frame.getHeight()+2*Metrics.FRAME_V_MARGIN) * zoomValue);
		resizeContents(width,height);
	}
	
	protected boolean checkFocusOnChilds(Control childs)
	{
		if (childs instanceof Composite)
		{
			Control[] child=((Composite)childs).getChildren();
			for (int i=0; i< child.length;i++)
			{
				if (child[i].isFocusControl())
				{
					return true;
				}
				else checkFocusOnChilds(child[i]);
			}
		}
		return false;
	}
	
	public boolean isFocusControl()
	{
		Control[] child=getChildren();
		for (int i=0; i< child.length;i++)
		{
			if (child[i].isFocusControl())
			{
				return true;
			}
			else checkFocusOnChilds(child[i]);
		}
		return false;
	}
	
	/**
	 * The frame to render (the sequence diagram)
	 * @param theFrame the frame to display
	 */
	protected void setFrame(Frame theFrame, boolean resetPosition)
	{
		reorderList.clear();
		selectedNodeList.clear();
		selProvider.setSelection(new StructuredSelection());
		frame = theFrame;
		if (resetPosition)
		{
			setContentsPos(0,0);
			resizeContents(frame);
			redraw();
		}
		//prepare the old overview to be reused
		if (overView != null)
			overView.dispose();
		overView = null;
		resizeContents(frame);
	}
	
	/**
	 * Returns the current Frame (the sequence diagram container)
	 * @return the frame
	 */
	protected Frame getFrame()
	{
		return frame;
	}
	
	/**
	 * Returns the selection provider for the current sequence diagram
	 * @return the selection provider
	 */
	public ISelectionProvider getSelectionProvider()
	{
		return selProvider;
	}
	
	public boolean setContentsPos(int x, int y)
	{
		if (x<0)
			x=0;
		if (y<0)
			y=0;
		if (frame == null)
			return false;
		if (x+getVisibleWidth()>getContentsWidth())
			x=getContentsWidth()-getVisibleWidth();
		if (y+getVisibleHeight()>getContentsHeight())
			y=getContentsHeight()-getVisibleHeight();
		int x1 = Math.round(x / zoomValue);
		int y2 = Math.round(y / zoomValue);
		int width = Math.round(getVisibleWidth()/zoomValue);
		int height = Math.round(getVisibleHeight()/zoomValue);
		frame.updateIndex(x1, y2, width, height);

		if (insertionCartet!=null&&insertionCartet.isVisible())
	    	insertionCartet.setVisible(false);
		
		return super.setContentsPos(x, y);
	}
	
	
	protected void contentsMouseHover(MouseEvent event)
	{
		GraphNode graphNode = null;
		if (frame != null)
		{
			int x = Math.round(event.x / zoomValue);
			int y = Math.round(event.y / zoomValue);
			graphNode = frame.getNodeAt(x, y);
			if (graphNode != null)
			{
				toolTipNode=graphNode;
				String postfix="";//$NON-NLS-1$
				//Determine if the tooltip must show the time difference between the current mouse position and 
				//the last selected graphNode
				if ((currentGraphNode!=null)&&(currentGraphNode instanceof ITimeRange)&&
					(graphNode instanceof ITimeRange)&&currentGraphNode!=graphNode&&
					((ITimeRange)graphNode).hasTimeInfo()&&((ITimeRange)currentGraphNode).hasTimeInfo())
				{
					double delta = ((ITimeRange)graphNode).getLastTime()-((ITimeRange)currentGraphNode).getLastTime();
					postfix = "\n";//$NON-NLS-1$
					postfix = postfix+MessageFormat.format(SDViewerPlugin.getResourceString("STR_ELAPSED_TIME"), //$NON-NLS-1$
							  new Object[] {SDViewerPlugin.timeToString(delta)} );
					if (!((ITimeRange)currentGraphNode).hasTimeInfo())
						return;
				}
				//Even if we do not have to display a time difference, we need at least to display
				//the hovered graphNode name.
				if (graphNode instanceof Lifeline)
				{
					Lifeline lifeline = (Lifeline)graphNode;
					toolTip.showToolTip(lifeline.getToolTipText()+postfix);
					setFocus(0);
				}
				else 
				{
					toolTip.showToolTip(graphNode.getName()+postfix);
					setFocus(0);
				}
			}
			else toolTip.hideToolTip();
		}
	}
	
	protected void setFocus(int newFocusShape) { 
	    focusedWidget = newFocusShape; 
	    if (focusedWidget == -1) { 
	    	getViewControl().getAccessible().setFocus(ACC.CHILDID_SELF); 
	    } else { 
	    	getViewControl().getAccessible().setFocus(focusedWidget); 
	    } 
	} 
	
	/** Timer for auto_scroll feature */
	   private AutoScroll auto_scroll_=null;
	   /** TimerTask for auto_scroll feature !=null when auto scroll is running */
	   private Timer auto_scroll_timer_=null;
	  /** TimerTask for auto scroll feature. */
	   private static class AutoScroll extends TimerTask
	   {
	     public int dx_, dy_;
	     public SDWidget sv_;
	     public AutoScroll( SDWidget _sv, int _dx, int _dy )
	     {
	       sv_=_sv; dx_=_dx;dy_=_dy; 
	     }	
	     public void run()
	     {
	        Display.getDefault().asyncExec(new Runnable() {
	          public void run() {
	          	if (sv_.isDisposed())
	          		return;
	          	sv_.dragX+=dx_;
	          	sv_.dragY+=dy_;
	            sv_.scrollBy( dx_, dy_ );
	          }
	        });
	     }
	   }
	
	protected void contentsMouseMoveEvent(MouseEvent e)
	{
		 toolTip.hideToolTip();
//		 super.contentsMouseMoveEvent(e);
		 if (!(isFocusControl()||getViewControl().isFocusControl()))
		 {
			 Control[] child=getParent().getChildren();
			 for (int i=0; i< child.length;i++)
				{
					if (child[i].isFocusControl())
					{
						getViewControl().setFocus();
						break;
					}
				}
		 }
		 setFocus(-1); 
		 
		if( ((e.stateMask&SWT.BUTTON_MASK)!=0 )&&
				((dragAndDrop!=null)||getDragAndDrop)&&(reorderMode||collapseProvider!=null))
		{
			getDragAndDrop=false;
			if (currentGraphNode instanceof Lifeline)
				dragAndDrop = (Lifeline)currentGraphNode;
			if (dragAndDrop!=null)
			{
			int dx=0;
			int dy=0;
			if (e.x>getContentsX()+getVisibleWidth())
			{
				dx=e.x-(getContentsX()+getVisibleWidth());
			}
			else if (e.x<getContentsX())
			{
				dx=-getContentsX()+e.x;
			}
			if (e.y>getContentsY()+getVisibleHeight())
			{
				dy = e.y-(getContentsY()+getVisibleHeight());
			}
			else if (e.y<getContentsY())
			{
				dy =-getContentsY()+e.y;
			}
			dragX=e.x;
			dragY=e.y;
			if( dx!=0 || dy != 0 )
		    {
				if( auto_scroll_==null)
		         {
				if( auto_scroll_timer_==null)
					{
		             auto_scroll_timer_ = new Timer(true);
		           }
		           auto_scroll_ = new AutoScroll( this, dx,dy );
		           auto_scroll_timer_.schedule( auto_scroll_, 0, 75 ); 
		         } else {
		           auto_scroll_.dx_ = dx;
		           auto_scroll_.dy_ = dy;
		         }
		    }else if( auto_scroll_ != null )
	         {
	           auto_scroll_.cancel();
	           auto_scroll_ = null;
	         }
		    dragX = Math.round(e.x / zoomValue);
		    dragY = Math.round(e.y/ zoomValue);
		    redraw();
		    Lifeline node=frame.getCloserLifeline(dragX);
		    if ((node!=null)&&(node!=dragAndDrop))
		    {
		    	int y=0;
		    	int y1=0;
		    	int height =Metrics.getLifelineHeaderFontHeigth()+2*Metrics.LIFELINE_HEARDER_TEXT_V_MARGIN;
				int hMargin=Metrics.LIFELINE_VT_MAGIN/4;
				int x=node.getX();
				int width=node.getWidth();
		    	if (frame.getVisibleAreaY()<node.getY()+node.getHeight()-height-hMargin)
		    		y=contentsToViewY(Math.round((node.getY()+node.getHeight())*zoomValue));
		    	else y=Math.round(height*zoomValue);
		    	
		    	if (frame.getVisibleAreaY()<contentsToViewY(node.getY()-hMargin))
	    			y1=contentsToViewY(Math.round((node.getY()-hMargin)*zoomValue));
		    	else y1=Math.round(height*zoomValue);
		    	
		    	int rx=Math.round(x*zoomValue);
		    	
		    	insertionCartet.setVisible(true);
		    	if ((insertionCartet.getImage()!=null)&&
		    			(!insertionCartet.getImage().isDisposed()))
		    		insertionCartet.getImage().dispose();
				if (rx<=e.x&&
					Math.round(rx+(width*zoomValue))>=e.x)
				{
					if (collapseProvider!=null)
					{
						ImageData data=collapaseCaretImg.getImageData();
						data=data.scaledTo(Math.round(collapaseCaretImg.getBounds().width*zoomValue),
								Math.round(collapaseCaretImg.getBounds().height*zoomValue));
						currentCaretImage=new Image(Display.getCurrent(),data);
						insertionCartet.setImage(currentCaretImage);
						insertionCartet.setLocation(contentsToViewX(rx+Math.round((width/2)*zoomValue))
								-currentCaretImage.getBounds().width/2,y);
					}
				}
				else if (reorderMode)
				{
					if (rx>e.x)
					{
						if (node.getIndex()>1&&frame.getLifeline(node.getIndex()-2)==dragAndDrop)
							return;
						ImageData data=arrowUpCaretImg.getImageData();
						data=data.scaledTo(Math.round(arrowUpCaretImg.getBounds().width*zoomValue),
								Math.round(arrowUpCaretImg.getBounds().height*zoomValue));
						currentCaretImage=new Image(Display.getCurrent(),data);
						insertionCartet.setImage(currentCaretImage);
						insertionCartet.setLocation(contentsToViewX(Math.round((x-Metrics.LIFELINE_SPACING/2)
								*zoomValue))-currentCaretImage.getBounds().width/2,y1);
					}
					else 
					{
						if (node.getIndex()<frame.lifeLinesCount()&&frame.getLifeline(node.getIndex())==dragAndDrop)
							return;
						ImageData data=arrowUpCaretImg.getImageData();
						data=data.scaledTo(Math.round(arrowUpCaretImg.getBounds().width*zoomValue),
								Math.round(arrowUpCaretImg.getBounds().height*zoomValue));
						currentCaretImage=new Image(Display.getCurrent(),data);
						insertionCartet.setImage(currentCaretImage);
						insertionCartet.setLocation(contentsToViewX(Math.round((x+width+Metrics.LIFELINE_SPACING/2)*
								zoomValue))-currentCaretImage.getBounds().width/2+1,y1);
					}
				}
				}
		    	else insertionCartet.setVisible(false);
			}
		}
		else super.contentsMouseMoveEvent(e);
	}
	
	protected void contentsMouseUpEvent(MouseEvent event)
	{
		//Just in case the diagram highlight a time compression region
		//this region need to be released when clicking everywhere
		insertionCartet.setVisible(false);
		if (dragAndDrop!=null)
		{
			if ((overView!=null)&&(!overView.isDisposed()))
				overView.dispose();
			overView = null;
			Lifeline node=frame.getCloserLifeline(dragX);
			if (node!=null)
			{
				int rx=Math.round(node.getX()*zoomValue);
				if (rx<=event.x&&Math.round(rx+(node.getWidth()*zoomValue))>=event.x)
				{
					if ((collapseProvider!= null)&&(dragAndDrop!=node))
						collapseProvider.collapseTwoLifelines((Lifeline)dragAndDrop,node);
				}
				else if (rx<event.x)
				{
					frame.insertLifelineAfter((Lifeline)dragAndDrop,node);
					if (node.getIndex()<frame.lifeLinesCount())
					{
						Lifeline temp[]={(Lifeline)dragAndDrop,
								frame.getLifeline(node.getIndex())};
						reorderList.add(temp);
					}
					else 
					{
						Lifeline temp[]={(Lifeline)dragAndDrop,null};
						reorderList.add(temp);
					}
				}
				else
				{
					frame.insertLifelineBefore((Lifeline)dragAndDrop,node);
					Lifeline temp[]={(Lifeline)dragAndDrop,node};
					reorderList.add(temp);
				}
			}
		}
		dragAndDrop=null;
		redraw();
		if (frame == null) {
			return;
		}
		frame.resetTimeCompression();
		
		//reset auto scroll if it's engaged  
	    if( auto_scroll_!=null )
	    {
	      auto_scroll_.cancel();
	      auto_scroll_=null;
	    }
		super.contentsMouseUpEvent(event);
	}
	
	protected void contentsMouseDownEvent(MouseEvent event)
	{
		//Just in case the diagram highlight a time compression region
		//this region need to be released when clicking everywhere
		if (frame == null) {
			return;
		}
		frame.resetTimeCompression();
		
		if ((event.stateMask&SWT.CTRL)!=0)
		{
			multiSelection =true;
		}
		else multiSelection = false;
		
		if (((zoomInMode)||(zoomOutMode))&&(event.button ==1))
		{
			int cx = Math.round(event.x / zoomValue);
			int cy = Math.round(event.y / zoomValue);
			if (zoomInMode)
			{
				if (zoomValue<64)
				zoomValue = zoomValue*(float)1.25;				
			}
			else zoomValue = zoomValue/(float)1.25;
			int x = Math.round(cx*zoomValue-getVisibleWidth()/2);
			int y = Math.round(cy*zoomValue-getVisibleHeight()/2);
			setContentsPos(x ,y);
			if (timeBar != null)
				timeBar.setZoom(zoomValue);
			//redraw also resize the scrollView content
			redraw();
		}
		else// if (event.button ==1)
		{
			GraphNode node = null;
			if (frame != null)
			{
				int x = Math.round(event.x / zoomValue);
				int y = Math.round(event.y / zoomValue);
				node = frame.getNodeAt(x, y);
				if ((event.button ==1)||((node!=null)&&!node.isSelected()&&!multiSelection))
				{
					performSelection(node);
					currentGraphNode = node;
				}
				redraw();
			}
		}
		if (dragAndDrop==null)
			super.contentsMouseDownEvent(event);
		getDragAndDrop=true;

	}

	/**
	 * Highlight the given GraphNode<br>
	 * The GraphNode is then displayed using the system default selection color
	 * @param node the GraphNode to highlight
	 */
	protected void performSelection (GraphNode node)
	{	
		StructuredSelection selection;
		if (multiSelection)
		{
			if (node!=null)
			{
				if (selectedNodeList.contains(node))
				{
					removeSelection(node);
				}
				else 
				{
					addSelection(node);
				}
			}
			else return;
		}
		else
		{
			clearSelection();
			if(node != null)
			{
				addSelection(node);
			}
		} 
	}
	
	public List getSelection() {
		return selectedNodeList;
	}
	
	public void addSelection(GraphNode node)
	{
		if (node == null)
			return;
		selectedNodeList.add(node);
		node.setSelected(true);
		currentGraphNode = node;
		StructuredSelection selection  = new StructuredSelection(selectedNodeList);
		selProvider.setSelection(selection);
	}
	
	public void addSelection(List list)
	{
		selectedNodeList.addAll(list);
		for (int i = 0; i<list.size(); i++)
		{
			((GraphNode)list.get(i)).setSelected(true);
		}
		StructuredSelection selection  = new StructuredSelection(selectedNodeList);
		selProvider.setSelection(selection);
	}
	
//	public void setFocusNode(GraphNode node)
//	{
//		if (frame != null)
//			frame.setFocusNode(node);	
//	}
	
/*	public GraphNode getFocusNode()
	{
		if (frame != null)
			return frame.getFocusNode();
		else return null;
	}*/
	
	public void removeSelection(GraphNode node)
	{
		selectedNodeList.remove(node);
		node.setSelected(false);
		StructuredSelection selection  = new StructuredSelection(selectedNodeList);
		selProvider.setSelection(selection);
	}
	
	public void removeSelection(List list)
	{
		selectedNodeList.removeAll(list);
		for (int i = 0; i<list.size(); i++)
		{
			((GraphNode)list.get(i)).setSelected(false);
		}
		StructuredSelection selection  = new StructuredSelection(selectedNodeList);
		selProvider.setSelection(selection);
	}
	
	/**
	 * Clear the list of GraphNodes which must be drawn selected
	 *
	 */
	public void clearSelection()
	{
		for (int i = 0; i<selectedNodeList.size(); i++)
		{
			((GraphNode)selectedNodeList.get(i)).setSelected(false);
		}
		selectedNodeList.clear();
		selProvider.setSelection(new StructuredSelection());
	}
	
	public void setSite(ViewPart viewSite)
	{
		
		site=viewSite;
		site.getSite().setSelectionProvider(selProvider);
		String[] sdScropes={"org.eclipse.ui.UML2SDScope","org.eclipse.ui.globalScope"};//$NON-NLS-1$ //$NON-NLS-2$
		site.getSite().getKeyBindingService().setScopes(sdScropes);
	}
	
	/**
	 * 
	 * @param gc the context
	 */
	protected void drawContents(GC gc, int clipx, int clipy, int clipw, int cliph) 
	{
		if (frame == null)
		{
			gc.setBackground(getDisplay().getSystemColor(SWT.COLOR_WHITE));
			gc.fillRectangle(0,0,getVisibleWidth(),getVisibleHeight());
			gc.dispose();
			return;
		}
		else 
		{
			Frame.setUserPref(SDViewPref.getInstance());
		}
		update();
		Rectangle area = getClientArea();
		Image dbuffer = null;
		GC gcim = null;
			
		if (!SDViewerPlugin.debugDoubleBuffer())
		{
			try {
				if (dbuffer!=null)
					dbuffer.dispose();
				dbuffer = new Image(getDisplay(), area.width, area.height);
			} catch (Exception e) {
				System.out.println(e.toString());
			}
			gcim = new GC(dbuffer);
		}
		else 
			gcim=gc;
		NGC context = new NGC (this, gcim);
		
		//Set the metrics to use for lifeline text and message text
		//using the Graphical Context
		Metrics.setLifelineFontHeight(context.getFontHeight(SDViewPref.getInstance().getFont(SDViewPref.PREF_LIFELINE)));
		Metrics.setLifelineFontWidth(context.getFontWidth(SDViewPref.getInstance().getFont(SDViewPref.PREF_LIFELINE)));
		Metrics.setLifelineWidth(SDViewPref.getInstance().getLifelineWidth());
		Metrics.setFrameFontHeight(context.getFontHeight(Frame.getUserPref().getFont(ISDPreferences.PREF_FRAME_NAME)));
		Metrics.setLifelineHeaderFontHeight(context.getFontHeight(Frame.getUserPref().getFont(ISDPreferences.PREF_LIFELINE_HEADER)));
		
		int syncMessFontH = context.getFontHeight(SDViewPref.getInstance().getFont(SDViewPref.PREF_SYNC_MESS));
		int syncMessRetFontH = context.getFontHeight(SDViewPref.getInstance().getFont(SDViewPref.PREF_SYNC_MESS_RET));
		int asyncMessFontH = context.getFontHeight(SDViewPref.getInstance().getFont(SDViewPref.PREF_ASYNC_MESS));
		int asyncMessRetFontH = context.getFontHeight(SDViewPref.getInstance().getFont(SDViewPref.PREF_ASYNC_MESS_RET));
		
		int messageFontHeight = 0;
		if (syncMessFontH>syncMessRetFontH)
			messageFontHeight = syncMessFontH;
		else messageFontHeight = syncMessRetFontH;
		if (messageFontHeight<asyncMessFontH)
			messageFontHeight=asyncMessFontH;
		if (messageFontHeight<asyncMessRetFontH)
			messageFontHeight=asyncMessRetFontH;
		Metrics.setMessageFontHeight(messageFontHeight);
		context.setFont(SDViewPref.getInstance().getFont(SDViewPref.PREF_LIFELINE));
		
		
		int width = Math.round((frame.getWidth()+2*Metrics.FRAME_H_MARGIN)* zoomValue);
		int height = Math.round((frame.getHeight()+2*Metrics.FRAME_V_MARGIN) * zoomValue);
/*		if (width<area.width)
			width=area.width;
		if (height<area.height)
			height=area.height;*/
		resizeContents(width,height);
		
		context.setBackground(Frame.getUserPref().getBackGroundColor(ISDPreferences.PREF_FRAME));
		context.fillRectangle(0,0,getContentsWidth(),Metrics.FRAME_V_MARGIN);
		context.fillRectangle(0,0,frame.getX(),getContentsHeight());
		context.fillRectangle(frame.getX()+frame.getWidth()+1,0,
								getContentsWidth()-(frame.getX()+frame.getWidth()+1),getContentsHeight());
		context.fillRectangle(0,frame.getY()+frame.getHeight()+1,getContentsWidth(),
							getContentsHeight()-(frame.getY()+frame.getHeight()+1));
		gcim.setLineWidth(1);
		
		frame.draw(context); 
		if (dragAndDrop instanceof Lifeline)
		{
			Lifeline node=(Lifeline)dragAndDrop;
			boolean isSelected=dragAndDrop.isSelected();
			node.setSelected(false);
			node.draw(context,dragX,dragY);
			node.setSelected(isSelected);
		}
		if (!SDViewerPlugin.debugDoubleBuffer())
		{
			try {
				gc.drawImage(dbuffer, 0, 0, area.width, area.height, 0, 0,  area.width, area.height);
			} catch (Exception e) {
				System.out.println(e.getMessage());
			}
			gcim.dispose();
			dbuffer.dispose();
		}
		gc.dispose();
		context.dispose();
		setHScrollBarIncrement(Math.round(SDViewPref.getInstance().getLifelineWidth()/2*zoomValue));
		setVScrollBarIncrement(Math.round(Metrics.getMessagesSpacing()*zoomValue));
		if ((timeBar != null)&&(frame.hasTimeInfo()))
		{
			timeBar.resizeContents(9,height+getHorizontalBarHeight());
			timeBar.setContentsPos(getContentsX(),getContentsY());
			timeBar.redraw();
			timeBar.update();
		}
		float xRatio=getContentsWidth()/getVisibleWidth();
		float yRatio=getContentsHeight()/getVisibleHeight();
		if (yRatio>xRatio)
		{
			setOverviewSize((int)(getVisibleHeight()*0.75));
		}
		else
		{
			setOverviewSize((int)(getVisibleWidth()*0.75));
		}
	} 
	
	/**
	 * Returns the GraphNode overView the mouse if any
	 * @return the GraphNode
	 */
	public GraphNode getMouseOverNode()
	{
		return currentGraphNode;
	}
	
	public void widgetDefaultSelected(SelectionEvent event)
	{
	}
	
	public void widgetSelected (SelectionEvent event)
	{
		if (event.widget == zoomIn)
			zoomValue = zoomValue*2;
		else if (event.widget == zoomOut)
			zoomValue = zoomValue/2;
		else {
			//SearchFilterDialog tt = new SearchFilterDialog(null);//display.getActiveShell());
		}
		redraw();
	}
	
	public void setZoomInMode(boolean value)
	{
		if (value)
			setZoomOutMode(false);
		zoomInMode = value;
	}
	
	public void setZoomOutMode(boolean value)
	{
		if (value)
			setZoomInMode(false);
		zoomOutMode = value;
	}
	
	/**
	 * Moves the Sequence diagram to ensure the given node 
	 * is visible and draw it selected
	 * @param node the GraphNode to move to
	 */
	public void moveTo(GraphNode node)
	{
		if (node == null)
			return;
		clearSelection();
		addSelection(node);
		ensureVisible(node);
	}
	
	/**
	 * Moves the Sequence diagram to ensure the given node 
	 * is visible
	 * @param node the GraphNode to move to
	 */
	public void ensureVisible(GraphNode node)
	{
		if (node == null)
			return;
		int x = Math.round(node.getX() * zoomValue);
		int y = Math.round(node.getY() * zoomValue);
		int width = Math.round(node.getWidth()* zoomValue);
		int height = Math.round(node.getHeight()* zoomValue);
		if (node instanceof BaseMessage)
		{
			if (height==0)
			{
				int header=Metrics.LIFELINE_HEARDER_TEXT_V_MARGIN*2+Metrics.getLifelineHeaderFontHeigth();
				height =  -Math.round((Metrics.getMessagesSpacing()+header)* zoomValue);
				y=y+Math.round(Metrics.SYNC_INTERNAL_MESSAGE_HEIGHT*zoomValue);
			}
		}
		if (node instanceof Lifeline)
		{
			y=getContentsY();
			height=getVisibleHeight();
		}
		ensureVisible(x,y,width,height,SWT.CENTER,true);
		redraw();
	}
	
	public float getZoomFactor()
	{
		return zoomValue; 
	}
	
	/**
	 * Called when property changed occurs in the preference page.
	 * "PREFOK" is fired when the user press the ok or apply button
	 * @param e the PropertyChangeEvent
	 */
	public void propertyChange(PropertyChangeEvent e)
	{
		if (frame !=null && !isDisposed()) {
			frame.resetTimeCompression();
		}
		if (e.getProperty().equals("PREFOK")) //$NON-NLS-1$
		{
			//Prepare the overview to be reused for the new
			//settings (especially the colors)
			if (overView!= null)
				overView.dispose();
			overView = null;
			redraw();
		}
	}
	
	public void widgetDisposed(DisposeEvent e)
	{
		if (overView!=null)
			overView.dispose();
		super.removeDisposeListener(this);
		if ((currentCaretImage!=null)&&(!currentCaretImage.isDisposed()))
			currentCaretImage.dispose();
		if ((arrowUpCaretImg!=null)&&(!arrowUpCaretImg.isDisposed()))
			arrowUpCaretImg.dispose();
		if ((collapaseCaretImg!=null)&&(!collapaseCaretImg.isDisposed()))
			collapaseCaretImg.dispose();
		SDViewPref.getInstance().removePropertyChangeListener(this);
		LoadersManager lm = LoadersManager.getLoadersManager();
		if (site != null && site instanceof SDView) {
			((SDView)site).resetProviders();
			if (lm != null) 
				lm.resetLoader(((SDView)site).getViewSite().getId());
		}
	}
	
	protected void drawOverview( GC gc, Rectangle r) 
	{
      float oldzoom=zoomValue;
      if (getContentsWidth() > getContentsHeight())
	  	zoomValue=(float)r.width/(float)getContentsWidth()*oldzoom;
	  else zoomValue=(float)r.height/(float)getContentsHeight()*oldzoom;
      if ((overView != null)&&((r.width!=overView.getBounds().width)||(r.height!=overView.getBounds().height)))
	  {
		  overView.dispose();
		  overView = null;
	  }
	  if (overView == null)
	  {  
		  int backX = getContentsX();
		  int backY = getContentsY();
		  setContentsPos(0,0);
		  overView = new Image(getDisplay(), r.width, r.height);
		  GC gcim = new GC(overView);
		  NGC context = new NGC (this, gcim);
			context.setBackground(SDViewPref.getInstance().getBackGroundColor(SDViewPref.PREF_FRAME));
		  frame.draw(context);
		  setContentsPos(backX,backY);
		  gcim.dispose();
		  context.dispose();
	  }
	  if ((overView != null)&&(r.width==overView.getBounds().width)&&(r.height==overView.getBounds().height))
	  	gc.drawImage(overView, 0, 0, r.width, r.height, 0, 0,  r.width, r.height);
	  
	  zoomValue=oldzoom;
	  
	  super.drawOverview(gc,r);
	}
	
	public void deltaSelected(Lifeline lifeline, int startEvent, int nbEvent, IColor color)
	{
		frame.highlightTimeCompression(lifeline,startEvent,nbEvent,color);
		ensureVisible(lifeline);
		redraw();
		update();
	}
	
	public void resetZoomFactor()
	{
		int currentX = Math.round(getContentsX()/zoomValue);
		int currentY = Math.round(getContentsY()/zoomValue);
		zoomValue = 1;
		if (timeBar != null && !timeBar.isDisposed()) {
			timeBar.setZoom(zoomValue);
		}
		redraw();
		update();
		setContentsPos(currentX,currentY);
	}
	
	/**
	 * Enable or disable the lifeline reodering using Drag and Drop
	 * @param mode - true to enable false otherwise
	 */
	public void setReorderMode(boolean mode)
	{
		reorderMode=mode;
	}
	
	
	/**
	 * Return the lifelines reorder sequence (using Drag and Drop)
	 * if the the reorder mode is turn on.
	 * Each ArryList element is of type Lifeline[2]
	 * with Lifeline[0] inserted before Lifeline[1] in the diagram
	 * @return - the reoder sequence
	 */
	public ArrayList getLifelineReoderList()
	{
		return reorderList;
	}
}
