/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCFullTraceObject;
import org.eclipse.hyades.models.trace.TRCHeapObject;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCObject;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.models.trace.TracePackage;
import org.eclipse.hyades.models.trace.impl.TRCClassImpl;
import org.eclipse.hyades.models.trace.impl.TRCMethodImpl;
import org.eclipse.hyades.models.trace.impl.TRCPackageImpl;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.hyades.trace.views.adapter.internal.MemoryStatisticPage;
import org.eclipse.hyades.trace.views.util.internal.ColumnData;
import org.eclipse.hyades.trace.views.util.internal.PerftraceUtil;
import org.eclipse.hyades.trace.views.util.internal.StatisticTableColumnInfo;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.custom.TableTreeItem;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;

/*
 * This view contains memory statistics data for
 * the package level, class level, and instance level.
 */
public class MemoryStatisticView extends MultiLevelStatisticView{

	private static int _level = DEFAULT_LEVEL;	
	
	public static final int INSTANCE_LEVEL = 3;		
	
	private final TRCPackageImpl.TRCPackageSnapshot pdelta1 = new TRCPackageImpl.TRCPackageSnapshot();
	private final TRCPackageImpl.TRCPackageSnapshot pdelta2 = new TRCPackageImpl.TRCPackageSnapshot();
	private final TRCClassImpl.TRCClassSnapshot cdelta1 = new TRCClassImpl.TRCClassSnapshot();
	private final TRCClassImpl.TRCClassSnapshot cdelta2 = new TRCClassImpl.TRCClassSnapshot();
	private final TRCMethodImpl.TRCMethodSnapshot mdelta1 = new TRCMethodImpl.TRCMethodSnapshot();
	private final TRCMethodImpl.TRCMethodSnapshot mdelta2 = new TRCMethodImpl.TRCMethodSnapshot();	
	
	public MemoryStatisticView(Composite parent, TraceViewerPage page){
		super(parent, page);
	}
	
	protected int getClassPosForLevel(int pos)
	{
        if (getLevel()== MemoryStatisticView.PACKAGE_LEVEL)
        {
           	switch (pos)
			{
           		case 2:
           			return 3;
           		case 3:
           			return 4;
           		case 4:
           			return 5;
           		case 5:
           			return 6;
           		case 6:
           			return 7;
			}
        }
        else if (getLevel() == MemoryStatisticView.INSTANCE_LEVEL)
        {
           	switch (pos)
			{
           		case 3:
           			return 5;
           		case 4:
           			return 6;
           		case 5:
           			return 7;
           		case 6:
           			return -1;
           		case 7:
           			return -1;           			
			}        	
        }
        return pos;
	}
	
	
	public class MemoryStatisticContentProvider extends MultiLevelStatisticContentProvider
	{
		
		public Object[] getChildren(Object element)
		{
			tmpList.clear();
			
			if(element instanceof TRCPackage)
				return ((TRCPackage)element).getClasses().toArray();
			else if (element instanceof TRCClass && getLevel() == MemoryStatisticView.INSTANCE_LEVEL)
			{
				TRCClass cls = (TRCClass)element;
				List list = cls.getObjects();

				/*
				 * #48680: Use only TRCFullTraceObjects if they
				 * are present. 
				 */
				Iterator iter = cls.getObjects().iterator();
				while (iter.hasNext()) {
					Object obj = iter.next();
					if (obj instanceof TRCFullTraceObject) {
						tmpList.add(obj);
					}
				}
					
				/*
				 * #48680: If there are no TRCFullTraceObjects,
				 * then include the TRCHeapObjects.
				 */
				if (tmpList.isEmpty()) {
					tmpList.addAll(list);
				}
					
				return tmpList.toArray();
			}
			else
				return tmpList.toArray(); 	
		}
		
		public Object getParent(Object element) {
			return null;
		}		

		public boolean hasChildren(Object element) {
			if (element instanceof TRCPackage)
				return true;
			else if(element instanceof TRCClass && getLevel() == MemoryStatisticView.INSTANCE_LEVEL) {
				TRCClass cls = (TRCClass)element;
			    return (cls.getObjects().size() > 0);
			}			
 
			return false;
		}
		
		/**
		 * return the list of elements to display in the table tree
		 */
		public Object[] getElements(Object inputElement) {

			if (getLevel() == MemoryStatisticView.PACKAGE_LEVEL)
			{
				Object[] packages = PerftraceUtil.getAllPackages(_page.getMOFObject());
				ArrayList list = new ArrayList();
				for(int idx=0; idx<packages.length; idx++)
				{
					TRCPackage pack = (TRCPackage)packages[idx];
					if(pack.getClasses().size() > 0)
					   list.add(pack);
				}
			
				return list.toArray();
			}
			else if (getLevel() == MemoryStatisticView.CLASS_LEVEL)
			{
				return PerftraceUtil.getAllClasses(_page.getMOFObject());
			}
			else if (getLevel() == MemoryStatisticView.INSTANCE_LEVEL)
			{
				return PerftraceUtil.getAllClasses(_page.getMOFObject());
			}
			else
			{
				return new ArrayList().toArray();
			}
		}
	}
	
	public class MemoryStatisticLabelProvider extends MultiLevelStatisticLabelProvider
	{
		public MemoryStatisticLabelProvider(StatisticView viewer) {
			super(viewer);
		}
		
		protected String getColumnTextPackageLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(obj instanceof TRCPackage)	
			{
				TRCPackage p = (TRCPackage) obj;
				TRCPackageImpl.TRCPackageSnapshot ps = (TRCPackageImpl.TRCPackageSnapshot)p.retrieveSnapshot();
				
				switch(pos)
				{
					case 0://package name
						return PerftraceUtil.getPackageName(p, _page.getMOFObject()).toString();
					case 1: //new item column
						return "";					
					case 2: //total inst
					
					  if(!info.isDeltaColumn())
					  {
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)ps.getTotalInstances()/_totalInst);
						  }
						  else
						  {
							  return String.valueOf(ps.getTotalInstances());
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES);
						return String.valueOf(pdelta1.getTotalInstances());					  	
					  }
					  					 
					case 3: //live inst
					
					  if(!info.isDeltaColumn())
					  {					
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)(ps.getTotalInstances() - ps.getCollectedInstances())/_activeInst);
						  }
						  else
						  {					
							  return String.valueOf(ps.getTotalInstances() - ps.getCollectedInstances());
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES);
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_INSTANCES);						
						return String.valueOf(pdelta1.getTotalInstances() - pdelta1.getCollectedInstances());					  	
					  }
					  
					case 4://collected
					
					  if(!info.isDeltaColumn())
					  {					
						  if(isShowPercent())
						  {
						  	  int inst = ps.getTotalInstances();
						  	  if(inst == 0) inst = 1;
						  	  
						  	  return TString.formatAsPercentage((double)ps.getCollectedInstances()/inst);
						  }
						  else
						  {					
						  		return String.valueOf(ps.getCollectedInstances());
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_INSTANCES);
						return String.valueOf(pdelta1.getCollectedInstances());					  	
					  }
					  
					case 5: //total size
					
					  if(!info.isDeltaColumn())
					  {										
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)ps.getTotalSize()/_totalSize);
						  }
						  else
						  {					
							  return String.valueOf(ps.getTotalSize());
						  }
					  }
					  else
					  {
					  	p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_SIZE);
					  	return String.valueOf(pdelta1.getTotalSize());					  	
					  }
					  
					  
					case 6: //active size
					  if(!info.isDeltaColumn())
					  {															
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)(ps.getTotalSize() - ps.getCollectedSize())/_activeSize);
						  }
						  else
						  {					
						  	return String.valueOf(ps.getTotalSize() - ps.getCollectedSize());
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_SIZE);
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_SIZE);
						
						return String.valueOf(pdelta1.getTotalSize() - pdelta1.getCollectedSize());					  	
					  }
				}
			}
			else
			{
				return getColumnTextClassLevel(obj, col);
			}
            							
			return "";
		}
		
		protected String getColumnTextClassLevel(Object obj, int col){
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = getClassPosForLevel(info.getColumnData().getInitalPos());			
            
			if (obj instanceof TRCClass) {
				TRCClass clas = (TRCClass) obj;
				TRCClassImpl.TRCClassSnapshot cs = (TRCClassImpl.TRCClassSnapshot)clas.retrieveSnapshot();
				
				switch (pos) {
					case 0 : //class name
						return ((TRCClass) obj).getName();
					case 1: //new item column
						return "";			
					case 2 :
   						return PerftraceUtil.getPackageName(clas.getPackage(), _page.getMOFObject()).toString();
					case 3 : //total inst
					  if(!info.isDeltaColumn())
					  {					
						  if(isShowPercent())
						  {
						  	  return PerftraceUtil.formatAsPercentage(((double)cs.getTotalInstances())/_totalInst);
						  }
						  else
						  {													  					
							return String.valueOf(cs.getTotalInstances());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						return String.valueOf(cdelta1.getTotalInstances());					  	
					  }

					case 4 : //live inst
					  if(!info.isDeltaColumn())
					  {										
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(((double)(cs.getTotalInstances()
									- cs.getCollectedInstances()))/_totalInst);
						  }
						  else
						  {													  										
							return String.valueOf(cs.getTotalInstances()- cs.getCollectedInstances());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						return String.valueOf(cdelta1.getTotalInstances() - cdelta1.getCollectedInstances());					  	
					  }

					case 5 : //collected
					  if(!info.isDeltaColumn())
					  {															
						  if(isShowPercent())
						  {
						  	  int size = cs.getTotalInstances();
						  	  if(size == 0) size = 1;
						  	   
						  	  return TString.formatAsPercentage(((double)cs.getCollectedInstances())/size);
						  }
						  else
						  {																						
							return String.valueOf(cs.getCollectedInstances());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						return String.valueOf(cdelta1.getCollectedInstances());					  	
					  }

					case 6 : //total size
					  if(!info.isDeltaColumn())
					  {																				
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(((double)cs.getTotalSize())/_totalSize);
						  }
						  else
						  {													  										
							return String.valueOf(cs.getTotalSize());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
						return String.valueOf(cdelta1.getTotalSize());					  	
					  }

					case 7 : //active size
					
					  if(!info.isDeltaColumn())
					  {																									
					  	  long size = cs.getTotalSize() - cs.getCollectedSize();					
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(((double) size)/_activeSize);
						  }
						  else
						  {													  															
							return String.valueOf(size);
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_SIZE);
						return String.valueOf(cdelta1.getTotalSize() - cdelta1.getCollectedSize());					  	
					  }
				}	  
			}

			return "";
		}

		protected String getColumnTextInstanceLevel(Object obj, int col){
			
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(obj instanceof TRCClass)	
			{
				return getColumnTextClassLevel(obj, col);
			}
			else
			{
				TRCObject object = (TRCObject)obj;				
				switch(pos)
				{
					case 0://class name
					{
						TRCClass clas = PerftraceUtil.getClass(object);
						if(clas ==object.getProcess().getClassClass())
						    return clas.getName();					
										
						return clas.getName() + "." + object.getId();
					}
					
					case 1:
					{
						return "";
					}
					  
					case 2: //package name
					{
						TRCClass clas = PerftraceUtil.getClass(object);						
					    return PerftraceUtil.getPackageName(clas.getPackage(), _page.getMOFObject()).toString();
					}
	
					case 3://collected
					{
					  if(object instanceof TRCFullTraceObject)
					  {
						if(((TRCFullTraceObject)object).getCollectTime() == 0)
						   return "0";
						 return "1";
					  }
					  return "";
				    } 
					  
					case 4: //total size
					{
						if(isShowPercent())
							return TString.formatAsPercentage(((double)((TRCObject)obj).getSize())/_totalSize);							
						else
							return String.valueOf(((TRCObject)obj).getSize());
				    } 
					  
					case 5: //active size
					{
						int size = 0;
						
						if(object instanceof TRCFullTraceObject)
						{
						
						  if(((TRCFullTraceObject)object).getCollectTime() != 0)
						    size = 0;
						  else
						  	size = object.getSize();
						}
						
						if(isShowPercent())
							return TString.formatAsPercentage(((double)size)/_activeSize);							
						else
							return String.valueOf(size);
				    } 
				}
			}
			return "";
		}
		
		public String getColumnText(Object obj, int col) {
			
			if (getLevel() == MemoryStatisticView.PACKAGE_LEVEL)
				return getColumnTextPackageLevel(obj, col);
			else if (getLevel() == MemoryStatisticView.CLASS_LEVEL)
				return getColumnTextClassLevel(obj, col);
			else if (getLevel() == MemoryStatisticView.INSTANCE_LEVEL)
				return getColumnTextInstanceLevel(obj, col);
			else
				return "";
		}
		
		protected Image getColumnImagePackageLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(info.isDeltaColumn())
			   return null;
			
			int state = 0;
			double d = 0;
			if(obj instanceof TRCPackage)	
			{
				TRCPackage p = (TRCPackage)obj;

				switch(pos)
				{
					case 1: //new item column
					  if(p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES)<0)
					  	  return TracePluginImages.getImage(TracePluginImages.IMG_DELTANEW);
					  else
					  	  return null;
					  
					case 2: //total inst
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES);					
					  state = pdelta1.getTotalInstances();
					  break;
					  					 
					case 3: //live inst	
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES);
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_INSTANCES);				
					  state = pdelta1.getTotalInstances() - pdelta1.getCollectedInstances();
					  break;
					  
					case 4://collected
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_INSTANCES);
					  state = pdelta1.getCollectedInstances();
					  break;
					  
					case 5: //total size
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_SIZE);
					  state = pdelta1.getTotalSize();
					  break;
					  
					case 6: //active size
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_SIZE);
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_SIZE);
					  state = pdelta1.getTotalSize() - pdelta1.getCollectedSize();
					  break;
				}				  				  				 				   				  
			}
			else
			{
				return getColumnImageClassLevel(obj, col);
			}
                  
			if (state < 0) 
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTADOWN);
			if(state > 0)
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTAUP);
      							
			return null;
		}
		
		protected Image getColumnImageClassLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = getClassPosForLevel(info.getColumnData().getInitalPos());			
			
			if(info.isDeltaColumn())
			   return null;
			
			int state = 0;
			double d = 0;
			if(obj instanceof TRCClass)	
			{
				TRCClass clas = (TRCClass)obj;

				switch(pos)
				{
					case 1: //new item column
					  if(clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES)<0)
					  	  return TracePluginImages.getImage(TracePluginImages.IMG_DELTANEW);
					  else
					  	  return null;					
				
					case 2: //package
						return TracePluginImages.getImage(TracePluginImages.IMG_PACKAGE_COL);

					case 3: //total inst
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);					
					  state = cdelta1.getTotalInstances();
					  break;
					  					 
					case 4: //live inst
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
					  state = cdelta1.getTotalInstances() - cdelta1.getCollectedInstances();
					  break;
					  
					case 5://collected
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
					  state = cdelta1.getCollectedInstances();
					  break;
					  
					case 6: //total size
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
					  state = cdelta1.getTotalSize();
					  break;
					  
					case 7: //active size
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_SIZE);
					  state = cdelta1.getTotalSize() - cdelta1.getCollectedSize();
					  break;
				}				  				  				 				   				  
			}
			else
			{
				return null;
			}
                  
			if (state < 0) 
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTADOWN);
			if(state > 0)
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTAUP);
      							
			return null;		
		}
		
		protected Image getColumnImageInstanceLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(info.isDeltaColumn())
			   return null;
			   
			int state = 0;
			double d = 0;
			if(obj instanceof TRCClass)	
			{
				return getColumnImageClassLevel(obj, col);
			}
			else
			{
				switch(pos)
				{
					case 1: //new item column
					  	  return null;		
					case 2: //package
						return TracePluginImages.getImage(TracePluginImages.IMG_PACKAGE_COL);
				}	
			}
			return null;
		}
		
		public Image getColumnImage(Object obj, int col) {
			if (getLevel() == MemoryStatisticView.PACKAGE_LEVEL)
				return getColumnImagePackageLevel(obj, col);
			else if (getLevel() == MemoryStatisticView.CLASS_LEVEL)
				return getColumnImageClassLevel(obj, col);
			else if (getLevel() == MemoryStatisticView.INSTANCE_LEVEL)
				return getColumnImageInstanceLevel(obj, col);			
			else
				return null;
		}
	}

	public class MemoryStatisticSorter extends MultiLevelStatisticSorter{
		
		protected int comparePackageLevel(Viewer viewer, Object e1, Object e2) {
			double d = 0;
			
 			if(e1 instanceof TRCPackage && e2 instanceof TRCPackage)	
			{
				TRCPackage p1 = (TRCPackage)e1;
				TRCPackage p2 = (TRCPackage)e2;
				TRCPackageImpl.TRCPackageSnapshot ps1 = (TRCPackageImpl.TRCPackageSnapshot)p1.retrieveSnapshot();
				TRCPackageImpl.TRCPackageSnapshot ps2 = (TRCPackageImpl.TRCPackageSnapshot)p2.retrieveSnapshot();

				switch(_pos)
				{
					case 0://package name
					  return (int) _sortSequence * (int)p1.getName().compareToIgnoreCase(p2.getName());

					case 1 : // new items
						  if(p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES) < 0
						  		&& !(p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES) < 0))
						  	  return _sortSequence;
						  else if(!(p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES) < 0)
						  		&& p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES) < 0)
						  	 return _sortSequence * (-1);
						  else
						  	  return 0;							  
					  
					case 2: //total inst
					  if(!_info.isDeltaColumn())
					  {	
						  return (int) _sortSequence * (ps1.getTotalInstances() - ps2.getTotalInstances());					  						  					
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES);
						return (int) _sortSequence * (pdelta1.getTotalInstances() - pdelta2.getTotalInstances());					  	
					  }
					  					 
					case 3: //live inst
					  if(!_info.isDeltaColumn())
					  {										
						  return (int) _sortSequence * (ps1.getTotalInstances() - ps1.getCollectedInstances()- p2.getTotalInstances() + p2.getCollectedInstances());
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES);
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_INSTANCES);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__COLLECTED_INSTANCES);
						
						return (int) _sortSequence * (pdelta1.getTotalInstances() - pdelta1.getCollectedInstances()
						                       - pdelta2.getTotalInstances() + pdelta2.getCollectedInstances());					  	
					  }
					  
					case 4://collected
					  if(!_info.isDeltaColumn())
					  {										
						  if(isShowPercent())
						  {
						  	  int inst = ps1.getTotalInstances();
						  	  if(inst == 0) inst = 1;					  	  
						  	  double size = ((double)ps1.getCollectedInstances())/inst;
						  	  
						  	  inst = ps2.getTotalInstances();
						  	  if(inst == 0) inst = 1;					  	  
						  	  size -= ((double)ps2.getCollectedInstances())/inst;
						  	  
							  return (int) _sortSequence * (int)(size*10000);					  	  					  	  
						  }
						  else
						  {
							  return (int) _sortSequence * (ps1.getCollectedInstances() - ps2.getCollectedInstances());
						  }
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_INSTANCES);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__COLLECTED_INSTANCES);
						return (int) _sortSequence * (pdelta1.getCollectedInstances() - pdelta2.getCollectedInstances());					  	
					  }
					  
					case 5: //total size
					  if(!_info.isDeltaColumn())
					  {										
						  return (int) _sortSequence * (ps1.getTotalSize() - ps2.getTotalSize());
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_SIZE);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__TOTAL_SIZE);
						return (int) _sortSequence * (pdelta1.getTotalSize() - pdelta2.getTotalSize());					  	
					  }
					  
					case 6: //active size
					  if(!_info.isDeltaColumn())
					  {										
						  return (int) _sortSequence * (ps1.getTotalSize() - ps1.getCollectedSize()- p2.getTotalSize() + p2.getCollectedSize());
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_SIZE);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__TOTAL_SIZE);
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_SIZE);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__COLLECTED_SIZE);
						
						return (int) _sortSequence * (pdelta1.getTotalSize() - pdelta1.getCollectedSize()
											   - pdelta2.getTotalSize() + pdelta2.getCollectedSize());					  	
					  }
				}
			}
 			else
 			{
 				return compareClassLevel(viewer, e1, e2);
 			}
 			return 0;
		}

		protected int compareClassLevel(Viewer viewer, Object e1, Object e2) {
			
			int pos = getClassPosForLevel(_pos);
			
            double d =0;
            
			if (e1 instanceof TRCClass && e2 instanceof TRCClass) {
					
				TRCClass c1 = (TRCClass) e1;
				TRCClass c2 = (TRCClass)e2;				
				TRCClassImpl.TRCClassSnapshot cs1 = (TRCClassImpl.TRCClassSnapshot)c1.retrieveSnapshot();
				TRCClassImpl.TRCClassSnapshot cs2 = (TRCClassImpl.TRCClassSnapshot)c2.retrieveSnapshot();
				
				switch (pos) {
					case 0 : //class name
						return (int) _sortSequence
							* (int) c1.getName().compareToIgnoreCase(c2.getName());
						
					case 1 : // new items
						  if(c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES) < 0
						  		&& !(c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__TOTAL_INSTANCES) < 0))
						  	  return _sortSequence;
						  else if(!(c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES) < 0)
						  		&& c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__TOTAL_INSTANCES) < 0)
						  	 return _sortSequence * (-1);
						  else
						  	  return 0;							

					case 2 :	//package name
						return (int) _sortSequence
							* (int) c1.getPackage().getName()
								.compareToIgnoreCase(c2.getPackage().getName());

					case 3 : //total inst
					  if(!_info.isDeltaColumn())
					  {																									
						return (int) _sortSequence
							* (cs1.getTotalInstances() - cs2.getTotalInstances());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						return (int) _sortSequence * (cdelta1.getTotalInstances() - cdelta2.getTotalInstances());					  	
					  }

					case 4 : //live inst
					  if(!_info.isDeltaColumn())
					  {																									
						return (int) _sortSequence
							* (cs1.getTotalInstances()
								- cs1.getCollectedInstances()
								- cs2.getTotalInstances()
								+ cs2.getCollectedInstances());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						return (int) _sortSequence * ((cdelta1.getTotalInstances() - cdelta1.getCollectedInstances())
							   - (cdelta2.getTotalInstances() - cdelta2.getCollectedInstances()));					  	
					  }

					case 5 : //collected
					  if(!_info.isDeltaColumn())
					  {																									
						  if(isShowPercent())
						  {
						  	  int inst = cs1.getTotalInstances();
						  	  if(inst == 0) inst = 1;					  	  
						  	  double size = (double)cs1.getCollectedInstances()/inst;
						  	  
						  	  inst = cs2.getTotalInstances();
						  	  if(inst == 0) inst = 1;					  	  
						  	  size -= (double)cs2.getCollectedInstances()/inst;
						  	  
							  return (int) _sortSequence * (int)(size*10000);					  	  					  	  
						  }
						  else
						  {										
							return (int) _sortSequence
								* (cs1.getCollectedInstances() - cs2.getCollectedInstances());
						  }
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						return (int) _sortSequence * (cdelta1.getCollectedInstances() - cdelta2.getCollectedInstances());					  	
					  }

					case 6 : //total size
					  if(!_info.isDeltaColumn())
					  {																									
						return (int) _sortSequence
							* (cs1.getTotalSize() - cs2.getTotalSize());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__TOTAL_SIZE);
						return (int) _sortSequence * (cdelta1.getTotalSize() - cdelta2.getTotalSize());					  	
					  }

					case 7 : //active size
					  if(!_info.isDeltaColumn())
					  {																									
						return (int) _sortSequence
							* (cs1.getTotalSize()
								- cs1.getCollectedSize()
								- cs2.getTotalSize()
								+ cs2.getCollectedSize());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__TOTAL_SIZE);
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_SIZE);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__COLLECTED_SIZE);
						return (int) _sortSequence * ((cdelta1.getTotalSize() - cdelta1.getCollectedSize())
							   - (cdelta2.getTotalSize() - cdelta2.getCollectedSize()));					  	
					  }
				}
			}
			return 0;
		}
		
		protected int compareInstanceLevel(Viewer viewer, Object e1, Object e2)
		{
			double d = 0;
			
 			if(e1 instanceof TRCClass)	
			{
 				return compareClassLevel(viewer, e1, e2);
			}
			else
			{				 
				TRCObject obj1 = (TRCObject) e1;
				TRCObject obj2 = (TRCObject) e2;  
				
				switch(_pos)
				{
					case 0://instance name
					  return (int) _sortSequence * (int)(obj1.getId() - obj2.getId());
					  
					case 1: //new item
						return 0;
					  
					case 2: //package
						return 0;					  
	
					case 3://collected
					
					if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCHeapObject)
					{

						TRCHeapObject object2 = (TRCHeapObject) e2;				
						TRCClass class2 = PerftraceUtil.getClass(object2);
						int collected1 = ((TRCFullTraceObject)obj1).getCollectTime() == 0 ? 0 : 1;
						int collected2 = class2.getCollectedSize() == 0 ? 0 : 1;
						return (int) _sortSequence * (collected1-collected2);			   

					}
					else if(obj1 instanceof TRCHeapObject && obj2 instanceof TRCFullTraceObject)
					{
						TRCHeapObject object1 = (TRCHeapObject) e1;		
						TRCClass class1 = PerftraceUtil.getClass(object1);
						int collected2 = ((TRCFullTraceObject)obj2).getCollectTime() == 0 ? 0 : 1;
						int collected1 = class1.getCollectedSize() == 0 ? 0 : 1;
						return (int) _sortSequence * (collected1-collected2);
					}
					else if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCFullTraceObject)
					{
						int collected1 = ((TRCFullTraceObject)obj1).getCollectTime() == 0 ? 0 : 1;
						int collected2 = ((TRCFullTraceObject)obj2).getCollectTime() == 0 ? 0 : 1;  
						return (int) _sortSequence * (collected1-collected2);

					}
					return 0;  

					case 4: //total size
					  return (int) _sortSequence * (obj1.getSize() - obj2.getSize());
					  
					case 5: //active size
						
					if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCHeapObject)
					{
		
						TRCHeapObject object2 = (TRCHeapObject) e2;				
						TRCClass class2 = PerftraceUtil.getClass(object2);
						int activeSize1 = ((TRCFullTraceObject)obj1).getCollectTime() == 0 ? obj1.getSize() : 0;
						int activeSize2 = class2.getCollectedSize() == 0 ? class2.getSize() : 0;
						return (int) _sortSequence * (activeSize1-activeSize2);			   
			
					}
					else if(obj1 instanceof TRCHeapObject && obj2 instanceof TRCFullTraceObject)
					{
						TRCHeapObject object1 = (TRCHeapObject) e1;		
						TRCClass class1 = PerftraceUtil.getClass(object1);
						int activeSize2 = ((TRCFullTraceObject)obj2).getCollectTime() == 0 ? obj1.getSize() : 0;
						int activeSize1 = class1.getCollectedSize() == 0 ? class1.getSize() : 0;
						return (int) _sortSequence * (activeSize1-activeSize2);
					}
					else if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCFullTraceObject)
					{
						int activeSize1 = ((TRCFullTraceObject)obj1).getCollectTime() == 0 ? obj1.getSize() : 0;
						int activeSize2 = ((TRCFullTraceObject)obj2).getCollectTime() == 0 ? obj2.getSize() : 0;  
						return (int) _sortSequence * (activeSize1-activeSize2);

					}
					return 0;  
				}			  				  				 				   				  
			}
			return 0;
		}
		
		public int compare(Viewer viewer, Object e1, Object e2) {
			
			if (getLevel() == MemoryStatisticView.PACKAGE_LEVEL)
				return comparePackageLevel(viewer, e1, e2);
			else if (getLevel() == MemoryStatisticView.CLASS_LEVEL)
				return compareClassLevel(viewer, e1, e2);
			else if (getLevel() == MemoryStatisticView.INSTANCE_LEVEL)
				return compareInstanceLevel(viewer, e1, e2);
			
			return 0;
		}
	}

	public class MemoryStatisticFilter extends StatisticFilter{
		
		public MemoryStatisticFilter() {
			super();
		}
		
		public boolean select(Viewer viewer, Object parent, Object element) {
			
			boolean flag = true;
			String compareText = "";

			switch (getLevel()) {
				case PACKAGE_LEVEL : //Package level
					if (!(element instanceof TRCPackage))
						return true;
					if (((TRCPackage) element).getName().equals(""))
						compareText =
							TraceUIPlugin.getString("DEFAULT_PACKAGE");
					else
						compareText =
							((TRCPackage) element).getName();
					break;
				case CLASS_LEVEL : //class level
					if (!(element instanceof TRCClass))
						return true;
					compareText = ((TRCClass) element).getName();
					break;
				case INSTANCE_LEVEL : //instance level
					if (!(element instanceof TRCClass))
						return true;
					compareText = ((TRCClass) element).getName();
					break;					
			}

			if (_noPattern)
				return true;

			if (!_caseSensitive) {
				compareText = compareText.toLowerCase();
			}
			if (_exactMatch) {
				return compareText.compareTo(_prefix) == 0;
			}
			if (_prefix != "") {
				flag = compareText.startsWith(_prefix);
			}
			if (flag && _suffix != "") {
				flag = compareText.endsWith(_suffix);
			}
			if (flag) {
				for (int k = 0; k < _textList.size(); k++) {

					String str1 = (String) _textList.get(k);

					int index = compareText.lastIndexOf(str1);
					if (index == -1) {
						flag = false;
						break;
					}

					compareText = compareText.substring(index + str1.length());
				}
			}
			return flag;
		}
	}
	
	public String getDefaultColumnsTemplate()
	{	
		String columns = super.getDefaultColumnsTemplate();
		
		if (columns == null)
		{
			if (getLevel() == INSTANCE_LEVEL)
				columns = getDefaultColumnsTemplateInstanceLevel();
			else
				columns = "";
		}
	
		return columns;
	}		
	
	protected String getDefaultColumnsTemplatePackageLevel()
	{
		return TraceUIPlugin.getString("STR_ST_PACKAGE")	+ ":0:" 
      		+ String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:200,"
   		+ TraceUIPlugin.getString("")			+ ":1:"
		    + String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE | ColumnData.NONRESIZABLE | ColumnData.NOT_VISIBLE_CHOOSE_COLUMNS) + ":14,"		
		+ TraceUIPlugin.getString("STR_ST_INSTANCES")	+ ":2:" 
		    + String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"
		+ TraceUIPlugin.getString("STR_ST_LIVE_INSTANCES")+ ":3:"
		    + String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"
		+ TraceUIPlugin.getString("STR_ST_COLLECTED")	+ ":4:"
		    + String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"			    
		+ TraceUIPlugin.getString("STR_ST_SIZE") 			+ ":5:"
		    + String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"						    
		+ TraceUIPlugin.getString("STR_ST_LIVE_STORAGE") 	+ ":6:"
		    + String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100";
	}

	protected String getDefaultColumnsTemplateClassLevel()
	{
		return TraceUIPlugin.getString("STR_ST_CLASS_NAME")
			+ ":0:"
			+ String.valueOf(
				ColumnData.NONDELETABLE
					| ColumnData.IS_VISIBLE
					| ColumnData.NONMOVABLE)
			+ ":200,"
			+ TraceUIPlugin.getString("")
			+ ":1:"
		    + String.valueOf(ColumnData.NONDELETABLE
		    					| ColumnData.IS_VISIBLE
								| ColumnData.NONMOVABLE
								| ColumnData.NONRESIZABLE
								| ColumnData.NOT_VISIBLE_CHOOSE_COLUMNS)
			+ ":14,"					
			+ TraceUIPlugin.getString("STR_ST_PACKAGE")
			+ ":2:"
			+ String.valueOf(ColumnData.IS_VISIBLE)
			+ ":left:100,"
			+ TraceUIPlugin.getString("STR_ST_INSTANCES")
			+ ":3:"
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)
			+ ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_LIVE_INSTANCES")
			+ ":4:"
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)
			+ ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_COLLECTED")
			+ ":5:"
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)
			+ ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_SIZE")
			+ ":6:"
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)
			+ ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_LIVE_STORAGE")
			+ ":7:"
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA)
			+ ":right:100";			
	}
	
	protected String getDefaultColumnsTemplateInstanceLevel()
	{
		return TraceUIPlugin.getString("STR_ST_CLASS_NAME") 	+ ":0:" 
      		+ String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:200,"
       	+ TraceUIPlugin.getString("")			+ ":1:"
			+ String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE | ColumnData.NONRESIZABLE | ColumnData.NOT_VISIBLE_CHOOSE_COLUMNS) + ":14,"		
		+ TraceUIPlugin.getString("STR_ST_PACKAGE") 		+ ":2:"
   			+ String.valueOf(ColumnData.IS_VISIBLE) + ":left:100,"
		+ TraceUIPlugin.getString("STR_ST_COLLECTED")		+ ":3:"
		    + String.valueOf(ColumnData.IS_VISIBLE) + ":right:100,"			
		+ TraceUIPlugin.getString("STR_ST_SIZE") 			+ ":4:"  
			+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:100,"				
		+ TraceUIPlugin.getString("STR_ST_LIVE_STORAGE") 	+ ":5:"  
			+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:100";	
	}
	
	public boolean isEmptyUpdate()
	{
		return PerftraceUtil.getTotalSize(_page.getMOFObject()) <= 0;
	}
	
	public IContentProvider getContentProvider()
	{
		return new MemoryStatisticContentProvider();
	}
	
	public LabelProvider getTableLabelProvider()
	{
		return new MemoryStatisticLabelProvider(this);
	}
	
	protected StatisticSorter getViewerSorterInstance()
	{
		return new MemoryStatisticSorter();
	}
	
	protected StatisticFilter getFilterInstance()
	{
		return new MemoryStatisticFilter();
	}	
	
	public String getColumnsPreferencesKey()
	{
		setLevel(getLevel());		

		if (getLevel() == MemoryStatisticView.PACKAGE_LEVEL)
			return PREFERENCE_KEY_PREFIX + "MemStatsPack60";
		else if (getLevel() == MemoryStatisticView.CLASS_LEVEL)
			return PREFERENCE_KEY_PREFIX + "MemStatsClass60";
		else if (getLevel() == MemoryStatisticView.CLASS_LEVEL)
			return PREFERENCE_KEY_PREFIX + "MemStatsInstance60";	
		
		return "";
	}
	
	protected String getViewTypeStr()
	{
		return org.eclipse.hyades.trace.views.adapter.internal.TraceConstants.MEMORY_STATS_VIEW;
	}	
	
	/**
	 * @param i:the level of information to display (package/class or method)
	 */
	public void setLevel(int i) {
		_level = i;
	}	
	
	public int getLevel()
	{
		return _level;
	}	
	
	public void updateButtons() {
		super.updateButtons();
			
		((MemoryStatisticPage) getTraceViewerPage()).percentMode().setChecked(isShowPercent());
		if (getLevel() == MemoryStatisticView.INSTANCE_LEVEL)
			((MemoryStatisticPage) getTraceViewerPage()).deltaColumns().setChecked(false);
		else
			((MemoryStatisticPage) getTraceViewerPage()).deltaColumns().setChecked(showingDeltaColumns());		
	}	
	
	protected void showPercentUpdate()
	{
		if(isShowPercent())
		{
			_totalCalls = PerftraceUtil.getTotalCalls(_page.getMOFObject());
			if (_totalCalls==0)_totalCalls=1;
			_totalInst = PerftraceUtil.getTotalInstances(_page.getMOFObject());
			if (_totalInst==0)_totalInst=1;			
			_activeInst = PerftraceUtil.getActiveInstances(_page.getMOFObject());
			if (_activeInst==0)_activeInst=1;
			_totalSize = PerftraceUtil.getTotalSize(_page.getMOFObject());
			if (_totalSize==0)_totalSize=1;
			_activeSize = PerftraceUtil.getActiveSize(_page.getMOFObject());
			if (_activeSize==0)_activeSize=1;
			_maxTime = PerftraceUtil.getMaximumTime(_page.getMOFObject());
			if (_maxTime==0)_maxTime=1;			
		}		
	}	
	
	
	protected void searchInLevel(TableTree table,
			TableTreeItem item,
			TRCPackage pack,
			TRCClass cls,
			TRCMethod meth,
			TRCObject trcObj)
	{
		
		super.searchInLevel(table, item, pack, cls, meth, trcObj);
		
		switch (getLevel()) {
			case INSTANCE_LEVEL :
				if (cls != null)
					searchInClassLevel(
							table,
							item,
							cls,
							null,
							trcObj);
				break;
		}
	}		
}
