/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.sample.report.svg;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Hashtable;
import java.util.Iterator;

import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.testprofile.TPFVerdict;

/**
 * This class contains 2 classes that are used to simplify the graph creation
 * by aggregating the required information in a convinent way.
 */
public class DayHelper
{
	/**
	 * The instances of this class contain the information that will be 
	 * plotted for a day in the graph.
	 */
	public static class GraphDay
	implements Cloneable
	{
		public static final int EXPECTED_ATTEMPT = -109;
		public static final int EXPECTED_ERROR = -108;
		public static final int EXPECTED_PASS = -107;
		public static final int EXPECTED_FAIL = -106;
		public static final int EXPECTED_INCONCLUSIVE = -105;
		
		public static final int ATTEMPT = -104;
		public static final int ERROR = -103;
		public static final int PASS = -102;
		public static final int FAIL = -101;
		public static final int INCONCLUSIVE = -100;
		
		private static boolean hasExpectedAttempt = false;
		private static boolean hasExpectedError = false;
		private static boolean hasExpectedPass = false;
		private static boolean hasExpectedFail = false;
		private static boolean hasExpectedInconclusive = false;
		
		private static boolean hasAttempt = false;
		private static boolean hasError = false;
		private static boolean hasPass = false;
		private static boolean hasFail = false;
		private static boolean hasInconclusive = false;
		
		private String day;

		private int expectedAttempt;
		private int expectedError;
		private int expectedPass;
		private int expectedFail;
		private int expectedInconclusive;

		private int attempt;
		private int error;
		private int pass;
		private int fail;
		private int inconclusive;
		
		/**
		 * Constructor for GraphDay
		 * 
		 * @param String The day in the format "yyyyMMdd"
		 */
		public GraphDay(String day)
		{
			try
			{
				SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
				this.day = format.format(format.parse(day));
				if(!this.day.equals(day))
					throw new ParseException("Invalid date format.", 0);
			}
			catch (ParseException pe)
			{
				throw new RuntimeException(pe.getMessage());
			}			
		}
		
		public static void reset()
		{
			hasExpectedAttempt = false;
			hasExpectedError = false;
			hasExpectedPass = false;
			hasExpectedFail = false;
			hasExpectedInconclusive = false;
			
			hasAttempt = false;
			hasError = false;
			hasPass = false;
			hasFail = false;
			hasInconclusive = false;
		}
		
		/**
		 * Returns the day in the yyyyMMdd format.
		 * @return String
		 */
		public String getDay()
		{
			return day;
		}
		
		public String getDayOfMonth()
		{
			return day.substring("yyyyMM".length());
		}
		
		/**
		 * Returns the day in the DateFormat.SHORT format.
		 * @return String
		 */
		public String getFormattedDay()
		{
			String day = getDay();
			SimpleDateFormat formatter = new SimpleDateFormat("yyyyMMdd");
			try
			{
				Date dateObj = formatter.parse(day);
				
				// Display format: SHORT
				DateFormat df = DateFormat.getDateInstance(DateFormat.SHORT);
				return df.format(dateObj);
			}
			catch(ParseException pe)
			{
			}
			
			return day;
		}

		/**
		 * Returns the statistic value for the indicated flag.
		 * @param int statistic flag
		 * @return int
		 */
		public int getStatistic(int flag)
		{
			switch(flag)
			{
				case EXPECTED_ATTEMPT:
					return expectedAttempt;
				
				case EXPECTED_ERROR:
					return expectedError;
				
				case EXPECTED_PASS:
					return expectedPass;
				
				case EXPECTED_FAIL:
					return expectedFail;
					
				case EXPECTED_INCONCLUSIVE:
					return expectedInconclusive;
				
				case ATTEMPT:
					return attempt;
				
				case ERROR:
					return error;
				
				case PASS:
					return pass;
					
				case FAIL:
					return fail;
					
				case INCONCLUSIVE:
					return inconclusive;
					
				default:
					return -1;
			}
		}

		/**
		 * Returns true if the specied statistic value was set.
		 * @param int statistic flag
		 * @return boolean
		 */
		public static boolean hasStatistic(int flag)
		{
			switch(flag)
			{
				case EXPECTED_ATTEMPT:
					return hasExpectedAttempt;
				
				case EXPECTED_ERROR:
					return hasExpectedError;
				
				case EXPECTED_PASS:
					return hasExpectedPass;
				
				case EXPECTED_FAIL:
					return hasExpectedFail;
					
				case EXPECTED_INCONCLUSIVE:
					return hasExpectedInconclusive;
				
				case ATTEMPT:
					return hasAttempt;
				
				case ERROR:
					return hasError;
					
				case PASS:
					return hasPass;
					
				case FAIL:
					return hasFail;
					
				case INCONCLUSIVE:
					return hasInconclusive;
					
				default:
					return false;
			}
		}
		
		/**
		 * Adds a value to the statistic value for the indicated flag.
		 * @param int statistic flag
		 * 
		 */
		public void addStatistic(int flag, int value)
		{
			switch(flag)
			{
				case EXPECTED_ATTEMPT:
					expectedAttempt+=value;
					hasExpectedAttempt = (expectedAttempt>0);
					break;
				
				case EXPECTED_ERROR:
					expectedError+=value;
					hasExpectedError = (expectedError>0);
					break;
				
				case EXPECTED_PASS:
					expectedPass+=value;
					hasExpectedPass = (expectedPass>0);
					break;
				
				case EXPECTED_FAIL:
					expectedFail+=value;
					hasExpectedFail = (expectedFail>0);
					break;
					
				case EXPECTED_INCONCLUSIVE:
					expectedInconclusive+=value;
					hasExpectedInconclusive = (expectedInconclusive>0);
					break;
				
				case ERROR:
					error+=value;
					hasError = (error>0);
					break;
				
				case ATTEMPT:
					attempt+=value;
					hasAttempt = (attempt>0);
					break;
				
				case PASS:
					pass+=value;
					hasPass = (pass>0);
					break;
					
				case FAIL:
					fail+=value;
					hasFail = (fail>0);
					break;
					
				case INCONCLUSIVE:
					inconclusive+=value;
					hasInconclusive = (inconclusive>0);
					break;
			}
		}

		/**
		 * @return GraphDay the current GraphDay after the day increment
		 */
		public GraphDay incrementDay()
		{
			try
			{
				SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
				format.parse(day);
				format.getCalendar().add(Calendar.DAY_OF_MONTH, 1);
				day = format.format(format.getCalendar().getTime());
			}
			catch(ParseException pe)
			{
				throw new RuntimeException(pe.getMessage());
			}
			
			return this;
		}

		/**
		 * Creates a clone of the GraphDay instance.
		 * 
		 * @see java.lang.Object#clone()
		 */
		public Object clone()
		{
			GraphDay graphDay = new GraphDay(day);

			graphDay.expectedAttempt = expectedAttempt;
			graphDay.expectedError = expectedError;
			graphDay.expectedPass = expectedPass;
			graphDay.expectedFail = expectedFail;
			graphDay.expectedInconclusive = expectedInconclusive;

			graphDay.attempt = attempt;
			graphDay.error = error;
			graphDay.pass = pass;
			graphDay.fail = fail;
			graphDay.inconclusive = inconclusive;
			
			return graphDay;
		}	
	}

	/**
	 * The instances of this class contain the testcase instance status per day.
	 */
	public static class DataDay
	implements Comparable
	{
		private String day;
		private Hashtable executionVerdictByTestcase;
		
		/**
		 * Constructor for DataDay
		 * 
		 * @param String The day in the format "yyyyMMdd"
		 */
		public DataDay(String day)
		{
			try
			{
				SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
				this.day = format.format(format.parse(day));
				if(!this.day.equals(day))
					throw new ParseException("Invalid date format.", 0);
			}
			catch (ParseException pe)
			{
				throw new RuntimeException(pe.getMessage());
			}
				
			executionVerdictByTestcase = new Hashtable();
		}
		
		/**
		 * Returns the day in the format yyyyMMdd
		 * @return String
		 */
		public String getDay()
		{
			return day;
		}
		
		/**
		 * Adds the verdict for a testcase.  Only the last added status will be
		 * considered.
		 * 
		 * @param TestcaseInstance
		 * @param ExecutionStatus
		 */
		public void addVerdict(TPFTestSuite testSuite, TPFVerdict verdict)
		{
			executionVerdictByTestcase.put(testSuite, verdict);
		}
		
		/**
		 * Returns the verdict of a testcase.
		 * 
		 * @param TestcaseInstance
		 * @return ExecutionStatus
		 */
		public TPFVerdict getVerdict(TPFTestSuite testSuite)
		{
			return (TPFVerdict)executionVerdictByTestcase.get(testSuite);
		}
		
		/**
		 * Returns an iterator over the testcase in this day.  The elements are
		 * returned in no particular order.
		 *
		 * @return an iterator over the testcase in this day.
		 */
		public Iterator testcaseIterator()
		{
			return executionVerdictByTestcase.keySet().iterator();
		}
		
		/**
		 * @see java.lang.Object#equals(Object)
		 */
		public boolean equals(Object object)
		{
			if(this == object)
				return true;
				
			if((object != null) && (object instanceof DataDay))
				return (hashCode() == object.hashCode());
				
			return false;
		}
		
		/**
		 * @see java.lang.Object#hashCode()
		 */
		public int hashCode()
		{
			return day.hashCode();
		}
		
		/**
		 * @see java.lang.Comparable#compareTo(Object)
		 */
		public int compareTo(Object object)
		{
			DataDay graphDay = (DataDay)object;
			if(this.equals(graphDay))
				return 0;
				
			if(graphDay == null)
				return 1;
				
			return ((hashCode()-graphDay.hashCode())>0)?1:-1;
		}
	}
}
