package org.eclipse.hyades.logging.events.cbe.impl;

import java.io.InputStream;
import java.security.AccessController;
import java.security.PrivilegedExceptionAction;

import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ContentHandler;
import org.eclipse.hyades.logging.events.cbe.FormattingException;
import org.eclipse.hyades.logging.events.cbe.TemplateContentHandler;
import org.eclipse.hyades.logging.events.cbe.util.EventFormatter;

/*******************************************************************************
 * Copyright (c) 2004 Hyades project. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Common
 * Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: IBM - Initial API and implementation
 ******************************************************************************/

/**
 * Concrete implementation of the <code>AbstractEventFactoryHome</code>
 * abstract event factory home implementation used for populating events with
 * properties from an event configuration template XML file.
 * <p>
 * This implementation contributes an associated <code>ContentHandler</code>
 * implementation specific to this event factory home implementation. The
 * associated <code>ContentHandler</code> is assigned to each event factory
 * created using this event factory home implementation. When events are created
 * from an event factory created using this event factory home implementation,
 * the associated <code>ContentHandler</code> is assigned to the newly created
 * event. The associated <code>ContentHandler</code> may be invoked on a newly
 * created event to populate the event's properties with properties from
 * <code>ContentHandler</code>'s configuration template.
 * <p>
 * The <code>ContentHandler</code> implementation used in this implementation
 * is the <code>TemplateContentHandler</code>. The
 * <code>TemplateContentHandler</code> permits storing event configuration
 * properties that are merged into an event upon event completion. These event
 * configuration properties are resolved by this implementation from an event
 * configuration template XML file.
 * <p>
 * The desired event configuration template XML file must by on the classpath
 * and accessible by this class. The event configuration template XML file is
 * loaded using the following look-up sequence:
 * <p>
 * <ol>
 * <li>The <code>EventXMLFileEventFactoryHomeImpl</code> class' class loader
 * is queried for the event configuration template XML file.</li>
 * <li>The system's class loader is queried for the event configuration
 * template XML file.</li>
 * <li>The current thread's context class loader is queried for the event
 * configuration template XML file.</li>
 * </ol>
 * <p>
 * This event factory home implementation consumes a configuration template XML
 * file in the form of a template event XML file. This template event XML file
 * is a well-formed XML document conforming to a predefined XML schema (e.g.
 * templateEvent.xsd).
 * <p>
 * The naming convention used for the template Event XML file is:
 * 
 * <factory name>.event.xml
 * 
 * where <factory name> is the unique dot-delimited name of the factory. For
 * example, the template event XML file for the com.company.sample factory would
 * be named com.company.sample.event.xml.
 * <p>
 * Configuration templates are located based on the hierarchal factory name,
 * starting from the full factory name (e.g. all name-space segments) and
 * iterating its ancestors to the highest ancestor (e.g. first name-space
 * segment). For example, the com.company.sample factory name would cause the
 * event factory home to attempt to locate the configuration template named
 * com.company.sample. If this configuration template cannot be located, the
 * event factory home would attempt to locate the configuration template named
 * com.company, and so on all the way to the configuration template named com.
 * <p>
 * 
 * @author Paul E Slauenwhite
 * @version 1.0.1
 * @since 1.0.1
 * @see org.eclipse.hyades.logging.events.cbe.impl.AbstractEventFactoryHome
 * @see org.eclipse.hyades.logging.events.cbe.TemplateContentHandler
 */
public class EventXMLFileEventFactoryHomeImpl extends AbstractEventFactoryHome {

    private final static String XML_FILE_EXTENSION = ".event.xml";

    private final static String LINE_SEPARATOR = System.getProperty("line.separator");

    /**
     * @see org.eclipse.hyades.logging.events.cbe.impl.AbstractEventFactoryHome#resolveContentHandler()
     */
    public ContentHandler resolveContentHandler() {
        return (new TemplateContentHandlerImpl());
    }

    /**
     * @see org.eclipse.hyades.logging.events.cbe.impl.AbstractEventFactoryHome#createContentHandler(java.lang.String)
     */
    public ContentHandler createContentHandler(String factoryName) {

        if ((factoryName != null) && (factoryName.trim().length() > 0)) {

            InputStream inputStream = null;

            //Attempt to find the actual location of the
            //configuration template file on the classpath:
            final String fileName = factoryName.concat(XML_FILE_EXTENSION);

            try {

                //Consult the current class' class loader to find the
                //actual local location of the configuration template file
                //using privileged security:
                inputStream = ((InputStream) (AccessController.doPrivileged(new PrivilegedExceptionAction() {

                    public Object run() throws Exception {
                        return (this.getClass().getClassLoader().getResourceAsStream(fileName));
                    }
                })));
            } catch (Throwable t) {
                //Ignore exception since the current class' class loader is
                //null (e.g. current class was loaded by the bootstrap
                //class loader) or insufficient security privileges for
                //accessing the current class' class loader.
            }

            if (inputStream == null) {

                try {

                    //If the current class' class loader cannot find the actual
                    //local location of the configuration template file,
                    //consult the system's class loader to find the actual
                    //local location of the configuration template file
                    //using privileged security:
                    inputStream = ((InputStream) (AccessController.doPrivileged(new PrivilegedExceptionAction() {

                        public Object run() throws Exception {
                            return (ClassLoader.getSystemClassLoader().getResourceAsStream(fileName));
                        }
                    })));
                } catch (Throwable t) {
                    //Ignore exception since insufficient security privileges
                    // for
                    //accessing the system's class loader.
                }

                if (inputStream == null) {

                    try {

                        //If the system's class loader cannot find the actual
                        //local location of the configuration template file,
                        //consult the current thread's class loader to find the
                        //actual local location of the configuration template
                        //file using privileged security:
                        inputStream = ((InputStream) (AccessController.doPrivileged(new PrivilegedExceptionAction() {

                            public Object run() throws Exception {
                                return (Thread.currentThread().getContextClassLoader().getResourceAsStream(fileName));
                            }
                        })));
                    } catch (Throwable t) {
                        //Ignore exception since insufficient security
                        // privileges for
                        //accessing the current thread's class loader.
                    }
                }
            }

            if (inputStream != null) {

                try {

                    CommonBaseEvent[] templateEvents = EventFormatter.eventsFromCanonicalXMLDoc(inputStream);

                    if ((templateEvents != null) && (templateEvents.length > 0) && (templateEvents[0] != null)) {

                        TemplateContentHandler contentHandler = ((TemplateContentHandler) (resolveContentHandler()));

                        contentHandler.setTemplateEvent(templateEvents[0]);

                        return contentHandler;
                    }
                } catch (FormattingException f) {

                    //Ignore since template file content for this factory
                    //cannot be loaded. Delegate to the parent factory.
                }
            }

            int dotIndex = factoryName.lastIndexOf('.');

            if (dotIndex != -1) { return (createContentHandler(factoryName.substring(0, dotIndex))); }
        }

        return null;
    }
}