package org.eclipse.hyades.logging.events.cbe.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.StringReader;
import java.net.URL;
import java.security.AccessController;
import java.security.PrivilegedExceptionAction;
import java.util.List;
import java.util.StringTokenizer;

import org.eclipse.hyades.logging.core.SerializationException;
import org.eclipse.hyades.logging.core.XmlUtility;
import org.eclipse.hyades.logging.events.cbe.AssociatedEvent;
import org.eclipse.hyades.logging.events.cbe.AssociationEngine;
import org.eclipse.hyades.logging.events.cbe.AvailableSituation;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.ConfigureSituation;
import org.eclipse.hyades.logging.events.cbe.ConnectSituation;
import org.eclipse.hyades.logging.events.cbe.ContextDataElement;
import org.eclipse.hyades.logging.events.cbe.CreateSituation;
import org.eclipse.hyades.logging.events.cbe.DependencySituation;
import org.eclipse.hyades.logging.events.cbe.DestroySituation;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.FeatureSituation;
import org.eclipse.hyades.logging.events.cbe.FormattingException;
import org.eclipse.hyades.logging.events.cbe.MsgCatalogToken;
import org.eclipse.hyades.logging.events.cbe.MsgDataElement;
import org.eclipse.hyades.logging.events.cbe.OtherSituation;
import org.eclipse.hyades.logging.events.cbe.ReportSituation;
import org.eclipse.hyades.logging.events.cbe.RequestSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.SituationType;
import org.eclipse.hyades.logging.events.cbe.StartSituation;
import org.eclipse.hyades.logging.events.cbe.StopSituation;
import org.eclipse.hyades.logging.events.cbe.impl.TemplateContentHandlerImpl;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * The <code>EventFormatter</code> class defines a utility object used for
 * converting, serializing and de-serializing <code>CommonBaseEvent</code> and
 * related objects.
 * <p>
 * <b>Notes:</b> 
 * <p>
 * <ul>
 * <li>
 * For converting between Document Object Model (DOM) and XML 
 * documents, see the <code>serialize()</code> and <code>deserialize()</code>
 * APIs in the <code>org.eclipse.hyades.logging.core.XmlUtility</code> class.
 * </li>
 * <li>
 * Some JAXP implementations (e.g. Sun Java v.1.4.x with Crimson) do not support 
 * schema validation.  As such, validation in these run-time environments is not 
 * supported.
 * </li>
 * </ul>
 * <p>
 * 
 * @author Richard K. Duggan
 * @author Paul E. Slauenwhite
 * @author John K. Gerken
 * @version 1.0.1
 * @since 1.0
 * @see org.eclipse.hyades.logging.events.cbe.CommonBaseEvent
 * @see org.eclipse.hyades.logging.events.cbe.AssociationEngine
 * @see org.eclipse.hyades.logging.events.cbe.util.SAXEventHandler
 * @see org.eclipse.hyades.logging.core.XmlUtility
 */
public final class EventFormatter {

    private final static String COMMON_BASE_EVENT_SCHEMA_NAME = "commonbaseevent1_0_1.xsd";

    private final static String COMMON_BASE_EVENT_SCHEMA_LOCATION_URL = resolveCommonBaseEventSchemaLocation();

    private static final String LINE_SEPARATOR = System.getProperty("line.separator");

    /**
     * Converts an <code>AssociationEngine</code> object into a formatted
     * serialized XML document.
     * <p>
     * 
     * @param associationEngine
     *            The <code>AssociationEngine</code> to serialize.
     * @return An XML document String representing the parameter
     *         <code>AssociationEngine</code> object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLDocString(AssociationEngine associationEngine) {
        return (toCanonicalXMLDocString(associationEngine, true));
    }

    /**
     * Converts an array of <code>AssociationEngine</code> objects into a
     * formatted serialized XML document.
     * <p>
     * 
     * @param associationEngines
     *            The array of <code>AssociationEngine</code> objects to
     *            serialize.
     * @return An XML document String representing the parameter array of
     *         <code>AssociationEngine</code> objects.
     * @since 1.0.1
     */
    public static synchronized String toCanonicalXMLDocString(AssociationEngine[] associationEngines) {
        return (toCanonicalXMLDocString(associationEngines, true));
    }

    /**
     * Converts an <code>AssociationEngine</code> object into a serialized XML
     * document. Formatting is based on the format parameter.
     * <p>
     * 
     * @param associationEngine
     *            The <code>AssociationEngine</code> to serialize.
     * @param format
     *            If the serialized XML document is formatted with
     *            platform-dependent new line(s) and tab(s).
     * @return An XML document String representing the parameter
     *         <code>AssociationEngine</code> object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLDocString(AssociationEngine associationEngine, boolean format) {
        return (toCanonicalXMLDocString(new AssociationEngine[] { associationEngine}, format));
    }

    /**
     * Converts an array of <code>AssociationEngine</code> objects into a
     * serialized XML document. Formatting is based on the format parameter.
     * <p>
     * 
     * @param associationEngines
     *            The array of <code>AssociationEngine</code> objects to
     *            serialize.
     * @param format
     *            If the serialized XML document is formatted with
     *            platform-dependent new line(s) and tab(s).
     * @return An XML document String representing the parameter array of
     *         <code>AssociationEngine</code> objects.
     * @since 1.0.1
     */
    public static synchronized String toCanonicalXMLDocString(AssociationEngine[] associationEngines, boolean format) {

        StringBuffer buffer = new StringBuffer(256);

        buffer.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
        buffer.append("<CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd\">");

        if (format) {
            buffer.append(LINE_SEPARATOR);
            buffer.append("\t");
        }

        if (associationEngines != null) {

            for (int index = 0; index < associationEngines.length; index++) {

                buffer.append(toCanonicalXMLString(associationEngines[index]));

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                }
            }
        }

        buffer.append("</CommonBaseEvents>");

        return (buffer.toString());
    }

    /**
     * Converts an <code>AssociationEngine</code> object into a serialized XML
     * fragment.
     * <p>
     * 
     * @param associationEngine
     *            The <code>AssociationEngine</code> to serialize.
     * @return An XML fragment String representing the parameter
     *         <code>AssociationEngine</code> object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLString(AssociationEngine associationEngine) {

        StringBuffer buffer = new StringBuffer(256);

        buffer.append("<AssociationEngine");

        String attributeValue = associationEngine.getId();

        //For performance reasons, do not normalize attribute value since of
        // type xsd:ID which is guaranteed to not contain mark-up characters:
        if (attributeValue != null) {
            buffer.append(" id=\"");
            buffer.append(attributeValue);
            buffer.append("\"");
        }

        attributeValue = associationEngine.getName();

        //For performance reasons, do not normalize attribute value since of
        // type xsd:Name which is guaranteed to not contain mark-up characters:
        if (attributeValue != null) {
            buffer.append(" name=\"");
            buffer.append(attributeValue);
            buffer.append("\"");
        }

        attributeValue = associationEngine.getType();

        //For performance reasons, do not normalize attribute value since of
        // type xsd:Name which is guaranteed to not contain mark-up characters:
        if (attributeValue != null) {
            buffer.append(" type=\"");
            buffer.append(attributeValue);
            buffer.append("\"");
        }

        buffer.append("/>");

        return (buffer.toString());
    }

    /**
     * Internalizes a passed XML document String into a passed
     * <code>AssociationEngine</code> object.
     * <p>
     * 
     * @param associationEngine
     *            The <code>AssociationEngine</code> object to populate.
     * @param xmlDocumentString
     *            The XML document containing the information used to populate
     *            the <code>AssociationEngine</code> object.
     * @since 1.0
     * @deprecated Use {@link #associationEnginesFromCanonicalXMLDoc(String)}.
     */
    public static synchronized void fromCanonicalXMLDocString(AssociationEngine associationEngine, String xmlDocumentString) {

        try {

            AssociationEngine[] engines = associationEnginesFromCanonicalXMLDoc(xmlDocumentString);

            if ((associationEngine != null) && (engines != null) && (engines[0] != null)) {

                associationEngine.setId(engines[0].getId());
                associationEngine.setName(engines[0].getName());
                associationEngine.setType(engines[0].getType());
            } else {
                associationEngine = null;
            }
        } catch (Exception e) {
            //Ignore for backwards compatibility.
        }
    }

    /**
     * Converts an XML document to an array of <code>AssociationEngine</code>
     * objects without validation.
     * <p>
     * This API will return <code>null</code> if no <code>AssociationEngine</code> fragments 
     * exist in the XML document.
     * <p>
     * 
     * @param xmlDocumentString
     *            The XML document to populate the
     *            <code>AssociationEngine</code>s array.
     * @return The <code>AssociationEngine</code>s array to be populated by
     *         the XML document, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized AssociationEngine[] associationEnginesFromCanonicalXMLDoc(String xmlDocumentString) throws FormattingException {
        return (associationEnginesFromCanonicalXMLDoc(xmlDocumentString,false));
    }

    /**
     * Potentially validates and c onverts an XML document to an array of <code>AssociationEngine</code>
     * objects.
     * <p>
     * This API will return <code>null</code> if no <code>AssociationEngine</code> fragments 
     * exist in the XML document.
     * <p>
     * 
     * @param xmlDocumentString
     *            The XML document to populate the
     *            <code>AssociationEngine</code>s array.
     * @param validate
     *            If the XML document is validated using the Common Base Event v1.0.1 schema.
     * @return The <code>AssociationEngine</code>s array to be populated by
     *         the XML document, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized AssociationEngine[] associationEnginesFromCanonicalXMLDoc(String xmlDocumentString,boolean validate) throws FormattingException {
        return (associationEnginesFromCanonicalXMLDoc(new InputSource(new StringReader(xmlDocumentString)),validate));
    }

    /**
     * Converts an XML document from an <code>InputSource</code> to an array
     * of <code>AssociationEngine</code> objects without validation.
     * <p>
     * This API will return <code>null</code> if no <code>AssociationEngine</code> fragments 
     * exist in the XML document.
     * <p>
     * 
     * @param InputSource
     *            The <code>InputSource</code> to populate the
     *            <code>AssociationEngine</code>s array.
     * @return The <code>AssociationEngine</code>s array to be populated by
     *         the XML document, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized AssociationEngine[] associationEnginesFromCanonicalXMLDoc(InputSource inputSource) throws FormattingException {
        return (associationEnginesFromCanonicalXMLDoc(inputSource,false));
    }
    
    /**
     * Potentially validates and c onverts an XML document from an <code>InputSource</code> to an array
     * of <code>AssociationEngine</code> objects.
     * <p>
     * This API will return <code>null</code> if no <code>AssociationEngine</code> fragments 
     * exist in the XML document.
     * <p>
     * 
     * @param InputSource
     *            The <code>InputSource</code> to populate the
     *            <code>AssociationEngine</code>s array.
     * @param validate
     *            If the XML document is validated using the Common Base Event v1.0.1 schema.
     * @return The <code>AssociationEngine</code>s array to be populated by
     *         the XML document, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized AssociationEngine[] associationEnginesFromCanonicalXMLDoc(InputSource inputSource, boolean validate) throws FormattingException {
        return (parse(inputSource,validate).getAssociationEngines());
    }

    /**
     * Converts an XML document from an <code>InputStream</code> to an array
     * of <code>AssociationEngine</code> objects without validation.
     * <p>
     * This API will return <code>null</code> if no <code>AssociationEngine</code> fragments 
     * exist in the XML document.
     * <p>
     * 
     * @param InputSource
     *            The <code>InputStream</code> to populate the
     *            <code>AssociationEngine</code>s array.
     * @return The <code>AssociationEngine</code>s array to be populated by
     *         the XML document, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized AssociationEngine[] associationEnginesFromCanonicalXMLDoc(InputStream inputStream) throws FormattingException {
        return (associationEnginesFromCanonicalXMLDoc(inputStream,false));
    }


    /**
     * Potentially validates and converts an XML document from an <code>InputStream</code> to an array
     * of <code>AssociationEngine</code> objects.
     * <p>
     * This API will return <code>null</code> if no <code>AssociationEngine</code> fragments 
     * exist in the XML document.
     * <p>
     * 
     * @param InputSource
     *            The <code>InputStream</code> to populate the
     *            <code>AssociationEngine</code>s array.
     * @param validate
     *            If the XML document is validated using the Common Base Event v1.0.1 schema.
     * @return The <code>AssociationEngine</code>s array to be populated by
     *         the XML document, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized AssociationEngine[] associationEnginesFromCanonicalXMLDoc(InputStream inputStream,boolean validate) throws FormattingException {
        return (associationEnginesFromCanonicalXMLDoc(new InputSource(inputStream),validate));
    }

    
    /**
     * Converts a Document Object Model (DOM) to an array of <code>AssociationEngine</code>>
     * objects without validation.
     * <p>
     * This API will return <code>null</code> if no <code>AssociationEngine</code> elements 
     * exist in the Document Object Model (DOM).
     * <p>
     * 
     * @param document
     *            The Document Object Model (DOM) to populate the <code>AssociationEngine</code>s
     *            array.
     * @return The <code>AssociationEngine</code>s array to be populated by the
     *         Document Object Model (DOM), or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized AssociationEngine[] associationEnginesFromCanonicalXMLDoc(Document document) throws FormattingException {
        return (associationEnginesFromCanonicalXMLDoc(document,false));
    }

    /**
     * Potentially validates and converts a Document Object Model (DOM) to an array of <code>AssociationEngine</code>
     * objects.
     * <p>
     * This API will return <code>null</code> if no <code>AssociationEngine</code> elements 
     * exist in the Document Object Model (DOM).
     * <p>
     * 
     * @param document
     *            The Document Object Model (DOM) to populate the <code>AssociationEngine</code>s
     *            array.
     * @param validate
     *            If the Document Object Model (DOM) is validated using the Common Base Event v1.0.1 schema.
     * @return The <code>AssociationEngine</code>s array to be populated by the
     *         Document Object Model (DOM), or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized AssociationEngine[] associationEnginesFromCanonicalXMLDoc(Document document,boolean validate) throws FormattingException {

        try {
            return (associationEnginesFromCanonicalXMLDoc(XmlUtility.serialize(document),validate));            
        } 
        catch (SerializationException s) {
            throw new FormattingException(s);
        }
    }

    /**
     * Converts an XML document from an <code>File</code> to an array of
     * <code>AssociationEngine</code> objects without validation.
     * <p>
     * This API will return <code>null</code> if no <code>AssociationEngine</code> fragments 
     * exist in the XML document.
     * <p>
     * 
     * @param InputSource
     *            The <code>File</code> to populate the
     *            <code>AssociationEngine</code>s array.
     * @return The <code>AssociationEngine</code>s array to be populated by
     *         the XML document, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized AssociationEngine[] associationEnginesFromCanonicalXMLDoc(File file) throws FormattingException {
        return (associationEnginesFromCanonicalXMLDoc(file,false));
    }
    
    /**
     * Potentially validates and converts an XML document from an <code>File</code> to an array of
     * <code>AssociationEngine</code> objects.
     * <p>
     * This API will return <code>null</code> if no <code>AssociationEngine</code> fragments 
     * exist in the XML document.
     * <p>
     * 
     * @param InputSource
     *            The <code>File</code> to populate the
     *            <code>AssociationEngine</code>s array.
     * @param validate
     *            If the XML document is validated using the Common Base Event v1.0.1 schema.
     * @return The <code>AssociationEngine</code>s array to be populated by
     *         the XML document, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized AssociationEngine[] associationEnginesFromCanonicalXMLDoc(File file,boolean validate) throws FormattingException {

        try {
            return (associationEnginesFromCanonicalXMLDoc(new InputSource(new FileInputStream(file)),validate));
        } catch (FileNotFoundException f) {
            throw new FormattingException(f);
        }
    }

    /**
     * Internalizes a passed XML fragment String into a passed
     * <code>AssociationEngine</code> object.
     * <p>
     * 
     * @param associationEngine
     *            The <code>AssociationEngine</code> object to populate.
     * @param xmlFragmentString
     *            The XML fragment containing the information used to populate
     *            the <code>AssociationEngine</code> object.
     * @since 1.0
     * @deprecated Use {@link #associationEngineFromCanonicalXML(String)}.
     */
    public static synchronized void fromCanonicalXMLString(AssociationEngine associationEngine, String xmlFragmentString) {

        try {

            AssociationEngine engine = associationEngineFromCanonicalXML(xmlFragmentString);

            if ((associationEngine != null) && (engine != null)) {

                associationEngine.setId(engine.getId());
                associationEngine.setName(engine.getName());
                associationEngine.setType(engine.getType());
            } else {
                associationEngine = null;
            }
        } catch (Exception e) {
            //Ignore for backwards compatibility.
        }
    }

    /**
     * Converts an XML fragment to an <code>AssociationEngine</code> object without validation.
     * <p>
     * This API will return <code>null</code> if no <code>AssociationEngine</code> fragment 
     * exists in the XML fragment.
     * <p>
     * 
     * @param xmlFragmentString
     *            The XML fragment to populate the
     *            <code>AssociationEngine</code> object.
     * @return The <code>AssociationEngine</code> to be populated by the XML
     *         fragment, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0
     */
    public static synchronized AssociationEngine associationEngineFromCanonicalXML(String xmlFragmentString) throws FormattingException {
        return (associationEngineFromCanonicalXML(xmlFragmentString,false));
    }
    
    /**
     * Potentially validates and converts an XML fragment to an <code>AssociationEngine</code> object.
     * <p>
     * This API will return <code>null</code> if no <code>AssociationEngine</code> fragment 
     * exists in the XML fragment.
     * <p>
     * 
     * @param xmlFragmentString
     *            The XML fragment to populate the
     *            <code>AssociationEngine</code> object.
     * @param validate
     *            If the XML fragment is validated using the Common Base Event v1.0.1 schema.
     * @return The <code>AssociationEngine</code> to be populated by the XML
     *         fragment, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0
     */
    public static synchronized AssociationEngine associationEngineFromCanonicalXML(String xmlFragmentString, boolean validation) throws FormattingException {

        StringBuffer buffer = new StringBuffer(1024);

        buffer.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
        buffer.append("<CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd\">");
        buffer.append(xmlFragmentString);
        buffer.append("</CommonBaseEvents>");

        AssociationEngine[] engines = associationEnginesFromCanonicalXMLDoc(buffer.toString(),validation);

        if (engines != null) { 
            return (engines[0]); 
        }

        return null;
    }

    /**
     * Converts an <code>CommonBaseEvent</code> object into a formatted
     * serialized XML document.
     * <p>
     * 
     * @param commonBaseEvent
     *            The <code>CommonBaseEvent</code> to serialize.
     * @return An XML document String representing the parameter
     *         <code>CommonBaseEvent</code> object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLDocString(CommonBaseEvent commonBaseEvent) {
        return (toCanonicalXMLDocString(commonBaseEvent, true));
    }

    /**
     * Converts an array of <code>CommonBaseEvent</code> objects into a
     * formatted serialized XML document.
     * <p>
     * 
     * @param commonBaseEvents
     *            The array of <code>CommonBaseEvent</code> objects to
     *            serialize.
     * @return An XML document String representing the parameter array of
     *         <code>CommonBaseEvent</code> objects.
     * @since 1.0.1
     */
    public static synchronized String toCanonicalXMLDocString(CommonBaseEvent[] commonBaseEvents) {
        return (toCanonicalXMLDocString(commonBaseEvents, true));
    }

    /**
     * Converts an <code>CommonBaseEvent</code> object into a serialized XML
     * document. Formatting is based on the format parameter.
     * <p>
     * 
     * @param commonBaseEvent
     *            The <code>CommonBaseEvent</code> object to serialize.
     * @param format
     *            If the serialized XML document is formatted with
     *            platform-dependent new line(s) and tab(s).
     * @return An XML document String representing the parameter
     *         <code>CommonBaseEvent</code> object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLDocString(CommonBaseEvent commonBaseEvent, boolean format) {
        return (toCanonicalXMLDocString(new CommonBaseEvent[] { commonBaseEvent}, format));
    }

    /**
     * Converts an array of <code>CommonBaseEvent</code> objects into a
     * serialized XML document. Formatting is based on the format parameter.
     * <p>
     * 
     * @param commonBaseEvents
     *            The array of <code>CommonBaseEvent</code> objects to
     *            serialize.
     * @param format
     *            If the serialized XML document is formatted with
     *            platform-dependent new line(s) and tab(s).
     * @return An XML document String representing the parameter array of
     *         <code>CommonBaseEvent</code> objects.
     * @since 1.0.1
     */
    public static synchronized String toCanonicalXMLDocString(CommonBaseEvent[] commonBaseEvents, boolean format) {

        //Create a buffer containing the top level element and namespace info:
        StringBuffer buffer = new StringBuffer(1024);

        buffer.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
        buffer.append("<CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd\">");

        if (format) {
            buffer.append(LINE_SEPARATOR);
        }

        if (commonBaseEvents != null) {

            for (int index = 0; index < commonBaseEvents.length; index++) {

                List associatedEvents = commonBaseEvents[index].getAssociatedEvents();

                //If associatedEvents exist, get the serialized association
                //engine:
                if (associatedEvents.size() > 0) {

                    AssociationEngine associationEngine = null;

                    for (int counter = 0; counter < associatedEvents.size(); counter++) {

                        associationEngine = ((AssociatedEvent) (associatedEvents.get(counter))).getAssociationEngineInfo();

                        if (associationEngine != null) {

                            if (format) {
                                buffer.append("\t");
                            }

                            buffer.append(toCanonicalXMLString(associationEngine));

                            if (format) {
                                buffer.append(LINE_SEPARATOR);
                            }
                        }
                    }
                }

                if (format) {

                    //Add the CommonBaseEvent element fragment with an extra
                    // tab after every platform dependent new line character to
                    // indent all CommonBaseEvents child elements:
                    StringTokenizer tokens = new StringTokenizer(toCanonicalXMLString(commonBaseEvents[index], format).trim(), LINE_SEPARATOR);

                    while (tokens.hasMoreTokens()) {

                        buffer.append("\t");
                        buffer.append(tokens.nextToken());
                        buffer.append(LINE_SEPARATOR);
                    }
                } else {

                    //Add the CommonBaseEvent element fragment:
                    buffer.append(toCanonicalXMLString(commonBaseEvents[index], format));
                }
            }
        }

        // Close and return the top level element:
        buffer.append("</CommonBaseEvents>");

        return buffer.toString();
    }

    /**
     * Converts an <code>CommonBaseEvent</code> object into a formatted
     * serialized XML fragment.
     * <p>
     * 
     * @param commonBaseEvent
     *            The <code>CommonBaseEvent</code> to serialize.
     * @return An XML fragment String representing the parameter
     *         <code>CommonBaseEvent</code> object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLString(CommonBaseEvent commonBaseEvent) {
        return (toCanonicalXMLString(commonBaseEvent, true));
    }

    /**
     * Converts an <code>CommonBaseEvent</code> object into a serialized XML
     * fragment. Formatting is based on the format parameter.
     * <p>
     * 
     * @param commonBaseEvent
     *            The <code>CommonBaseEvent</code> to serialize.
     * @param format
     *            If the serialized XML fragment is formatted with
     *            platform-dependent new line(s) and tab(s).
     * @return An XML fragment String representing the parameter
     *         <code>CommonBaseEvent</code> object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLString(CommonBaseEvent commonBaseEvent, boolean format) {

        //Start with a 1K buffer to load the XML String:
        StringBuffer buffer = new StringBuffer(1024);

        buffer.append("<CommonBaseEvent");

        //For performance reasons, do not normalize attribute value since of
        // type xsd:dateTime which is guaranteed to not contain mark-up
        // characters:
        if (commonBaseEvent.isSetCreationTime()) {
            buffer.append(" creationTime=\"");
            buffer.append(commonBaseEvent.getCreationTime());
            buffer.append("\"");
        }

        String attributeValue = commonBaseEvent.getExtensionName();

        //For performance reasons, do not normalize attribute value since of
        // type xsd:Name which is guaranteed to not contain mark-up characters:
        if (attributeValue != null) {
            buffer.append(" extensionName=\"");
            buffer.append(attributeValue);
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getGlobalInstanceId();

        //For performance reasons, do not normalize attribute value since of
        // type xsd:ID which is guaranteed to not contain mark-up characters:
        if (attributeValue != null) {
            buffer.append(" globalInstanceId=\"");
            buffer.append(attributeValue);
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getLocalInstanceId();

        if (attributeValue != null) {
            buffer.append(" localInstanceId=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getMsg();

        if (attributeValue != null) {
            buffer.append(" msg=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        //For performance reasons, do not normalize attribute value since of
        // type xsd:long which is guaranteed to not contain mark-up characters:
        if (commonBaseEvent.isSetElapsedTime()) {
            buffer.append(" elapsedTime=\"");
            buffer.append(commonBaseEvent.getElapsedTime());
            buffer.append("\"");
        }

        //For performance reasons, do not normalize attribute value since of
        // type xsd:short which is guaranteed to not contain mark-up characters:
        if (commonBaseEvent.isSetPriority()) {
            buffer.append(" priority=\"");
            buffer.append(commonBaseEvent.getPriority());
            buffer.append("\"");
        }

        //For performance reasons, do not normalize attribute value since of
        // type xsd:short which is guaranteed to not contain mark-up characters:
        if (commonBaseEvent.isSetRepeatCount()) {
            buffer.append(" repeatCount=\"");
            buffer.append(commonBaseEvent.getRepeatCount());
            buffer.append("\"");
        }

        //For performance reasons, do not normalize attribute value since of
        // type xsd:long which is guaranteed to not contain mark-up characters:
        if (commonBaseEvent.isSetSequenceNumber()) {
            buffer.append(" sequenceNumber=\"");
            buffer.append(commonBaseEvent.getSequenceNumber());
            buffer.append("\"");
        }

        //For performance reasons, do not normalize attribute value since of
        // type xsd:short which is guaranteed to not contain mark-up characters:
        if (commonBaseEvent.isSetSeverity()) {
            buffer.append(" severity=\"");
            buffer.append(commonBaseEvent.getSeverity());
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getVersion();

        if (attributeValue != null) {
            buffer.append(" version=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        buffer.append(">");

        List contextDataElements = commonBaseEvent.getContextDataElements();
        ContextDataElement contextDataElement = null;

        for (int i = 0; i < contextDataElements.size(); i++) {

            contextDataElement = ((ContextDataElement) (contextDataElements.get(i)));

            if (contextDataElement != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t");
                }

                buffer.append("<contextDataElements");

                attributeValue = contextDataElement.getName();

                //For performance reasons, do not normalize attribute value
                // since of type xsd:Name which is guaranteed to not contain
                // mark-up characters:
                if (attributeValue != null) {
                    buffer.append(" name=\"");
                    buffer.append(attributeValue);
                    buffer.append("\"");
                }

                attributeValue = contextDataElement.getType();

                //For performance reasons, do not normalize attribute value
                // since of type xsd:Name which is guaranteed to not contain
                // mark-up characters:
                if (attributeValue != null) {
                    buffer.append(" type=\"");
                    buffer.append(attributeValue);
                    buffer.append("\"");
                }

                buffer.append(">");

                attributeValue = contextDataElement.getContextId();

                if (attributeValue != null) {

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t\t");
                    }

                    //For performance reasons, do not normalize element value
                    // since of type xsd:NMTOKEN which is guaranteed to not
                    // contain mark-up characters:
                    buffer.append("<contextId>");
                    buffer.append(attributeValue);
                    buffer.append("</contextId>");
                } else {

                    attributeValue = contextDataElement.getContextValue();

                    if (attributeValue != null) {

                        if (format) {
                            buffer.append(LINE_SEPARATOR);
                            buffer.append("\t\t");
                        }

                        buffer.append("<contextValue>");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("</contextValue>");
                    }
                }

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t");
                }

                buffer.append("</contextDataElements>");
            }
        }

        List extendedDataElements = commonBaseEvent.getExtendedDataElements();

        for (int i = 0; i < extendedDataElements.size(); i++) {
            buffer.append(getExtendedDataElementXML(((ExtendedDataElement) (extendedDataElements.get(i))), "extendedDataElements", format, (format ? 1 : 0)));
        }

        List associatedEvents = commonBaseEvent.getAssociatedEvents();
        AssociatedEvent associatedEvent = null;

        for (int i = 0; i < associatedEvents.size(); i++) {

            associatedEvent = ((AssociatedEvent) (associatedEvents.get(i)));

            if (associatedEvent != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t");
                }

                buffer.append("<associatedEvents");

                String resolvedEvents = associatedEvent.getResolvedEvents();

                //For performance reasons, do not normalize element value since
                // of type xsd:NMTOKENS which is guaranteed to not contain
                // mark-up characters:
                if (resolvedEvents != null) {

                    buffer.append(" resolvedEvents=\"");
                    buffer.append(resolvedEvents);
                    buffer.append("\"");
                }

                buffer.append(">");

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                AssociationEngine associationEngineInfo = associatedEvent.getAssociationEngineInfo();

                if (associationEngineInfo != null) {

                    buffer.append("<associationEngineInfo");

                    //Add the 'Info' to '<AssociationEngine'.
                    buffer.append(toCanonicalXMLString(associationEngineInfo).trim().substring(18)); //18
                                                                                                     // is
                                                                                                     // the
                                                                                                     // length
                                                                                                     // of
                                                                                                     // '<AssociationEngine'.
                } else {

                    attributeValue = associatedEvent.getAssociationEngine();

                    //For performance reasons, do not normalize element value
                    // since of type xsd:NMTOKEN which is guaranteed to not
                    // contain mark-up characters:
                    if (attributeValue != null) {

                        buffer.append("<associationEngine>");
                        buffer.append(attributeValue);
                        buffer.append("</associationEngine>");
                    }
                }

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t");
                }

                buffer.append("</associatedEvents>");
            }
        }

        ComponentIdentification componentId = commonBaseEvent.getReporterComponentId();

        if (componentId != null) {

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("<reporterComponentId");

            attributeValue = componentId.getApplication();

            if (attributeValue != null) {
                buffer.append(" application=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponent();

            if (attributeValue != null) {
                buffer.append(" component=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponentIdType();

            if (attributeValue != null) {
                buffer.append(" componentIdType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getExecutionEnvironment();

            if (attributeValue != null) {
                buffer.append(" executionEnvironment=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getInstanceId();

            if (attributeValue != null) {
                buffer.append(" instanceId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocation();

            if (attributeValue != null) {
                buffer.append(" location=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocationType();

            //For performance reasons, do not normalize attribute value since
            // of type xsd:Name which is guaranteed to not contain mark-up
            // characters:
            if (attributeValue != null) {
                buffer.append(" locationType=\"");
                buffer.append(attributeValue);
                buffer.append("\"");
            }

            attributeValue = componentId.getProcessId();

            if (attributeValue != null) {
                buffer.append(" processId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getSubComponent();

            if (attributeValue != null) {
                buffer.append(" subComponent=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getThreadId();

            if (attributeValue != null) {
                buffer.append(" threadId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponentType();

            if (attributeValue != null) {
                buffer.append(" componentType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append("/>");
        }

        componentId = commonBaseEvent.getSourceComponentId();

        if (componentId != null) {

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("<sourceComponentId");

            attributeValue = componentId.getApplication();

            if (attributeValue != null) {
                buffer.append(" application=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponent();

            if (attributeValue != null) {
                buffer.append(" component=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponentIdType();

            if (attributeValue != null) {
                buffer.append(" componentIdType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getExecutionEnvironment();

            if (attributeValue != null) {
                buffer.append(" executionEnvironment=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getInstanceId();

            if (attributeValue != null) {
                buffer.append(" instanceId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocation();

            if (attributeValue != null) {
                buffer.append(" location=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocationType();

            //For performance reasons, do not normalize attribute value since
            // of type xsd:Name which is guaranteed to not contain mark-up
            // characters:
            if (attributeValue != null) {
                buffer.append(" locationType=\"");
                buffer.append(attributeValue);
                buffer.append("\"");
            }

            attributeValue = componentId.getProcessId();

            if (attributeValue != null) {
                buffer.append(" processId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getSubComponent();

            if (attributeValue != null) {
                buffer.append(" subComponent=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getThreadId();

            if (attributeValue != null) {
                buffer.append(" threadId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponentType();

            if (attributeValue != null) {
                buffer.append(" componentType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append("/>");
        }

        MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();

        if (msgDataElement != null) {

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("<msgDataElement");

            attributeValue = msgDataElement.getMsgLocale();

            //For performance reasons, do not normalize attribute value since
            // of type xsd:language which is guaranteed to not contain mark-up
            // characters:
            if (attributeValue != null) {
                buffer.append(" msgLocale=\"");
                buffer.append(attributeValue);
                buffer.append("\"");
            }

            buffer.append(">");

            List tokens = msgDataElement.getMsgCatalogTokens();
            MsgCatalogToken msgCatalogToken = null;

            for (int i = 0; i < tokens.size(); i++) {

                msgCatalogToken = ((MsgCatalogToken) (tokens.get(i)));

                if ((msgCatalogToken != null) && (msgCatalogToken.getValue() != null)) {

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t\t");
                    }

                    buffer.append("<msgCatalogTokens value=\"");
                    buffer.append(XmlUtility.normalize(msgCatalogToken.getValue()));
                    buffer.append("\"/>");
                }
            }

            attributeValue = msgDataElement.getMsgId();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgId>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgId>");
            }

            attributeValue = msgDataElement.getMsgIdType();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                //For performance reasons, do not normalize element value since
                // of type xsd:Name which is guaranteed to not contain mark-up
                // characters:
                buffer.append("<msgIdType>");
                buffer.append(attributeValue);
                buffer.append("</msgIdType>");
            }

            attributeValue = msgDataElement.getMsgCatalogId();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgCatalogId>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgCatalogId>");
            }

            attributeValue = msgDataElement.getMsgCatalogType();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                //For performance reasons, do not normalize element value since
                // of type xsd:Name which is guaranteed to not contain mark-up
                // characters:
                buffer.append("<msgCatalogType>");
                buffer.append(attributeValue);
                buffer.append("</msgCatalogType>");
            }

            attributeValue = msgDataElement.getMsgCatalog();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgCatalog>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgCatalog>");
            }

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("</msgDataElement>");
        }

        Situation situation = commonBaseEvent.getSituation();

        if (situation != null) {

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("<situation");

            attributeValue = situation.getCategoryName();

            if (attributeValue != null) {
                buffer.append(" categoryName=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append(">");

            SituationType situationType = situation.getSituationType();

            if (situationType != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<situationType xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"");

                attributeValue = situationType.getReasoningScope();

                if (situationType instanceof StartSituation) {

                    buffer.append(" xsi:type=\"StartSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((StartSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((StartSituation) (situationType)).getSituationQualifier();

                    if (attributeValue != null) {

                        buffer.append(" situationQualifier=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                } else if (situationType instanceof StopSituation) {

                    buffer.append(" xsi:type=\"StopSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((StopSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((StopSituation) (situationType)).getSituationQualifier();

                    if (attributeValue != null) {

                        buffer.append(" situationQualifier=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                } else if (situationType instanceof ConnectSituation) {

                    buffer.append(" xsi:type=\"ConnectSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((ConnectSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((ConnectSituation) (situationType)).getSituationDisposition();

                    if (attributeValue != null) {

                        buffer.append(" situationDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                } else if (situationType instanceof ReportSituation) {

                    buffer.append(" xsi:type=\"ReportSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((ReportSituation) (situationType)).getReportCategory();

                    if (attributeValue != null) {

                        buffer.append(" reportCategory=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }

                else if (situationType instanceof FeatureSituation) {

                    buffer.append(" xsi:type=\"FeatureSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((FeatureSituation) (situationType)).getFeatureDisposition();

                    if (attributeValue != null) {

                        buffer.append(" featureDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                } else if (situationType instanceof ConfigureSituation) {

                    buffer.append(" xsi:type=\"ConfigureSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((ConfigureSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                } else if (situationType instanceof DependencySituation) {

                    buffer.append(" xsi:type=\"DependencySituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((DependencySituation) (situationType)).getDependencyDisposition();

                    if (attributeValue != null) {

                        buffer.append(" dependencyDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                } else if (situationType instanceof CreateSituation) {

                    buffer.append(" xsi:type=\"CreateSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((CreateSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }

                else if (situationType instanceof DestroySituation) {

                    buffer.append(" xsi:type=\"DestroySituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((DestroySituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }

                else if (situationType instanceof AvailableSituation) {

                    buffer.append(" xsi:type=\"AvailableSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((AvailableSituation) (situationType)).getOperationDisposition();

                    if (attributeValue != null) {

                        buffer.append(" operationDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((AvailableSituation) (situationType)).getProcessingDisposition();

                    if (attributeValue != null) {

                        buffer.append(" processingDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((AvailableSituation) (situationType)).getAvailabilityDisposition();

                    if (attributeValue != null) {

                        buffer.append(" availabilityDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }

                else if (situationType instanceof RequestSituation) {

                    buffer.append(" xsi:type=\"RequestSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((RequestSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((RequestSituation) (situationType)).getSituationQualifier();

                    if (attributeValue != null) {

                        buffer.append(" situationQualifier=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }

                else if (situationType instanceof OtherSituation) {

                    buffer.append(" xsi:type=\"OtherSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append(">");

                    String anyData = ((OtherSituation) (situationType)).getAny();

                    if (anyData != null) {

                        if (format) {
                            buffer.append(LINE_SEPARATOR);
                            buffer.append("\t\t\t");
                        }

                        //Do NOT normalize since this string is assumed to be a
                        // valid XML fragment:
                        buffer.append(anyData);
                    }

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t\t");
                    }

                    buffer.append("</situationType>");
                } else {
                    buffer.append("/>");
                }
            }

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("</situation>");
        }

        List anyElements = commonBaseEvent.getAny();
        String anyElement = null;

        for (int counter = 0; counter < anyElements.size(); counter++) {

            anyElement = ((String) (anyElements.get(counter)));

            if (anyElement != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t");
                }

                //Do NOT normalize since this string is assumed to be a
                // valid XML fragment:
                buffer.append(anyElement);
            }
        }

        if (format) {
            buffer.append(LINE_SEPARATOR);
        }

        buffer.append("</CommonBaseEvent>");

        return (buffer.toString());
    }

    /**
     * Converts an XML document to an <code>CommonBaseEvent</code> object.
     * <p>
     * If the XML document contains more than one <code>CommonBaseEvent</code>
     * element, the first <code>CommonBaseEvent</code> element if used.
     * 
     * @param commonBaseEvent
     *            The <code>CommonBaseEvent</code> to be populated by the XML
     *            document.
     * @param xmlDocumentString
     *            The XML document to populate the <code>CommonBaseEvent</code>
     *            object.
     * @since 1.0
     * @deprecated Use {@link #eventsFromCanonicalXMLDoc(String)}.
     */
    public static synchronized void fromCanonicalXMLDocString(CommonBaseEvent commonBaseEvent, String xmlDocumentString) {

        try {

            CommonBaseEvent[] events = eventsFromCanonicalXMLDoc(xmlDocumentString);

            if ((commonBaseEvent != null) && (events != null) && (events[0] != null)) {

                TemplateContentHandlerImpl contentHandler = new TemplateContentHandlerImpl();
                contentHandler.setTemplateEvent(events[0]);

                commonBaseEvent.setContentHandler(contentHandler);

                commonBaseEvent.complete();

                commonBaseEvent.setContentHandler(null);
            } else {
                commonBaseEvent = null;
            }
        } catch (Exception e) {
            //Ignore for backwards compatibility.
        }
    }

    /**
     * Converts an XML document to an array of <code>CommonBaseEvent</code>
     * objects without validation.
     * <p>
     * This API will return <code>null</code> if no <code>CommonBaseEvent</code> fragments 
     * exist in the XML document.
     * <p>
     * 
     * @param xmlDocumentString
     *            The XML document to populate the <code>CommonBaseEvent</code>s
     *            array.
     * @return The <code>CommonBaseEvent</code>s array to be populated by the
     *         XML document, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized CommonBaseEvent[] eventsFromCanonicalXMLDoc(String xmlDocumentString) throws FormattingException {
        return (eventsFromCanonicalXMLDoc(xmlDocumentString,false));
    }

    /**
     * Potentially validates and converts an XML document to an array of <code>CommonBaseEvent</code>
     * objects.
     * <p>
     * This API will return <code>null</code> if no <code>CommonBaseEvent</code> fragments 
     * exist in the XML document.
     * <p>
     * 
     * @param xmlDocumentString
     *            The XML document to populate the <code>CommonBaseEvent</code>s
     *            array.
     * @param validate
     *            If the XML document is validated using the Common Base Event v1.0.1 schema.
     * @return The <code>CommonBaseEvent</code>s array to be populated by the
     *         XML document, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized CommonBaseEvent[] eventsFromCanonicalXMLDoc(String xmlDocumentString,boolean validate) throws FormattingException {
        return (eventsFromCanonicalXMLDoc(new InputSource(new StringReader(xmlDocumentString)),validate));
    }

    /**
     * Converts a Document Object Model (DOM) to an array of <code>CommonBaseEvent</code>
     * objects without validation.
     * <p>
     * This API will return <code>null</code> if no <code>CommonBaseEvent</code> elements 
     * exist in the Document Object Model (DOM).
     * <p>
     * 
     * @param document
     *            The Document Object Model (DOM) to populate the <code>CommonBaseEvent</code>s
     *            array.
     * @return The <code>CommonBaseEvent</code>s array to be populated by the
     *         Document Object Model (DOM), or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized CommonBaseEvent[] eventsFromCanonicalXMLDoc(Document document) throws FormattingException {
        return (eventsFromCanonicalXMLDoc(document,false));
    }

    /**
     * Potentially validates and converts a Document Object Model (DOM) to an array of <code>CommonBaseEvent</code>
     * objects.
     * <p>
     * This API will return <code>null</code> if no <code>CommonBaseEvent</code> elements 
     * exist in the Document Object Model (DOM).
     * <p>
     * 
     * @param document
     *            The Document Object Model (DOM) to populate the <code>CommonBaseEvent</code>s
     *            array.
     * @param validate
     *            If the Document Object Model (DOM) is validated using the Common Base Event v1.0.1 schema.
     * @return The <code>CommonBaseEvent</code>s array to be populated by the
     *         Document Object Model (DOM), or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized CommonBaseEvent[] eventsFromCanonicalXMLDoc(Document document,boolean validate) throws FormattingException {

        try {
            return (eventsFromCanonicalXMLDoc(XmlUtility.serialize(document),validate));            
        } 
        catch (SerializationException s) {
            throw new FormattingException(s);
        }
    }
   
    /**
     * Converts an XML document from an <code>InputSource</code> to an array
     * of <code>CommonBaseEvent</code> objects without validation.
     * <p>
     * This API will return <code>null</code> if no <code>CommonBaseEvent</code> fragments 
     * exist in the XML document.
     * <p>
     * 
     * @param InputSource
     *            The <code>InputSource</code> to populate the
     *            <code>CommonBaseEvent</code>s array.
     * @return The <code>CommonBaseEvent</code>s array to be populated by the
     *         XML document, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized CommonBaseEvent[] eventsFromCanonicalXMLDoc(InputSource inputSource) throws FormattingException {
        return (eventsFromCanonicalXMLDoc(inputSource,false));
    }

    /**
     * Potentially validates and converts an XML document from an <code>InputSource</code> to an array
     * of <code>CommonBaseEvent</code> objects.
     * <p>
     * This API will return <code>null</code> if no <code>CommonBaseEvent</code> fragments 
     * exist in the XML document.
     * <p>
     * 
     * @param InputSource
     *            The <code>InputSource</code> to populate the
     *            <code>CommonBaseEvent</code>s array.
     * @param validate
     *            If the XML document is validated using the Common Base Event v1.0.1 schema.
     * @return The <code>CommonBaseEvent</code>s array to be populated by the
     *         XML document, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized CommonBaseEvent[] eventsFromCanonicalXMLDoc(InputSource inputSource,boolean validate) throws FormattingException {
        return (parse(inputSource,validate).getCommonBaseEvents());
    }
    
    /**
     * Converts an XML document from an <code>InputStream</code> to an array
     * of <code>CommonBaseEvent</code> objects without validation.
     * <p>
     * This API will return <code>null</code> if no <code>CommonBaseEvent</code> fragments 
     * exist in the XML document.
     * <p>
     * 
     * @param InputSource
     *            The <code>InputStream</code> to populate the
     *            <code>CommonBaseEvent</code>s array.
     * @return The <code>CommonBaseEvent</code>s array to be populated by the
     *         XML document, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized CommonBaseEvent[] eventsFromCanonicalXMLDoc(InputStream inputStream) throws FormattingException {
        return (eventsFromCanonicalXMLDoc(inputStream,false));
    }

    /**
     * Potentially validates and converts an XML document from an <code>InputStream</code> to an array
     * of <code>CommonBaseEvent</code> objects.
     * <p>
     * This API will return <code>null</code> if no <code>CommonBaseEvent</code> fragments 
     * exist in the XML document.
     * <p>
     * 
     * @param InputSource
     *            The <code>InputStream</code> to populate the
     *            <code>CommonBaseEvent</code>s array.
     * @param validate
     *            If the XML document is validated using the Common Base Event v1.0.1 schema.
     * @return The <code>CommonBaseEvent</code>s array to be populated by the
     *         XML document, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized CommonBaseEvent[] eventsFromCanonicalXMLDoc(InputStream inputStream,boolean validate) throws FormattingException {
        return (eventsFromCanonicalXMLDoc(new InputSource(inputStream),validate));
    }

    /**
     * Converts an XML document from an <code>File</code> to an array of
     * <code>CommonBaseEvent</code> objects without validation.
     * <p>
     * This API will return <code>null</code> if no <code>CommonBaseEvent</code> fragments 
     * exist in the XML document.
     * <p>
     * 
     * @param InputSource
     *            The <code>File</code> to populate the
     *            <code>CommonBaseEvent</code>s array.
     * @return The <code>CommonBaseEvent</code>s array to be populated by the
     *         XML document, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized CommonBaseEvent[] eventsFromCanonicalXMLDoc(File file) throws FormattingException {
        return (eventsFromCanonicalXMLDoc(file,false));
    }

    /**
     * Potentially validates and converts an XML document from an <code>File</code> to an array of
     * <code>CommonBaseEvent</code> objects.
     * <p>
     * This API will return <code>null</code> if no <code>CommonBaseEvent</code> fragments 
     * exist in the XML document.
     * <p>
     * 
     * @param InputSource
     *            The <code>File</code> to populate the
     *            <code>CommonBaseEvent</code>s array.
     * @param validate
     *            If the XML document is validated using the Common Base Event v1.0.1 schema.
     * @return The <code>CommonBaseEvent</code>s array to be populated by the
     *         XML document, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0.1
     */
    public static synchronized CommonBaseEvent[] eventsFromCanonicalXMLDoc(File file,boolean validate) throws FormattingException {

        try {
            return (eventsFromCanonicalXMLDoc(new InputSource(new FileInputStream(file)),validate));
        } 
        catch (FileNotFoundException f) {
            throw new FormattingException(f);
        }
    }

    /**
     * Converts an XML fragment to an <code>CommonBaseEvent</code> object.
     * <p>
     * 
     * @param commonBaseEvent
     *            The <code>CommonBaseEvent</code> to be populated by the XML
     *            fragment.
     * @param xmlFragmentString
     *            The XML fragment to populate the <code>CommonBaseEvent</code>
     *            object.
     * @since 1.0
     * @deprecated Use {@link #eventFromCanonicalXML(String)}.
     */
    public static synchronized void fromCanonicalXMLString(CommonBaseEvent commonBaseEvent, String xmlFragmentString) {

        try {

            CommonBaseEvent event = eventFromCanonicalXML(xmlFragmentString);

            if ((commonBaseEvent != null) && (event != null)) {

                TemplateContentHandlerImpl contentHandler = new TemplateContentHandlerImpl();
                contentHandler.setTemplateEvent(event);

                commonBaseEvent.setContentHandler(contentHandler);

                commonBaseEvent.complete();

                commonBaseEvent.setContentHandler(null);
            } else {
                commonBaseEvent = null;
            }
        } catch (Exception e) {
            //Ignore for backwards compatibility.
        }
    }

    /**
     * Converts an XML fragment to an <code>CommonBaseEvent</code> object without validation.
     * <p>
     * This API will return <code>null</code> if no <code>CommonBaseEvent</code> fragment 
     * exists in the XML fragment.
     * <p>
     * 
     * @param xmlFragmentString
     *            The XML fragment to populate the <code>CommonBaseEvent</code>
     *            object.
     * @return The <code>CommonBaseEvent</code> to be populated by the XML
     *         fragment, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0
     */
    public static synchronized CommonBaseEvent eventFromCanonicalXML(String xmlFragmentString) throws FormattingException {
        return (eventFromCanonicalXML(xmlFragmentString,false));
    }

    /**
     * Potentially validates and converts an XML fragment to an <code>CommonBaseEvent</code> object.
     * <p>
     * This API will return <code>null</code> if no <code>CommonBaseEvent</code> fragment 
     * exists in the XML fragment.
     * <p>
     * 
     * @param xmlFragmentString
     *            The XML fragment to populate the <code>CommonBaseEvent</code>
     *            object.
     * @param validate
     *            If the XML fragment is validated using the Common Base Event v1.0.1 schema.
     * @return The <code>CommonBaseEvent</code> to be populated by the XML
     *         fragment, or null.
     * @throws FormattingException
     *             A de-serialization error has occurred.
     * @since 1.0
     */
    public static synchronized CommonBaseEvent eventFromCanonicalXML(String xmlFragmentString,boolean validate) throws FormattingException {

        StringBuffer buffer = new StringBuffer(1024);

        buffer.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
        buffer.append("<CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd\">");
        buffer.append(xmlFragmentString);
        buffer.append("</CommonBaseEvents>");

        CommonBaseEvent[] events = eventsFromCanonicalXMLDoc(buffer.toString(),validate);

        if (events != null) { 
            return (events[0]); 
        }

        return null;
    }

    /**
     * Private API to serialize the parameter <code>ExtendedDataElement</code>
     * object to an XML fragment.
     * <p>
     * 
     * @param extendedDataElement
     *            The <code>ExtendedDataElement</code> object to be serialized
     *            to an XML fragment.
     * @param tagName
     *            The name of the root tag or element of the serialized XML
     *            fragment. If null, "extendedDataElements" is used.
     * @param format
     *            If the serialized XML fragment is formatted with
     *            platform-dependent new line(s) and tab(s).
     * @param indent
     *            The number of spaces each tag or element of the serialized XML
     *            fragment is indented. This value is only used if the
     *            <code>format</code> parameter is true.
     * @return The serialized XML fragment of the parameter
     *         <code>ExtendedDataElement</code> object.
     */
    private static String getExtendedDataElementXML(ExtendedDataElement extendedDataElement, String tagName, boolean format, int indent) {

        if (extendedDataElement != null) {

            StringBuffer buffer = new StringBuffer(512);

            if (format) {

                buffer.append(LINE_SEPARATOR);

                for (int counter = 0; counter < indent; counter++) {
                    buffer.append("\t");
                }
            }

            buffer.append("<");

            if (tagName != null) {
                buffer.append(tagName);
            } else {
                buffer.append("extendedDataElements");
            }

            String attributeValue = extendedDataElement.getName();

            //For performance reasons, do not normalize attribute value since
            // of type xsd:Name which is guaranteed to not contain mark-up
            // characters:
            if (attributeValue != null) {
                buffer.append(" name=\"");
                buffer.append(attributeValue);
                buffer.append("\"");
            }

            attributeValue = extendedDataElement.getType();

            //For performance reasons, do not normalize attribute value since
            // of type xsd:Name which is guaranteed to not contain mark-up
            // characters:
            if (attributeValue != null) {
                buffer.append(" type=\"");
                buffer.append(attributeValue);
                buffer.append("\"");
            }

            buffer.append(">");

            if (extendedDataElement.getTypeAsInt() == ExtendedDataElement.TYPE_HEX_BINARY_VALUE) {

                String hexValues = extendedDataElement.getHexValue();

                if (hexValues != null) {

                    if (format) {

                        buffer.append(LINE_SEPARATOR);

                        for (int counter = 0; counter < (indent + 1); counter++) {
                            buffer.append("\t");
                        }
                    }

                    //For performance reasons, do not normalize element value
                    // since of type xsd:hexBinary which is guaranteed to not
                    // contain mark-up characters:
                    buffer.append("<hexValue>");
                    buffer.append(hexValues);
                    buffer.append("</hexValue>");
                }
            } else {

                List values = extendedDataElement.getValues();

                String valuesString = null;

                for (int counter = 0; counter < values.size(); counter++) {

                    valuesString = ((String) (values.get(counter)));

                    if (valuesString != null) {

                        if (format) {

                            buffer.append(LINE_SEPARATOR);

                            for (int index = 0; index < (indent + 1); index++) {
                                buffer.append("\t");
                            }
                        }

                        buffer.append("<values>");
                        buffer.append(XmlUtility.normalize(valuesString));
                        buffer.append("</values>");
                    }
                }
            }

            List childDataElements = extendedDataElement.getChildren();

            for (int counter = 0; counter < childDataElements.size(); counter++) {
                buffer.append(getExtendedDataElementXML(((ExtendedDataElement) childDataElements.get(counter)), "children", format, (format ? (indent + 1) : 0)));
            }

            if (format) {

                buffer.append(LINE_SEPARATOR);

                for (int counter = 0; counter < indent; counter++) {
                    buffer.append("\t");
                }
            }

            buffer.append("</");

            if (tagName != null) {
                buffer.append(tagName);
            } else {
                buffer.append("extendedDataElements");
            }

            buffer.append(">");

            return (buffer.toString());
        } else {
            return "";
        }
    }
    
    private static String resolveCommonBaseEventSchemaLocation(){
        
        URL url = null;

        //Attempt to find the actual location of the
        //schema file on the classpath:
        try {

            //Consult the current class' class loader to find the
            //actual local location of the schema file
            //using privileged security:
            url = ((URL) (AccessController.doPrivileged(new PrivilegedExceptionAction() {

                public Object run() throws Exception {
                    return (this.getClass().getClassLoader().getResource(COMMON_BASE_EVENT_SCHEMA_NAME));
                }
            })));
        } catch (Throwable t) {
            //Ignore exception since the current class' class loader is
            //null (e.g. current class was loaded by the bootstrap
            //class loader) or insufficient security privileges for
            //accessing the current class' class loader.
        }

        if (url == null) {

            try {

                //If the current class' class loader cannot find the actual
                //local location of the schema file,
                //consult the system's class loader to find the actual
                //local location of the schema file
                //using privileged security:
                url = ((URL) (AccessController.doPrivileged(new PrivilegedExceptionAction() {

                    public Object run() throws Exception {
                        return (ClassLoader.getSystemClassLoader().getResource(COMMON_BASE_EVENT_SCHEMA_NAME));
                    }
                })));
            } catch (Throwable t) {
                //Ignore exception since insufficient security privileges
                // for
                //accessing the system's class loader.
            }

            if (url == null) {

                try {

                    //If the system's class loader cannot find the actual
                    //local location of the schema file,
                    //consult the current thread's class loader to find the
                    //actual local location of the schema
                    //file using privileged security:
                    url = ((URL) (AccessController.doPrivileged(new PrivilegedExceptionAction() {

                        public Object run() throws Exception {
                            return (Thread.currentThread().getContextClassLoader().getResource(COMMON_BASE_EVENT_SCHEMA_NAME));
                        }
                    })));
                } catch (Throwable t) {
                    //Ignore exception since insufficient security
                    // privileges for
                    //accessing the current thread's class loader.
                }
            }
        }
        
        if (url != null) {
            return (url.toExternalForm());
        }
        
        return null;
    }
    
    private static SAXEventHandler parse(InputSource inputSource,boolean validate) throws FormattingException{
            
        try {
            
	        SAXEventHandler saxEventHandler = new SAXEventHandler();
	
	        XMLReader xmlReader = null;
	        
            //NOTE: The Sun JDK v.1.4.x that includes the Crimson XML parser implementation
            //does not have a default XML reader implementation.
            
            //Attempt to resolve the XML reader:
	        try {
	            xmlReader = XMLReaderFactory.createXMLReader();
                
            } 
	        catch (SAXException s) {
                
	            //NOTE: The Sun JDK v.1.4.x that includes the Crimson XML parser implementation
	            //uses the value of the system property "org.xml.sax.driver" as the full name of 
	            //XML reader Java class.
	            System.setProperty("org.xml.sax.driver","org.apache.crimson.parser.XMLReaderImpl");
	            
	            xmlReader = XMLReaderFactory.createXMLReader();
            }
            
	        xmlReader.setContentHandler(saxEventHandler);        	
	        xmlReader.setErrorHandler(saxEventHandler);        	  
	
	        xmlReader.setFeature("http://xml.org/sax/features/validation", validate);
	        xmlReader.setFeature("http://xml.org/sax/features/namespaces", validate);
	        
            //NOTE: The Sun JDK v.1.4.x that includes the Crimson XML parser implementation
            //does not support schema validation.
            
            //Attempt to configure the XML parser to validate the XML file based on the associated schema(s):
	        if(validate){
	            
	            try {                    
		            
	                xmlReader.setFeature("http://apache.org/xml/features/validation/schema", true);
		            xmlReader.setProperty("http://apache.org/xml/properties/schema/external-schemaLocation", "http://www.ibm.com/AC/commonbaseevent1_0_1 ".concat(COMMON_BASE_EVENT_SCHEMA_LOCATION_URL));
	            } 
	            catch (SAXNotRecognizedException s) {

	                //If the XML parser cannot be configured to validate the XML file based on the associated schema(s), configure a non-validating XML parser:
	                xmlReader.setFeature("http://xml.org/sax/features/validation", false);
			        xmlReader.setFeature("http://xml.org/sax/features/namespaces", false);
                }
	        }
	        
	        xmlReader.parse(inputSource);
	        
	        return saxEventHandler;
        } 
        catch (Exception e) {
            throw new FormattingException(e);
        }
    }
}