/*******************************************************************************
 * Copyright (c) 2003,2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.trace.sample.loganalyzer;

import java.lang.reflect.InvocationTargetException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.analysis.engine.ILogAnalyzer;
import org.eclipse.hyades.log.ui.internal.LogUIPlugin;
import org.eclipse.hyades.log.ui.internal.util.LogUIConstants;
import org.eclipse.hyades.log.ui.internal.views.ExtensionPointHandler;
import org.eclipse.hyades.log.ui.internal.views.LogContentProvider;
import org.eclipse.hyades.log.ui.internal.views.LogPage;
import org.eclipse.hyades.log.ui.internal.views.LogPaneTreeViewer;
import org.eclipse.hyades.log.ui.internal.views.LogViewer;
import org.eclipse.hyades.log.ui.internal.views.ExtensionPointHandler.ConfigurationElement;
import org.eclipse.hyades.models.hierarchy.util.ISDBLoader;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;

/**
 * @author sleeloy
 *
 * This action class will analyze an apache error log using 
 * a specific symptom database.  The corresponding viewers are 
 * also refeshed to reflect the analyzed error log file.
 */
public class AnalyzeLogAction implements IRunnableWithProgress {

	private String symptomDBPath;
	private String projectName;
	private String sdbFileName;
	
	private IConfigurationElement newSymptomDBLoaderElement = null;
	private boolean newSymptomDBLoaderElementInit;
	
	private ExtensionPointHandler eph = ExtensionPointHandler.getExtensionPointHandler();
	protected EObject mofObject;
	private LogPaneTreeViewer logPaneTreeViewer;	
	/**
	 * This method is were all the work is done.  The error log file is analyzed
	 * using a specified symptom database.  The Log Tree Viewer is also refreshed
	 * to reflect the analyzed error log file.
	 * 
	 * @see org.eclipse.jface.operation.IRunnableWithProgress#run(org.eclipse.core.runtime.IProgressMonitor)
	 * @param monitor - shows the progress of this action.
	 */
	public void run(IProgressMonitor monitor)
		throws InvocationTargetException, InterruptedException {
			//Need to set the symptom database path
			IPreferenceStore store = LogUIPlugin.getDefault().getPreferenceStore();
			
			if(sdbFileName.length()>0 && sdbFileName.endsWith("xml")){				
				importDB(monitor);
				
			}else			
			if(symptomDBPath.length()>0 && symptomDBPath.endsWith("xml")){
				sdbFileName = Platform.getLocation()+"/"+symptomDBPath; 
				importDB(monitor);
			}
			String value = getNewSDBPathPreference(store.getString(LogUIConstants.SYMPTOM_DB_PATH_KEY), symptomDBPath);
			store.setValue(LogUIConstants.SYMPTOM_DB_PATH_KEY, value);
			store.setValue(LogUIConstants.PROJECT_NAME_KEY, projectName);
			store.setValue(LogUIConstants.SDB_FILE_NAME_KEY, sdbFileName);
			
			//Load symptom database
			List  objects = getObjectsToAnalyze();			
			List aLogAnalyzerList = eph.getLogAnalyzers();
			ILogAnalyzer aLogAnalyzer = null;
			ILogAnalyzer newInstance = null;
			if(aLogAnalyzerList != null && aLogAnalyzerList.size() >0)
			{
				for (Iterator iter = aLogAnalyzerList.iterator();iter.hasNext();) {
					ConfigurationElement config = (ConfigurationElement) iter.next();
					if(config.getAnalyzer() instanceof ILogAnalyzer){
						aLogAnalyzer = (ILogAnalyzer)config.getAnalyzer();
					}
				}
								
			}
			if(aLogAnalyzer==null)
			    return;
			
			aLogAnalyzer.loadDatabase();
			if(aLogAnalyzer.errorMsg()==null){	
				aLogAnalyzer.analyze(objects, null);
			}
			logPaneTreeViewer.refresh();

			//Need to send an event so that the Analysis Result area will be updated.
			Event event = new Event();
			event.item = ((Tree) logPaneTreeViewer.getControl()).getItems()[0];
			logPaneTreeViewer.getControl().notifyListeners(SWT.Selection, event);			

	}
	/**
	 * 
	 * @return
	 */
	private String getNewSDBPathPreference(String previousValue, String symptomDBPath)
	{
		//	case 1
		// nothing is stores in the preference
		// previous.length() == 0 if nothing is stored in the preference
		if (previousValue.length() == 0) 
		{
			return symptomDBPath + ",1"; 
		}
		 else
		 {
			 int index = previousValue.indexOf(symptomDBPath);
		
		
			 if (index < 0)
			 {
				 // case 2
				 // symptomdbpath does not exist in preference (ie.db has not been loaded)
				   return symptomDBPath + ",1" + ";" + previousValue;
			   }	
			   else
			   {
	
				   if ((previousValue.substring(symptomDBPath.length() + index + 1,symptomDBPath.length() + index + 2 )).equals("0"))
				   {
					   //case 3
					   //loaded but is followed by ,0 - meaning not used for log analysis
					   //change 0 to 1
					   String temp1 = previousValue.substring(0, index + symptomDBPath.length() + 1);
					   String temp2 = previousValue.substring(index + symptomDBPath.length() + 2, previousValue.length());
					   return temp1 + "1" + temp2;
				   }
	
				   else if ((previousValue.substring(symptomDBPath.length() + index + 1, symptomDBPath.length() + index + 2)).equals("1"))
				   {
					   // case 4
					   // loaded and is follow by a ,1
					   // do nothing
					   return previousValue;
				   }
		
				 }
		 }
		 return "";
	}
	
	/**
	 * This class retrieves the entire content selection from the Log Viewer.  All the contents
	 * in the log viewer will be analyzed.
	 * 
	 * @return List that represents the records shown in the Log Viewer.
	 */
	public List getObjectsToAnalyze(){
		//return the entire content selection
		try {
					IWorkbenchPage persp = UIPlugin.getActivePage();
					LogViewer logviewer = (LogViewer)persp.showView("org.eclipse.hyades.log.ui.internal.views.LogViewer");
					LogPage logPage = (LogPage)logviewer.getPage(mofObject);
					if(logPage==null){
						logviewer.addViewPage(mofObject);
						logPage = (LogPage)logviewer.getPage(mofObject);
					}
					
					logPaneTreeViewer = logPage.getView().getViewer();
					LogContentProvider logContentProvider = (LogContentProvider)logPaneTreeViewer.getContentProvider();
					return Arrays.asList(logContentProvider.getChildren(mofObject));
		} catch (PartInitException e) {
			e.printStackTrace();
		}
		return Collections.EMPTY_LIST;
	}


	/**
	 * Returns the symptom database location that is used to analyze the error log file.
	 * @return String that represents the symptom database location that is used to analyze the error log file.
	 */
	public String getSymptomDBPath() {
		return symptomDBPath;
	}

	/**
	 * Sets the symptom database location that is used to analyze the error log file.
	 * @param string - that represents the symptom database location that is used to analyze the error log file.
	 */
	public void setSymptomDBPath(String string) {
		symptomDBPath = string;
	}
	
	/**
	 * Sets the project Name of where to import the db into
	 * @param pName
	 */
	public void setProjectName(String pName)
	{
		projectName = pName;
	}
		
	/**
	 * specifies what the sdb file should be called in the workspace
	 * @param string
	 */
	public void setSDBFileName(String string)
	{
		sdbFileName = string;
	}
		
	

	/**
	 * Returns the Mof object that represents the error log file.  This object will be analyzed using the 
	 * specified symptom database.
	 * @return EObject represents the error log file that will be analyzed.
	 */
	public EObject getMofObject() {
		return mofObject;
	}

	/**
	 * Sets the Mof object that represents the error log file.  This object will be analyzed using the 
	 * specified symptom database.
	 * @param object - represents the error log file that will be analyzed.
	 */
	public void setMofObject(EObject object) {
		mofObject = object;
	}

	public void importDB(IProgressMonitor monitor) {
		//imports the DB since it is not yet within the workspace		
		final String PLATFORM_PREFIX = "platform:/resource";
		try {
			
//			String symptomdbPath = LogUIPlugin.getDefault().getPreferenceStore().getString(LogUIConstants.SYMPTOM_DB_PATH_KEY);
//			String pName = LogUIPlugin.getDefault().getPreferenceStore().getString(LogUIConstants.PROJECT_NAME_KEY);
//			String sdbFileName = LogUIPlugin.getDefault().getPreferenceStore().getString(LogUIConstants.SDB_FILE_NAME_KEY);
//
			IPath path = new Path(symptomDBPath);
			
			ISDBLoader l = getSymDBLoader();
			symptomDBPath = path.removeFileExtension().addFileExtension(l.getExtension()).toString();
			l.loadDatabase(new Path(sdbFileName).toString(), new Path(PLATFORM_PREFIX + symptomDBPath).toString(), monitor);			

		} catch (Exception e) {
			e.printStackTrace();
		}

	}
	
	public ISDBLoader getSymDBLoader() {
		try {
			if (!newSymptomDBLoaderElementInit) {
				newSymptomDBLoaderElementInit = true;
				loadExtention();

			}

			if (newSymptomDBLoaderElement != null)
				return (ISDBLoader) newSymptomDBLoaderElement
						.createExecutableExtension("class");
		} catch (CoreException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return null;
	}

	public void loadExtention() {
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IConfigurationElement[] configurationElements = registry.getConfigurationElementsFor("org.eclipse.hyades.models.hierarchy.sdbLoader");//$NON-NLS-1$		
		if (configurationElements!=null) {
			for (int i = 0; i < configurationElements.length; i++) {
				IConfigurationElement configurationElement = configurationElements[i];
				if (configurationElement.getName().equals("loader")) {//$NON-NLS-1$
//					System.out.println(configurationElement.getAttribute("name")+" is loaded");
					newSymptomDBLoaderElement = configurationElement;
					break;
				}
			}
		}
	}
}
