/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.log.ui.internal.navigator;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.log.ui.internal.LogUIPlugin;
import org.eclipse.hyades.models.hierarchy.CorrelationContainerProxy;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.util.HierarchyResourceSetImpl;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.ui.internal.extension.NavigatorExtensionUtil;
import org.eclipse.hyades.ui.internal.logicalfolder.LogicalFolder;
import org.eclipse.hyades.ui.internal.util.ResourceUtil;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;

/**
 * @author marcelop
 * @since 1.1.0
 */
public class LogContentProvider 
implements ITreeContentProvider, IDisposable
{
	public final static String SD_FILE_EXTENSION = "trcdbxmi";
	public final static String SYMPTOM_FILE_EXTENSION = "symptom"; 
	//public final static String CORR_FILE_EXTENSION = "corrxmi";
	
	private TreeViewer treeViewer;
	
	private LogicalFolder logFolder;
	private LogicalFolder sdFolder;
	private LogicalFolder corrFolder;
	
	private boolean filesLoaded = false;
	private LogNavigator navigator;

	public LogContentProvider(){		
	}
	 
	public LogContentProvider(LogNavigator navigator){
		this.navigator = navigator;
	}
	
	/**
	 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
	 */
	public void dispose()
	{
		if (logFolder != null)
			logFolder.dispose();
		if (sdFolder != null)
			sdFolder.dispose();
			
		treeViewer = null;
	}
	
	public ResourceSet getResourceSet()
	{
		return HierarchyResourceSetImpl.getInstance();
	}
	
	public TreeViewer getTreeViewer()
	{
		return treeViewer;
	}
	
	protected void loadFiles()
	{
		String[] extensions = new String[]{SD_FILE_EXTENSION, SYMPTOM_FILE_EXTENSION, TraceConstants.MONITOR_EXT};
		Map filesByExtension = ResourceUtil.getFiles(ResourcesPlugin.getWorkspace().getRoot(), extensions, null);

		getSDFolder().getChildren().addAll((List)filesByExtension.get(SD_FILE_EXTENSION));
		getSDFolder().getChildren().addAll((List)filesByExtension.get(SYMPTOM_FILE_EXTENSION));

		EObject[] resourceSetObj = EMFUtil.loadMonitorsFromResourceSet(getResourceSet());
		loadIntoFolders(resourceSetObj);
		
		for(Iterator i = ((List)filesByExtension.get(TraceConstants.MONITOR_EXT)).iterator(); i.hasNext();)
		{
			EObject[] eObjects = EMFUtil.load(getResourceSet(), (IFile)i.next());
			loadIntoFolders(eObjects);				
		}
		
		filesLoaded = true;
	}
		
	private void loadIntoFolders(EObject[] eObjects){
		
		for (int j = 0, maxj = eObjects.length; j < maxj; j++)
		{
			if(eObjects[j] instanceof TRCMonitor)
			{					
				getCorrelationFolder().getChildren().addAll(((TRCMonitor)eObjects[j]).getCorrelationContainerProxies());
				for(Iterator k = ((TRCMonitor)eObjects[j]).getNodes().iterator(); k.hasNext();){
					TRCNode node = (TRCNode)k.next();
					getLogFolder().getChildren().add(node);
				}

			}
		}	
	}
	
	public void refreshFromLocal(){
		filesLoaded = false;
		clearFolders();
		getTreeViewer().refresh();	
	}

	public LogicalFolder getLogFolder()
	{
		if (logFolder == null)
		{
			logFolder = new LogicalFolder(LogUIPlugin.getResourceString("LOG_NAV_LOGFLD"))
			{
				protected List createChildren()
				{
					return new UniqueEList();
				}
			};
		}
		return logFolder;
	}

	public LogicalFolder getSDFolder()
	{
		if (sdFolder == null)
		{
			sdFolder = new LogicalFolder(LogUIPlugin.getResourceString("LOG_NAV_SDFLD"))
			{
				protected List createChildren()
				{
					return new UniqueEList();
				}
			};
		}
		return sdFolder;
	}
	
	public LogicalFolder getCorrelationFolder(){
		if (corrFolder == null)
		{
			corrFolder = new LogicalFolder(LogUIPlugin.getResourceString("LOG_NAV_CORRFLD"))
			{
				protected List createChildren()
				{
					return new UniqueEList();
				}
			};
		}
		
		return corrFolder;
	}

	/**
	 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
	 */
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
	{
		clearFolders();
		if(viewer instanceof TreeViewer)
			treeViewer = (TreeViewer)viewer;
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
	 */
	public Object getParent(Object element)
	{
		if(element instanceof TRCAgentProxy)
			return getLogFolder();	
		if(element instanceof CorrelationContainerProxy){
			return getCorrelationFolder();
		}
	
		if (element instanceof IFile)
		{
			if(SD_FILE_EXTENSION.equals(((IFile)element).getFileExtension())
				|| SYMPTOM_FILE_EXTENSION.equals(((IFile)element).getFileExtension()))
				return getSDFolder();
		}
		
		if(element instanceof LogicalFolder)
			return ((LogicalFolder)element).getParent();

		return null;
	}

	/**
	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
	 */
	public Object[] getElements(Object inputElement)
	{
		List list = new ArrayList();
		list.add(getLogFolder());
		list.add(getSDFolder());
		list.add(getCorrelationFolder());
		list.addAll(NavigatorExtensionUtil.getAllChildren(inputElement, navigator.getID()));
		return list.toArray();
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
	 */
	public boolean hasChildren(Object element)
	{
		if (NavigatorExtensionUtil.hasChildren(element, navigator.getID()))
			return true;
	
		if(element instanceof LogicalFolder)
		{
			if(!filesLoaded)
				loadFiles();
			return !((LogicalFolder)element).getChildren().isEmpty();
		}
		if(element instanceof CorrelationContainerProxy){
			return ((CorrelationContainerProxy)element).getCorrelatedAgents().size() > 0;
		}
		if(element instanceof TRCNode){
			return EMFUtil.getLogAgentsFromNode((TRCNode)element).size()>0;
		}
		
		return false;
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
	 */
	public Object[] getChildren(Object parentElement)
	{
		List list = new ArrayList();
		
		if(parentElement==getLogFolder()){
			getLogFolder().getChildren().clear();
			getLogFolder().getChildren().addAll(getNodesFromResourceSet());
			list.addAll(getLogFolder().getChildren());	
		}
		
		else if(parentElement instanceof LogicalFolder)
			list.addAll(((LogicalFolder)parentElement).getChildren());

		else if(parentElement instanceof CorrelationContainerProxy){
			list.addAll(((CorrelationContainerProxy)parentElement).getCorrelatedAgents());
		}
		if(parentElement instanceof TRCNode){
			list.addAll(EMFUtil.getLogAgentsFromNode((TRCNode)parentElement));
		}
				
		list.addAll(NavigatorExtensionUtil.getAllChildren(parentElement, navigator.getID()));
		return list.toArray();
	}
			
	private void clearFolders(){

		if (logFolder != null)
			logFolder.getChildren().clear();
		if (sdFolder != null)
			sdFolder.getChildren().clear();
		if(corrFolder!=null)
			corrFolder.getChildren().clear();	
	
	}
	
	
	private List getNodesFromResourceSet(){
		
		List resources = getResourceSet().getResources();
		int size = resources.size();
		List nodes = new ArrayList();		
		for(int idx=0; idx<size;idx++){
			Resource res = (Resource)resources.get(idx);
			List ext = res.getContents();
			Iterator iter = ext.iterator();
			while (iter.hasNext())
			{
				Object obj = iter.next();
				if (obj instanceof TRCMonitor){					
					for (Iterator iterator = ((TRCMonitor)obj).getNodes().iterator();
						iterator.hasNext();
						) {
						
						TRCNode node = (TRCNode)iterator.next();
						if(EMFUtil.getLogAgentsFromNode(node).size()>0 && !nodes.contains(node))	
							nodes.add(node);						
					}
				}
			}			
		}
		return nodes;

	}
}
