/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Builder.java,v 1.5 2005/02/16 22:20:35 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.builder;

import java.util.Map;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.logging.adapter.Adapter;
import org.eclipse.hyades.logging.adapter.AdapterException;

/**
 * @author rduggan
 *
 * This is a builder for the Generic Log Adapter files.  This builder
 * responds to full and incremental builds.
 */
public class Builder extends IncrementalProjectBuilder {
	
	/**
	 * Parameterless constructor required for reflective construction.
	 */
	public Builder() {
		super();
	}
	
	/**
	 * @see org.eclipse.core.internal.events.InternalBuilder#build(int, java.util.Map, org.eclipse.core.runtime.IProgressMonitor)
	 */
	protected IProject[] build(int kind, Map args, IProgressMonitor monitor) throws CoreException {
		if (kind == IncrementalProjectBuilder.FULL_BUILD) {
			fullBuild(monitor);
		} 
		else {
			IResourceDelta delta = getDelta(getProject());
			if (delta == null) {
				fullBuild(monitor);
			}
			else {
				incrementalBuild(delta, monitor);
			}
		}
		return null;
	}
	
	/**
	 * Perform full build of a project
	 * @param monitor progress monitor for the build.  This is currently not used.
	 * @throws CoreException
	 */
	private void fullBuild(IProgressMonitor monitor) throws CoreException {
		getProject().accept(new MyBuildVisitor());
	}
	
	/**
	 * Performa a incremental build of a project.
	 * @param delta the resources that have changed.
	 * @param monitor the progress monitor.  This is currently not used.
	 * @throws CoreException
	 */
	private void incrementalBuild(IResourceDelta delta, IProgressMonitor monitor) throws CoreException {
		delta.accept(new MyBuildDeltaVisitor());

	}
	
	/**
	 * Build the resource.  This is where the GLA is instantiated and asked to validate the
	 * content.
	 * @param resource the resource that is to be built.  This must be a IResouce.FILE
	 */
	private void buildResource(IResource resource) {
		/* Clear any existing markers for this resource */
		try {
			resource.deleteMarkers("org.eclipse.hyades.logging.adapter.glaproblemmarker", true, IResource.DEPTH_INFINITE);
		} catch (CoreException e) {
			/*  TODO:  Handle this exception */
		}
		String path=resource.getLocation().toOSString();
		
		/* bugzilla 67337 - Only validate the file if its name has an extension of .adapter */
		int periodIndex = path.lastIndexOf('.');
		
		if (periodIndex >= 0 && path.substring(periodIndex).equalsIgnoreCase(".adapter")) {
			/* Run the adapter.  It will add new Markers for us */
			Adapter adapter = new Adapter();
			adapter.setComponentConfigPath(path);
			adapter.setContextConfigPath(path);
			try {
				adapter.validate();
			}
			catch(AdapterException e) {
				/* When validation is run, entries are added to the task list.  There is several things that may result in the failure
				 * before the validation even makes it to that stage.  Therefore we need to add them to the task list here
				 */
				try {
					IMarker marker=resource.createMarker("org.eclipse.hyades.logging.adapter.glaproblemmarker");
					if (marker.exists()) {
						marker.setAttribute(IMarker.MESSAGE, e.getLocalizedMessage());
						marker.setAttribute(IMarker.PRIORITY, IMarker.PRIORITY_HIGH);
						/* allow the user to edit the issue */
						marker.setAttribute(IMarker.USER_EDITABLE, true);
					}
				} catch (CoreException eprime) {
				     // We should be able to ignore this
				}
			}
		}
		
	}
	
	/**
	 * 
	 * @author rduggan
	 *
	 * This is the build vistor for full builds
	 */
	class MyBuildVisitor implements IResourceVisitor {
		
		public MyBuildVisitor() {	
		}
		
		public boolean visit(IResource res) {
			if(res.getType()==IResource.FILE) {
				buildResource(res);
			}
			return true;
		}
	}
	
	/**
	 * 
	 * @author rduggan
	 *
	 * This is the build vistor for incremental builds.
	 */
	class MyBuildDeltaVisitor implements IResourceDeltaVisitor {
		
		public MyBuildDeltaVisitor() {
		}
		
		public boolean visit(IResourceDelta delta) {
			IResource resource=delta.getResource();
			if(resource.getType()==IResource.FILE) {
				buildResource(resource);
			}
			return true;
		}
	}

}
