/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DrawableToolTip.java,v 1.16 2005/03/25 14:20:52 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.view;


import org.eclipse.hyades.uml2sd.ui.SDViewerPlugin;
import org.eclipse.hyades.uml2sd.ui.drawings.impl.ColorImpl;
import org.eclipse.hyades.uml2sd.util.TimeUtil;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;



/**
 * This class is used to reproduce the same tooltip behavior on Windows and Linux
 * when the mouse move hover the time compression bar
 * Used to display enlapsed time using a tooltip.
 * The tooltip is composed of 2 parts, the text value and below, a scale to visually
 * locate the value in a time range (usually the min an max enlapsed time in the whole diagram)
 * @author sveyrier
 */
public class DrawableToolTip implements PaintListener {

	/**
	 * The parent control where the tooltip must be drawn
	 */
	private Composite parent = null;
	/**
	 * The tooltip shell
	 */
	private Shell toolTipShell = null;
	/**
	 * Time range datas
	 */
	private double min,max,currentValue;
	
	private static int H_MARGIN = 10;
	private static int V_MARGIN = 10;
	private static int TEXT_SCALE_MARGIN = 20;
	private static int SCALE_LENGTH = 100;
	/**
	 * Unit name
	 */
	private String unitName = null;
	
	protected String msg;
	
	/**
	 * The color array used to repesent the 10 time range slices
	 */
	private Color[] col;

	public DrawableToolTip(Composite _parent)
	{
		parent = _parent;
		toolTipShell = new Shell(parent.getShell(),SWT.ON_TOP);
		toolTipShell.setLayout(new RowLayout());
		toolTipShell.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_INFO_BACKGROUND));
		toolTipShell.addPaintListener(this);
		toolTipShell.setSize(200,50);
		
		col = new Color[10];
		col[0] = new Color(Display.getDefault(),255,229,229);
		col[1] = new Color(Display.getDefault(),255,204,204);
		col[2] = new Color(Display.getDefault(),255,178,178);
		col[3] = new Color(Display.getDefault(),255,153,153);
		col[4] = new Color(Display.getDefault(),255,127,127);
		col[5] = new Color(Display.getDefault(),255,102,102);
		col[6] = new Color(Display.getDefault(),255,76,76);
		col[7] = new Color(Display.getDefault(),255,51,51);
		col[8] = new Color(Display.getDefault(),255,25,25);
		col[9] = new Color(Display.getDefault(),255,0,0);
	}
	
	/**
	 * Display the tooltip using the given time range(min,max) the current value and the time unit
	 * The tooltip will stay on screen until it is told otherwise
	 * 
	 * @param _value the current in the scale
	 * @param _min the scale min
	 * @param _max the scale max
	 * @param unit the scale unit
	 */
	public void showToolTip(double _value, double _min, double _max, String unit)
	{
		min = _min;
		max = _max;
		currentValue = _value;
		unitName =unit;
		
		int w = toolTipShell.getBounds().width;
		int h = toolTipShell.getBounds().height;
		Point hr = Display.getDefault().getCursorLocation();
		toolTipShell.setBounds(hr.x,hr.y+26,w,h);
		toolTipShell.setVisible(true);
	}
	
	/**
	 * Hide the tooltip
	 */
	public void hideToolTip()
	{
		toolTipShell.setVisible(false);
	}
	
	/**
	 * Draw the tooltip text on the control widget when a paint event is received
	 */
	public void paintControl(PaintEvent event)
	{
		msg=TimeUtil.timeToString(currentValue);	 //$NON-NLS-1$
		Point size = event.gc.textExtent(msg);		
		if (size.x<SCALE_LENGTH)
			size.x = SCALE_LENGTH;
		event.gc.drawText(msg,H_MARGIN,V_MARGIN,true); 
		event.gc.drawLine(H_MARGIN,V_MARGIN+TEXT_SCALE_MARGIN+size.y,H_MARGIN+SCALE_LENGTH,
					V_MARGIN+TEXT_SCALE_MARGIN+size.y);
					
		int step = SCALE_LENGTH/10;
			
//		double gr=Math.log(1+max-min)/10;
		double gr=(max-min)/10;
		double delta = currentValue-min;
		int colIndex = 0;
		ColorImpl color;
		if (gr!=0)
		{
//			colIndex = Math.round((float)(Math.log(1+delta)/gr));
			colIndex = Math.round((float)(delta/gr));
			if (colIndex > col.length)
				colIndex=col.length;
			else if (colIndex <= 1)
				colIndex=1;
		}else colIndex=1;
		
		
		
		for (int i=0;i<=10;i++)
		{
			if (i<10)
				event.gc.setBackground(col[i]);
			if ((i<colIndex)&&(i<10))
				event.gc.fillRectangle(H_MARGIN+i*step,V_MARGIN+TEXT_SCALE_MARGIN+size.y-5,step,11);
			if (i==0)
				event.gc.drawText(SDViewerPlugin.getResourceString("MIN_LABEL"), //$NON-NLS-1$
						H_MARGIN,size.y+2*V_MARGIN+TEXT_SCALE_MARGIN,true);
			if (i==0)
			{
				int len  = event.gc.textExtent(SDViewerPlugin.getResourceString("MAX_LABEL")).x; //$NON-NLS-1$
				event.gc.drawText(SDViewerPlugin.getResourceString("MAX_LABEL"), //$NON-NLS-1$
					H_MARGIN+SCALE_LENGTH-len+1,size.y+2*V_MARGIN+TEXT_SCALE_MARGIN,true);
			}
			int lineWidth = 10;
			if ((i==0)||(i==10))
				lineWidth = 14;
			event.gc.drawLine(H_MARGIN+i*step,V_MARGIN+TEXT_SCALE_MARGIN+size.y-lineWidth/2,
							H_MARGIN+i*step,V_MARGIN+TEXT_SCALE_MARGIN+size.y+lineWidth/2);
		}
		toolTipShell.setSize(size.x+2*H_MARGIN+2,2*size.y+3*V_MARGIN+TEXT_SCALE_MARGIN);
	}
	
	public String getText()
	{
		return msg;
	}
	
	public String getAccessibleText()
	{
		return TimeUtil.accessibleTimeToString(currentValue);
	}
	
	/**
	 * Dispose the system resource used by this kind of toolTips
	 * (a colors array essentialy)
	 *
	 */
	public void dispose()
	{
		for (int i=0; i<col.length;i++)
			col[i].dispose();	
	}
}
