package org.eclipse.hyades.logging.log4j;

import org.apache.log4j.spi.Filter;
import org.apache.log4j.spi.LoggingEvent;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CommonBaseEventFilter.java,v 1.3 2005/02/16 22:20:54 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/** 
 * Extension of the Log4J <code>org.apache.log4j.spi.Filter</code> abstract 
 * class used to provide control over what is logged at a more granular level 
 * than logging level for <code>org.apache.log4j.Logger</code> instances.
 * <p>
 * Only <code>org.apache.log4j.spi.LoggingEvent</code> instances that contain an 
 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code> as its
 * <code>message</code> are logged when this filter is utilized.
 * <p>
 * An instance of this filter will be returned from the <code>getFilter()</code> API on 
 * <code>org.apache.log4j.Appender</code> instances.
 * <p>  
 * Users may configure a <code>org.apache.log4j.Appender</code> to use the 
 * <code>CommonBaseEventFilter</code> by adding the following the entry to the 
 * <code>&lt;appender&gt;</code> element in the <code>log4j.xml</code> properties file:
 * <p>
 * <code>
 * &lt;!-- CommonBaseEventFilter --&gt;
 * &lt;!-- Filters all events that do not have a Common Base Event as their message:. --&gt;
 * &lt;filter class="org.eclipse.hyades.logging.log4j.CommonBaseEventFilter"/&gt;
 * </code>
 * <p>
 * Alternatively, an instantiation of this filter class may be set directly to 
 * <code>org.apache.log4j.Appender</code> instances by using the 
 * <code>addFilter()</code> API.
 * <p>
 * NOTE: The Log4J classes must be on the CLASSPATH at run-time to
 * utilize this filter class.
 * <p>
 *
 *  
 * @author    Paul E. Slauenwhite
 * @version   November 24, 2004
 * @since     July 20, 2004
 * @see       org.apache.log4j.spi.Filter
 * @see       org.apache.log4j.spi.LoggingEvent
 */
public class CommonBaseEventFilter extends Filter {

    /**
     * The API will only <code>DENY</code> the parameter <code>org.apache.log4j.spi.LoggingEvent</code> 
     * instance if it does NOT contain an <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code> 
     * as its <code>message</code>.  Otherwise, the API will return <code>DENY</code> the 
     * parameter <code>org.apache.log4j.spi.LoggingEvent</code>.  A <code>null</code> parameter 
     * <code>org.apache.log4j.spi.LoggingEvent</code> will cause the API to return <code>NEUTRAL</code>.
     * <p>
     *
     * @param loggingEvent The <code>org.apache.log4j.spi.LoggingEvent</code> to decide upon.
     * @return The decision of the filter.
     * @see org.apache.log4j.spi.Filter#decide(LoggingEvent)
     */
    public int decide(LoggingEvent loggingEvent) {
        
        if (loggingEvent != null) {
            
            if (loggingEvent.getMessage() instanceof CommonBaseEvent) {
                return ACCEPT;
            }
            
            return DENY;
        }

        return NEUTRAL;
    }
}
