package org.eclipse.hyades.logging.log4j;

import org.apache.log4j.Layout;
import org.apache.log4j.spi.LoggingEvent;
import org.eclipse.hyades.internal.logging.core.Constants;
import org.eclipse.hyades.logging.core.IExternalizableToXml;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.CompletionException;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.EventFactoryHome;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext;
import org.eclipse.hyades.logging.events.cbe.util.EventFormatter;
import org.eclipse.hyades.logging.events.cbe.util.EventHelpers;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: XmlLayout.java,v 1.8 2005/04/19 02:52:20 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/** 
 * Extension of the Log4J <code>org.apache.log4j.Layout</code> abstract
 * class used by <code>org.apache.log4j.Appender</code> instances to format 
 * <code>org.apache.log4j.spi.LoggingEvent</code> instances to XML.
 * <p>
 * If the <code>org.apache.log4j.spi.LoggingEvent</code> instance contains an 
 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code> as its
 * <code>message</code> (e.g. instance of the 
 * <code>org.eclipse.hyades.logging.core.IExternalizableToXml</code> interface), 
 * the encapsulated Common Base Event is serialized to XML based on the Common Base 
 * Event v.1.0.1 specification.  Otherwise, the <code>org.apache.log4j.spi.LoggingEvent</code>
 * is serialized to Common Base Event XML based on a generic serialization algorithm as 
 * implemented in <code>org.eclipse.hyades.logging.events.cbe.util.EventHelpers#convertObjectToCommonBaseEvent(Object,int)</code>.
 * <p>
 * Callers may provide a Common Base Event XML file template (see 
 * org.eclipse.hyades.logging.core/schema/templateEvent.xsd) to configure the 
 * newly created Common Base Event before serialization to XML.  The template 
 * Event XML file is a well-formed XML document conforming to a predefined XML schema 
 * (e.g. templateEvent.xsd).  The naming convention used for the template Event XML file 
 * is:
 * <p>
 * &lt;logger name&gt;.event.xml
 * <p>
 * where &lt;logger name&gt; is the name of the logger specified in the parameter 
 * <code>org.apache.log4j.spi.LoggingEvent</code>.  For example, the template event XML file 
 * for the <code>myLogger</code> logger factory would be named <code>myLogger.event.xml</code>.
 * <p>
 * The desired event configuration template XML file must by on the classpath
 * and accessible by this class. The event configuration template XML file is
 * loaded using the following look-up sequence:
 * <p>
 * <ol>
 * <li>The <code>XmlLayout</code> class' class loader
 * is queried for the event configuration template XML file.</li>
 * <li>The system's class loader is queried for the event configuration
 * template XML file.</li>
 * <li>The current thread's context class loader is queried for the event
 * configuration template XML file.</li>
 * </ol>
 * <p>
 * An instance of this class will be returned from the <code>getLayout()</code> 
 * API on <code>org.apache.log4j.Appender</code> instances.
 * <p>  
 * Users may configure a <code>org.apache.log4j.Appender</code> to use the 
 * <code>XmlLayout</code> by adding the following the entry to the 
 * <code>&lt;appender&gt;</code> element in the <code>log4j.xml</code> 
 * properties file:
 * <p>
 * <code>
 * &lt;!-- XmlLayout --&gt;
 * &lt;!-- Serializes events to XML: --&gt;
 * &lt;layout class="org.eclipse.hyades.logging.log4j.XmlLayout" /&gt;
 * </code>
 * <p>
 * Alternatively, an instantiation of this layout class may be set directly on
 * <code>org.apache.log4j.Appender</code> instances by using the 
 * <code>setLayout()</code> API.
 * <p>
 * NOTE: The Log4J classes must be on the CLASSPATH at run-time to
 * utilize this appender class.
 * <p>
 *  
 * 
 * @author    Paul E. Slauenwhite
 * @version   April 18, 2005
 * @since     July 20, 2004
 * @see       org.apache.log4j.Layout
 * @see       org.apache.log4j.spi.LoggingEvent
 * @see       org.eclipse.hyades.logging.events.cbe.impl.EventXMLFileEventFactoryHomeImpl
 */
public class XmlLayout extends Layout {

    /**
     * Private static reference to the platform-dependent line separator character.
     */
    private static final String LINE_SEPARATOR = System.getProperty("line.separator");

    /**
     * Instance variable which holds a reference to a
     * <code>org.eclipse.hyades.logging.events.cbe.EventFactory</code> for
     * generating event instances.
     */
    private final static EventFactory EVENT_FACTORY = EventFactoryContext.getInstance().getSimpleEventFactoryHome().getAnonymousEventFactory();

    /**
     * Instance variable which holds a reference to a
     * <code>org.eclipse.hyades.logging.events.cbe.EventFactoryHome</code> for
     * generating pre-configured event instances.
     */
    private final static EventFactoryHome EVENT_FACTORY_HOME = EventFactoryContext.getInstance().getEventFactoryHome("org.eclipse.hyades.logging.events.cbe.impl.EventXMLFileEventFactoryHomeImpl", XmlLayout.class.getClassLoader());

    /**
     * Generates the XML representation of the parameter 
     * <code>org.apache.log4j.spi.LoggingEvent</code> instance.
     * <p>
     * If the <code>org.apache.log4j.spi.LoggingEvent</code> instance contains an 
	 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code> as its
	 * <code>message</code> (e.g. instance of the 
	 * <code>org.eclipse.hyades.logging.core.IExternalizableToXml</code> interface), 
	 * the encapsulated Common Base Event is serialized to XML based on the Common Base 
	 * Event v.1.0.1 specification.  Otherwise, the <code>org.apache.log4j.spi.LoggingEvent</code>
	 * is serialized to Common Base Event XML based on a generic serialization algorithm as 
	 * implemented in <code>org.eclipse.hyades.logging.events.cbe.util.EventHelpers#convertObjectToCommonBaseEvent(Object,int)</code>.
     * <p>
     * Callers may provide a Common Base Event XML file template (see 
     * org.eclipse.hyades.logging.core/schema/templateEvent.xsd) to configure the 
     * newly created Common Base Event before serialization to XML.  See class comment 
     * header for more details.
     * <p>
     * 
     * @param loggingEvent The <code>org.apache.log4j.spi.LoggingEvent</code> instance to be formatted to XML.
     * @return The XML representation of the parameter <code>org.apache.log4j.spi.LoggingEvent</code> instance.
     * @see org.apache.log4j.Layout#format(LoggingEvent)
     */
    public String format(LoggingEvent loggingEvent) {

        if(loggingEvent != null){
            
            Object message = loggingEvent.getMessage();

            if (message instanceof IExternalizableToXml){
                return ((((IExternalizableToXml) message).externalizeCanonicalXmlString()).concat(LINE_SEPARATOR));
            }                        
        }
        
        //Map the parameter LoggingEvent's properties to Common Base Event properties:
        CommonBaseEvent commonBaseEvent = null;
        String loggerName = loggingEvent.getLoggerName();
        
        if((loggerName != null) && (loggerName.trim().length() > 0)){

            //Retrieve an EventFactory instance based on the logger name and create a pre-configured Common Base Event based 
            //on the an XML configuration template file that exists on the class loader's or context class loader's classpath.
            commonBaseEvent = EVENT_FACTORY_HOME.getEventFactory(loggerName).createCommonBaseEvent();
		    
            try {
                commonBaseEvent.complete();
            } 
            catch (CompletionException c) {
                //Ignore since content completion is based on 'best-effort'.
            }
        }
        else{
            commonBaseEvent = EVENT_FACTORY.createCommonBaseEvent();
        }

        EventHelpers.convertObjectToCommonBaseEvent(commonBaseEvent, loggingEvent,4);
        
        return (EventFormatter.toCanonicalXMLString(commonBaseEvent, true).concat(Constants.LINE_SEPARATOR));
    }

    /**
     * If the layout handles the throwable object contained within LoggingEvent, 
     * then the layout should return false. Otherwise, if the layout ignores 
     * throwable object, then the layout should return true. 
     * 
     * @return boolean true, if the layout handles the throwable object contained within LoggingEvent, other false.
     * @see org.apache.log4j.Layout#ignoresThrowable()
     */
    public boolean ignoresThrowable() {
        return true;
    }

    /**
     * Activate the options that were previously set with calls to option setters. 
     * This allows to defer activiation of the options until all options have been set. 
     * This is required for components which have related options that remain ambigous 
     * until all are set.  For example, the FileAppender has the File and Append options 
     * both of which are ambigous until the other is also set.
     * 
     * @see org.apache.log4j.spi.OptionHandler#activateOptions()
     */
    public void activateOptions() {
    }
}
