/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BasePanel.java,v 1.3 2005/02/16 22:21:48 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.manual.runner.ui.panel;

import java.awt.LayoutManager;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.EventObject;

import javax.swing.JPanel;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;

import org.eclipse.hyades.test.manual.runner.ui.util.EditorItem;
import org.eclipse.hyades.test.manual.runner.ui.util.IRefreshable;

/**
 * @author marcelop
 * @since 1.0.2
 */
public abstract class BasePanel
extends JPanel implements IRefreshable, ActionListener, CaretListener
{
	private Object currentData;
	private boolean internalChange = false;
	
	/**
	 * Constructor for BasePanel.
	 * @param layout
	 * @param isDoubleBuffered
	 */
	public BasePanel(LayoutManager layout, boolean isDoubleBuffered)
	{
		super(layout, isDoubleBuffered);
		internalInitialize();
		createControl();
	}

	/**
	 * Constructor for BasePanel.
	 * @param layout
	 */
	public BasePanel(LayoutManager layout)
	{
		super(layout);
		internalInitialize();
		createControl();
	}

	/**
	 * Constructor for BasePanel.
	 * @param isDoubleBuffered
	 */
	public BasePanel(boolean isDoubleBuffered)
	{
		super(isDoubleBuffered);
		internalInitialize();
		createControl();
	}

	/**
	 * Constructor for BasePanel.
	 */
	public BasePanel()
	{
		super();
		internalInitialize();
		createControl();
	}

	private void internalInitialize()
	{
		try
		{
			initialize();
		}
		catch(Throwable t)
		{
		}
	}
	
	protected void initialize()
	{
	}	
	
	protected abstract void createControl();

	public void refreshContent(Object data)
	{
		if((data == null) || !(data instanceof EditorItem))
		{
			invalidData();
			setEditable(false);
			return;
		}
			
		EditorItem editorItem = (EditorItem)data;
		data = editorItem.getData();
		
		if(EditorItem.REFRESH_STATE.equals(data))
		{
			setEditable(editorItem.isEditable());
			return;
		}

		if(EditorItem.REFRESH.equals(data))
		{
			refreshPanel(true);
			return;
		}
		
		if((data == null) || (!isValidData(data)))
		{
			invalidData();
			setEditable(false);
			return;
		}
			
		boolean sameObject = data.equals(currentData);
		setCurrentData(data);
		setEditable(editorItem.isEditable());
		refreshPanel(sameObject);
	}
	
	protected void setCurrentData(Object data)
	{
		currentData = data;
	}

	protected Object getCurrentData()
	{
		return currentData;
	}
	
	protected abstract boolean isValidData(Object data);
	protected abstract void refreshPanel(boolean sameObject);
	
	protected abstract void invalidData();
	protected abstract void setEditable(boolean editable);
	
	/**
	 * @see ActionListener#actionPerformed(ActionEvent)
	 */
	public final void actionPerformed(ActionEvent event)
	{
		updateObject(event);
	}

	/**
	 * @see CaretListener#caretUpdate(CaretEvent)
	 */
	public void caretUpdate(CaretEvent event)
	{
		updateObject(event);
	}
	
	public final void updateObject(EventObject event)
	{
		internalChange = true;
		try
		{
			updateCurrentData(event);
		}
		finally
		{
			internalChange = false;
		}		
	}

	protected void updateCurrentData(EventObject event)
	{
	}
}
