/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ValidateEditListener.java,v 1.4 2005/02/16 22:21:33 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.sdb.internal.util;

import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.events.ShellAdapter;
import org.eclipse.swt.events.ShellEvent;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.hyades.sdb.internal.SDbPlugin;

public class ValidateEditListener extends ShellAdapter implements IValidateEditListener {
	private ResourceStateValidator fValidator;
	private boolean fNeedsStateValidation = true;
	private Shell fShell;
	private IWorkbenchPart fPart;
	private boolean fHasReadOnlyFiles = false;
	private boolean firstReadOnlyFileAttempt = true;
	private boolean fMessageUp = false;
	private boolean fIsActivating = false;
	private boolean fIsDeactivating = false;

	/**
	 * Constructor for ValidateEditListener.
	 */
	public ValidateEditListener(IWorkbenchPart part, ResourceStateValidator aValidator) {
		super();
		fPart = part;
		fValidator = aValidator;
		if (part != null)
			part.getSite().getPage().addPartListener(this);

		if (getShell() != null)
			getShell().addShellListener(this);
			
	}
	
	protected Shell getShell() {
		if (fShell == null) {
			if (fPart != null && fPart.getSite() != null)
				fShell = fPart.getSite().getShell();
		}
		return fShell;
	}

	/**
	 * @see IValidateEditListener#getValidator()
	 */
	public ResourceStateValidator getValidator() {
		return fValidator;
	}
	/**
	 * @see IValidateEditListener#getNeedsStateValidation()
	 */
	public boolean getNeedsStateValidation() {
		return fNeedsStateValidation;
	}

	/**
	 * @see IValidateEditListener#setNeedsStateValidation(boolean)
	 */
	public void setNeedsStateValidation(boolean needsStateValidation) {
		fNeedsStateValidation = needsStateValidation;
	}

	/**
	 * @see ResourceStateValidatorPresenter#promptForInconsistentFileRefresh(List)
	 */
	public boolean promptForInconsistentFileRefresh(List inconsistentFiles) {
		if (inconsistentFiles == null || inconsistentFiles.size() == 0) // this case should never occur.
			return false;
		String message = null;
		String title = SDbPlugin.getResourceString("SYMPTOMDB_MESSAGE");
		if (inconsistentFiles.size() == 1) {
			
			Object file = inconsistentFiles.get(0);
			String fileString = null;
			if (file instanceof Resource){
				fileString = ((Resource) file).getURI().devicePath();
			}
			else{
				fileString = file.toString();
			}
			message = SDbPlugin.getResourceString("STR_INCONS_FILE_EDITOR_INFO_");
			message = TString.change(message,"%1",fileString);
			
		} else {

			message = SDbPlugin.getResourceString("STR_INCONS_FILES_EDITOR_INFO_");
			for (Iterator iterator = inconsistentFiles.iterator(); iterator.hasNext();) {
				message += "    ";
				Object file = (Object) iterator.next();
				if (file instanceof Resource)
					message += ((Resource) file).getURI().devicePath();
				else
					message += file.toString();
				if (iterator.hasNext())
					message += ",";
				message += "\n";
			}
		}
		message += SDbPlugin.getResourceString("STR_UPDATE_EDITOR_INFO_");
		return MessageDialog.openQuestion(getShell(), title, message);
	}

	/**
	 * @see ResourceStateValidatorPresenter#getValidateEditContext()
	 */
	public Object getValidateEditContext() {
		return getShell();
	}

	protected void handleActivation() {
		if (fIsActivating) return;
		fIsActivating = true;
		try {
			fValidator.checkActivation(this);
			updatePartReadOnly();
		} catch (CoreException e) {
			// do nothing for now
		} finally {
			fIsActivating = false;
		}
	}
	
	/**
	 * @see org.eclipse.swt.events.ShellListener#shellActivated(ShellEvent)
	 */
	public void shellActivated(ShellEvent event) {
		handleActivation();
	}

	/**
	 * @see IPartListener#partActivated(IWorkbenchPart)
	 */
	public void partActivated(IWorkbenchPart part) {
		if (part == fPart)
			handleActivation();
	}

	/**
	 * @see IPartListener#partBroughtToTop(IWorkbenchPart)
	 */
	public void partBroughtToTop(IWorkbenchPart part) {
	}

	/**
	 * @see IPartListener#partClosed(IWorkbenchPart)
	 */
	public void partClosed(IWorkbenchPart part) {
		if (part == fPart){
			part.getSite().getPage().removePartListener(this);
		}
		if (getShell() != null)
			getShell().removeShellListener(this);		
	}

	/**
	 * @see IPartListener#partDeactivated(IWorkbenchPart)
	 */
	public void partDeactivated(IWorkbenchPart part) {
		if (part == fPart) {
			if (fIsDeactivating) return;
			fIsDeactivating = true;			
			try {
				fValidator.lostActivation(this);
				updatePartReadOnly();
			} catch (CoreException e) {
				// do nothing for now
			}finally {
				fIsDeactivating = true;
			}
		}
	}

	/**
	 * @see IPartListener#partOpened(IWorkbenchPart)
	 */
	public void partOpened(IWorkbenchPart part) {
	}


	public IStatus validateState() {
		if (fNeedsStateValidation) {
			setNeedsStateValidation(false);
			IStatus status = null;
			try {
				status = fValidator.validateState(this);
				if (status.getSeverity() == IStatus.ERROR) {
					ErrorDialog.openError(getShell() , SDbPlugin.getResourceString("SYMPTOMDB_MESSAGE"), 
					SDbPlugin.getResourceString("STR_CHECKOUT_ERR_INFO_"), status);
					setNeedsStateValidation(true);
				}
				fValidator.checkActivation(this);
				updatePartReadOnly();
			} catch (CoreException e) {
				// do nothing for now
			}
			return status;
		}
		return ResourceStateValidator.OK_STATUS;
	}
	
	/**
	 * @see ResourceStateValidatorPresenter#promptForInconsistentFileOverwrite(List)
	 */
	public boolean promptForInconsistentFileOverwrite(List inconsistentFiles) {

		if (inconsistentFiles == null || inconsistentFiles.size() == 0) // this case should never occur.
			return false;
		String message = "";
		String title = SDbPlugin.getResourceString("SYMPTOMDB_MESSAGE");
		
		if (inconsistentFiles.size() == 1) {

			Object file = inconsistentFiles.get(0);
			String fileString = null;
			if (file instanceof Resource)
				fileString = ((Resource) file).getURI().devicePath();
			else
				fileString = file.toString();
			message = SDbPlugin.getResourceString("STR_INCONSISTENT_FILE_INFO_");
			message = TString.change(message,"%1",fileString);

		} else {

			message = SDbPlugin.getResourceString("STR_INCONSISTENT_FILES_INFO_");
			for (Iterator iterator = inconsistentFiles.iterator(); iterator.hasNext();) {
				message += "    ";
				Object file = (Object) iterator.next();
				if (file instanceof Resource)
					message += ((Resource) file).getURI().devicePath();
				else
					message += file.toString();
				if (iterator.hasNext())
					message += ",";
				message += "\n";
			}
		}
		message += SDbPlugin.getResourceString("STR_FILEOVERWRITE_INFO_");
		return MessageDialog.openQuestion(getShell(), title, message);

	}
	
	protected boolean checkReadOnly() {
		fHasReadOnlyFiles = fValidator.checkReadOnly();
		return fHasReadOnlyFiles;
	}
	/**
	 * @see IValidateEditListener#hasReadOnlyFiles()
	 */
	public boolean hasReadOnlyFiles() {
		if (firstReadOnlyFileAttempt) {
			checkReadOnly();
			firstReadOnlyFileAttempt = false;
		}
		return fHasReadOnlyFiles;
	}
	
	/**
	 * Method updatePartReadOnly.
	 */
	protected void updatePartReadOnly() {
		if (!getNeedsStateValidation()) {
			checkReadOnly();
			setNeedsStateValidation(true);
		}
	}
	public boolean checkSave() throws CoreException {
		if(!validateState().isOK())
			return false;
		return getValidator().checkSave(this);
	}
	
	public void setShell(Shell aShell) {
		fShell = aShell;
	}
	
	protected IWorkbenchPart getWorkbenchPart(){
		return fPart;
	}
	
}
