/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 *
 * $Id: QueryUtils.java,v 1.7 2005/04/07 20:10:57 slavescu Exp $
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util.internal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.hierarchy.extensions.BinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.Operand;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.extensions.TimeBasedCorrelationQuery;
import org.eclipse.hyades.models.hierarchy.extensions.WhereExpression;
import org.eclipse.hyades.models.hierarchy.util.ContainmentTraverser;
/**
 * @author Marius Slavescu (slavescu@ca.ibm.com)
 * @since 3.3
 *  
 */
public class QueryUtils {
	public static final String DOLLAR_SIGN = "$";
	protected static Map subTypesIndex = new HashMap();
	public static SimpleSearchQuery removeClasses(SimpleSearchQuery simpleSearchQuery, final Set classes) {
		SimpleSearchQuery query = (SimpleSearchQuery) EcoreUtil.copy(simpleSearchQuery);
		final Map alreadyProcessed = new HashMap();
		final Set removeSet = new HashSet();
		List roots = new ArrayList();
		roots.add(query.getWhereExpression());
		ContainmentTraverser containmentTraverser = new ContainmentTraverser(roots) {
			protected boolean beforeChildren(EObject element) {
				String name = getElementName(element);
				if (name != null && name.startsWith(DOLLAR_SIGN))
					return super.beforeChildren(element);
				if (alreadyProcessed.containsKey(element))
					return super.beforeChildren(element);
				// initialize evaluation result
				alreadyProcessed.put(element, null);
				if (element instanceof SimpleOperand) {
					SimpleOperand o = (SimpleOperand) element;
					EClass c = o.getType();
					if (c == null && o.getFeature() != null) {
						c = o.getFeature().getEContainingClass();
					}
					if (c != null && classes.contains(c)) {
						BinaryExpression be = getParentBinaryExpression(o);
						if (be != null)
							removeSet.add(be);
					}
				}
				return super.beforeChildren(element);
			}
		};
		containmentTraverser.traverse();
		removeElementsAndCompactQuery(query, removeSet);
		return query;
	}
	public static BinaryExpression getParentBinaryExpression(EObject o) {
		if (o != null) {
			if (o instanceof BinaryExpression)
				return (BinaryExpression) o;
			else
				return getParentBinaryExpression(o.eContainer());
		}
		return null;
	}
	public static void removeElementsAndCompactQuery(SimpleSearchQuery query, Set elements) {
		for (Iterator iter = elements.iterator(); iter.hasNext();) {
			EObject element = (EObject) iter.next();
			removeElementAndCompactContainers(element);
		}
	}
	public static void removeElementAndCompactContainers(EObject element) {
		if (element == null)
			return;
		EObject container = element.eContainer();
		if (container == null)
			return;
		if (container instanceof LogicalExpression) {
			LogicalExpression le = (LogicalExpression) container;
			le.getArguments().remove(element);
			if (le.getArguments().isEmpty()) {
				removeElementAndCompactContainers(le);
			} else {
				compactWhereExpression(le);
			}
		}
		//		else if (container instanceof SimpleSearchQuery) {
		//			((SimpleSearchQuery) container).setWhereExpression(null);
		//		}
	}
	public static void compactWhereExpression(WhereExpression we) {
		if (we instanceof LogicalExpression && ((LogicalExpression) we).getArguments().size() > 0) {
			compactLogicalExpression((LogicalExpression) we);
			return;
		} else {
			compactBinaryExpression((BinaryExpression) we);
		}
	}
	public static void compactLogicalExpression(LogicalExpression le) {
		if (le.getArguments().size() == 1 && le.getOperator() == LogicalOperators.NOT_LITERAL) {
			return;
		}
		if (le.getArguments().size() > 1) {
			return;
		}
		EObject container = le.eContainer();
		if (container == null)
			return;
		if (container instanceof LogicalExpression) {
			LogicalExpression c = (LogicalExpression) container;
			if (le.getArguments().size() == 1)
				c.getArguments().set(c.getArguments().indexOf(le), le.getArguments().get(0));
			else
				c.getArguments().remove(le);
			compactLogicalExpression(c);
		}
	}
	public static void compactBinaryExpression(BinaryExpression be) {
		if (be.getRightOperands().size() > 0) {
			return;
		}
		EObject container = be.eContainer();
		if (container == null)
			return;
		if (container instanceof LogicalExpression) {
			LogicalExpression c = (LogicalExpression) container;
			c.getArguments().remove(be);
			compactLogicalExpression(c);
		}
	}
	public static String getElementName(EObject element) {
		if (element instanceof WhereExpression)
			return ((WhereExpression) element).getName();
		else if (element instanceof Operand)
			return ((Operand) element).getName();
		return element.toString();
	}
	public static Set getAllSubTypes(EClass c) {
		Set s = (Set) subTypesIndex.get(c);
		if (s == null) {
			s = new HashSet();
			subTypesIndex.put(c, s);
			for (Iterator iter = EPackage.Registry.INSTANCE.entrySet().iterator(); iter.hasNext();) {
				Map.Entry packageEntry = (Map.Entry) iter.next();
				if (packageEntry.getValue() == null || !(packageEntry.getValue() instanceof EPackage))
					continue;
				for (Iterator iterator = ((EPackage) packageEntry.getValue()).getEClassifiers().iterator(); iterator.hasNext();) {
					EClassifier classifier = (EClassifier) iterator.next();
					if (classifier instanceof EClass) {
						EClass class1 = (EClass) classifier;
						if (class1.getEAllSuperTypes().contains(c)) {
							s.add(class1);
						}
					}
				}
			}
		}
		return s;
	}
	public static void addSubTypes(EClass c, Map m, Collection l) {
		for (Iterator iter = QueryUtils.getAllSubTypes(c).iterator(); iter.hasNext();) {
			EClass element = (EClass) iter.next();
			Collection l1 = (Collection) m.get(element);
			if (l1 != null)
				l1.addAll(l);
			else
				m.put(element, l);
		}
	}
	public static void addSuperTypes(EClass c, Map m, Collection l) {
		for (Iterator iter = c.getEAllSuperTypes().iterator(); iter.hasNext();) {
			Object element = iter.next();
			Collection l1 = (Collection) m.get(element);
			if (l1 != null)
				l1.addAll(l);
			else
				m.put(element, l);
		}
	}
	public static void addSuperTypes(EClass c, Collection s) {
		for (Iterator iter = c.getEAllSuperTypes().iterator(); iter.hasNext();) {
			Object element = iter.next();
			s.add(element);
		}
	}
	public static void addSubTypes(EClass c, Collection s) {
		for (Iterator iter = QueryUtils.getAllSubTypes(c).iterator(); iter.hasNext();) {
			Object element = iter.next();
			s.add(element);
		}
	}
	public static Map getClassPredicatesIndex(SimpleSearchQuery query) {
		final Map whereExpressionValuesIndex = new HashMap();
		final Map classPredicatesIndex = new HashMap();
		final Map namesIndex = new HashMap();
		List roots = new ArrayList();
		roots.add(query.getWhereExpression());
		ContainmentTraverser containmentTraverser = new ContainmentTraverser(roots) {
			protected boolean beforeChildren(EObject element) {
				String name = QueryUtils.getElementName(element);
				if (name != null && name.startsWith(QueryUtils.DOLLAR_SIGN))
					return super.beforeChildren(element);
				if (whereExpressionValuesIndex.containsKey(element))
					return super.beforeChildren(element);
				// initialize evaluation result
				whereExpressionValuesIndex.put(element, null);
				if (element instanceof SimpleOperand) {
					SimpleOperand o = (SimpleOperand) element;
					EClass c = o.getType();
					if (c != null) {
						Collection l = (Collection) classPredicatesIndex.get(c);
						if (l == null) {
							l = new HashSet();
							classPredicatesIndex.put(c, l);
							addSubTypes(c, classPredicatesIndex, l);
						}
						//						if (!l.contains(element))
						l.add(element);
					} else if (o.getFeature() != null) {
						c = o.getFeature().getEContainingClass();
						Collection l = (Collection) classPredicatesIndex.get(c);
						if (l == null) {
							l = new HashSet();
							classPredicatesIndex.put(c, l);
							addSubTypes(c, classPredicatesIndex, l);
						}
						//						if (!l.contains(element))
						l.add(element);
					}
				}
				if (name != null)
					namesIndex.put(name, element);
				return super.beforeChildren(element);
			}
		};
		containmentTraverser.traverse();
		return classPredicatesIndex;
	}
	public static Map getClassPredicatesNoSubClassesIndex(SimpleSearchQuery query) {
		final Map whereExpressionValuesIndex = new HashMap();
		final Map classPredicatesIndex = new HashMap();
		final Map namesIndex = new HashMap();
		List roots = new ArrayList();
		if (query.getWhereExpression() == null)
			return classPredicatesIndex;
		roots.add(query.getWhereExpression());
		ContainmentTraverser containmentTraverser = new ContainmentTraverser(roots) {
			protected boolean beforeChildren(EObject element) {
				String name = QueryUtils.getElementName(element);
				if (name != null && name.startsWith(QueryUtils.DOLLAR_SIGN))
					return super.beforeChildren(element);
				if (whereExpressionValuesIndex.containsKey(element))
					return super.beforeChildren(element);
				// initialize evaluation result
				whereExpressionValuesIndex.put(element, null);
				if (element instanceof SimpleOperand) {
					SimpleOperand o = (SimpleOperand) element;
					EClass c = o.getType();
					if (c != null) {
						List l = (List) classPredicatesIndex.get(c);
						if (l == null) {
							l = new ArrayList();
							classPredicatesIndex.put(c, l);
						}
						if (!l.contains(element))
							l.add(element);
					} else if (o.getFeature() != null) {
						c = o.getFeature().getEContainingClass();
						List l = (List) classPredicatesIndex.get(c);
						if (l == null) {
							l = new ArrayList();
							classPredicatesIndex.put(c, l);
						}
						if (!l.contains(element))
							l.add(element);
					}
				}
				if (name != null)
					namesIndex.put(name, element);
				return super.beforeChildren(element);
			}
		};
		containmentTraverser.traverse();
		return classPredicatesIndex;
	}
	public static void removeClass(SimpleSearchQuery searchQuery, EClass clazz) {
		Set s = new HashSet();
		s.add(clazz);
		removeClasses(searchQuery, s);
	}
	/**
	 * @return
	 */
	public static SimpleSearchQuery getEmptyQuery() {
		SimpleSearchQuery query = ExtensionsFactory.eINSTANCE.createSimpleSearchQuery();
		query.setName("EmptyFilter");
		LogicalExpression logicalExpression = ExtensionsFactory.eINSTANCE.createLogicalExpression();
		logicalExpression.setOperator(LogicalOperators.AND_LITERAL);
		query.setWhereExpression(logicalExpression);
		return query;
	}
	public static TimeBasedCorrelationQuery getEmptyTimeBasedCorrelationQuery() {
		TimeBasedCorrelationQuery query = ExtensionsFactory.eINSTANCE.createTimeBasedCorrelationQuery();
		query.setName("EmptyFilter");
		LogicalExpression logicalExpression = ExtensionsFactory.eINSTANCE.createLogicalExpression();
		logicalExpression.setOperator(LogicalOperators.AND_LITERAL);
		query.setWhereExpression(logicalExpression);
		return query;
	}
}