/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: OutOfProcessStrategy.java,v 1.2 2005/05/17 03:49:04 sschneid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.client.strategies;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;

import org.eclipse.hyades.automation.core.Service;

/**
 * The out-of-process strategy is used as the default and launches the
 * automation server and executes the automation command via the Eclipse
 * command-line in a separate Eclipse process.
 * 
 * @author Scott E. Schneider
 */
class OutOfProcessStrategy extends AbstractExecutionStrategy {

    /**
     * The automation command identifier to be used as the command name on the
     * command-line
     */
    private static final String AUTOMATION_COMMAND_IDENTIFIER = "hyades.automation.command";

    /**
     * The automation data identifier to be used as the command name on the
     * command-line
     */
    private static final String AUTOMATION_DATA_IDENTIFIER = "hyades.automation.data";

    /**
     * The automation server identifier, specifies the command to run in Eclipse
     * via the command-line
     */
    private static final String AUTOMATION_SERVER_IDENTIFIER = "org.eclipse.hyades.execution.core.server";

    /**
     * This debug flag is used by developers to turn on helpful debug support,
     * in this case it sets up the eclipse process to be remotely debugged via
     * Eclipse
     */
    private static final boolean DEBUG = false;

    /**
     * The service memento binary filename to use when passing service state
     * from client to server, all non-transient state is sent
     */
    private static final String SERVICE_MEMENTO_FILENAME = "memento.dat";

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.automation.core.Service.Executable#execute(org.eclipse.hyades.automation.core.Service)
     */
    public Object execute(Service service) {

        try {

            // Create and write the memento object
            Service.Memento memento = service.createMemento();
            FileOutputStream fileOutput = new FileOutputStream(SERVICE_MEMENTO_FILENAME);
            ObjectOutputStream objectOutput = new ObjectOutputStream(fileOutput);
            objectOutput.writeObject(memento);
            objectOutput.close();
            fileOutput.close();

            // Build the command-line string to launch Eclipse
            StringBuffer command = new StringBuffer(service.getRoot());
            command.append("\\eclipse.exe -clean -nosplash ");
            command.append("-application ");
            command.append(OutOfProcessStrategy.AUTOMATION_SERVER_IDENTIFIER + " ");
            command.append("-vmargs ");
            if (DEBUG) {
                command.append("-Xdebug -Xnoagent -Djava.compiler=NONE ");
                command.append("-Xrunjdwp:transport=dt_socket,server=y,suspend=y,address=8000 ");
            }
            command.append("-D" + OutOfProcessStrategy.AUTOMATION_COMMAND_IDENTIFIER);
            command.append("=execute ");
            command.append("-D" + OutOfProcessStrategy.AUTOMATION_DATA_IDENTIFIER);
            command.append("=" + OutOfProcessStrategy.SERVICE_MEMENTO_FILENAME);

            // Render the string and execute the automation server
            String commandString = command.toString();
            Process process = Runtime.getRuntime().exec(commandString);
            try {
                int returnCode = process.waitFor(); // 0 is successl
                return new Integer(returnCode);
            } catch (Throwable e) {
                e.printStackTrace();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        // The automation execution failed, return null
        return null;

    }

}