/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AbstractService.java,v 1.3 2005/05/20 21:38:57 sschneid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.core;

import java.util.Properties;

/**
 * The abstract service class that all published services should extend from,
 * classes must extend from this class in order to hydrate the dehydrated
 * service state that is passed in to the automation server
 * 
 * @author Scott E. Schneider
 * @provisional
 */
public abstract class AbstractService implements Service {

    /**
     * The service memento class, used to save and then restore the service
     * state from outside of Eclipse to within Eclipse
     * 
     * @author Scott E. Schneider
     */
    private class Memento implements Service.Memento {

        /**
         * The service identifier to command
         */
        private String identifier;

        /**
         * The service state stored and accessed as name value properties
         */
        private Properties properties;

        /**
         * Constructs a memento, only the abstract service class can create
         * these, protects encapsulation
         */
        private Memento() {
            this.identifier = AbstractService.this.state.identifier;
            this.properties = AbstractService.this.state.properties;
        }

        /**
         * Constructs a memento for this service's state given a service
         * identifier
         * 
         * @param identifier
         *            the automatable service to command
         */
        private Memento(String identifier) {
            this.identifier = identifier;
            this.properties = new Properties();
        }

    }

    /**
     * The service's persisting state
     */
    private transient AbstractService.Memento state;

    /**
     * The service is constructed (for use by the Eclipse platform extension
     * registry's executable extension method)
     */
    protected AbstractService() {
        // Constructed via executable extension on server-side
    }

    /**
     * Constructs an abstract service given the specified state
     * 
     * @param state
     *            the service's state
     */
    protected AbstractService(Service.Memento state) {
        if (state instanceof AbstractService.Memento) {
            this.state = (AbstractService.Memento) state;
        }
    }

    /**
     * The abstract service is constructed with the given service identifier
     * 
     * @param identifier
     *            the service identifier
     */
    protected AbstractService(String identifier) {
        this.state = new Memento(identifier);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.automation.core.Service#configure(java.util.Properties)
     */
    public void configure(Properties properties) {
        this.state.properties = properties;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.automation.core.Service#createMemento()
     */
    public Service.Memento createMemento() {
        return new Memento();
    }

    /**
     * Returns the identifier for this service
     * 
     * @return the service identifier (typically the extension point id of the
     *         implementor, identifies the specific service to be run on the
     *         server)
     */
    protected String getIdentifier() {
        return this.state.identifier;
    }

    /**
     * Retrives the service's persisting properties
     * 
     * @return the properties associated with this service
     */
    protected Properties getProperties() {
        return this.state.properties;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.automation.core.Service#getRoot()
     */
    public String getRoot() {
        return "";
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.automation.core.Service#setMemento(org.eclipse.hyades.automation.core.Service.Memento)
     */
    public void setMemento(Service.Memento state) {
        if (state instanceof AbstractService.Memento) {
            this.state = (AbstractService.Memento) state;
        }
    }

}