package org.eclipse.hyades.logging.events;

import java.util.HashMap;

import org.eclipse.hyades.logging.events.exceptions.CannotCreateEventFactoryException;
import org.eclipse.hyades.logging.events.exceptions.MissingValueException;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EventFactoryHome.java,v 1.6 2005/02/16 22:20:41 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Utility class to obtain a new factory instance.
 * 
 * @version 1.0.1
 * @since 1.0
 * @deprecated This class is deprecated as of 2004-05-17. Use the corresponding class available in org.eclipse.hyades.logging.events.cbe*
 */
public final class EventFactoryHome {

    /** 
     * A prefix used to derive the expected factory class name
     * from the event class name
     */
    protected static final String INTERFACE_PREFIX = "I";

    /** 
     * Synchronization lock object 
     */
    private static final Object LOCK = new String();

    /** 
     * Hash map of known event classes 
     */
    private static HashMap knownClasses = null;

    /**
     * Method to obtain an instance of an event factory.
     * 
     * @param instanceName		The name of the factory instance.
     * 
     * @param eventInterface	The interface for events created
     * 							by this factory.
     * 
     * <p>If a factory instance already exists with the desired name
     * and event interface, it will be returned.</p>
     * 
     * <p>If no such instance yet exists, it will be created. The
     * factory class (whose name is derived by removing any leading 'I'
     * from the event interface class name, then appending 'Factory') will
     * be loaded and instantiated, and the instance name will be set via
     * the init() method.</p>
     * 
     * @return		The new (or pre-existing) event factory instance.
     * 
     * @throws CannotCreateEventFactoryException
     * 
     * Thrown if the desired event factory does not exist and cannot be
     * created. If the event interface is named <em>IFooEvent</em>, the
     * factory class must be named <em>FooEventFactoryImpl</em> (no 'I'),
     * and must be a public class that implements <code>IEventFactory</code>
     * and allows new instances to be created.
     **/

    public static IEventFactory getFactory(String instanceName, String eventInterface) throws CannotCreateEventFactoryException {

        // Make sure instance name and event interface class name specified.

        if (instanceName == null) {
            throw new CannotCreateEventFactoryException(new MissingValueException("instanceName"));
        }

        if (eventInterface == null) {
            throw new CannotCreateEventFactoryException(new MissingValueException("eventInterface"));
        }

        IEventFactory eventFactory = null;

        synchronized (LOCK) {

            // Initialize the per-class map, if not already

            if (knownClasses == null) {
                knownClasses = new HashMap();
            }

            // Get the list of named instances for the event class
            // (or initialize a new list, if none exists yet)

            HashMap instances = (HashMap) knownClasses.get(eventInterface);

            if (instances == null) {
                instances = new HashMap();
                knownClasses.put(eventInterface, instances);
            }

            // See if the named event factory already exists

            eventFactory = (IEventFactory) instances.get(instanceName);

            // If if does not exist, try to instantiate it

            if (eventFactory == null) {
                String factoryClassName;

                if (eventInterface.startsWith(INTERFACE_PREFIX)) {
                    factoryClassName = eventInterface.substring(1);
                }
                else {
                    factoryClassName = eventInterface;
                }

                try {
                    Class factoryClass = Class.forName(factoryClassName);
                    eventFactory = (IEventFactory) factoryClass.newInstance();
                }
                catch (Exception ex) {
                    throw new CannotCreateEventFactoryException(ex);
                }

                eventFactory.init(instanceName);

                instances.put(instanceName, eventFactory);
            }

        } /* end synchronized section */

        return eventFactory;

    } /* end method getFactory() */

    /**
     * Method to release an instance of an event factory, when
     * no longer needed.
     * 
     * @param factory			The factory instance.
     **/
    public void releaseFactory(IEventFactory factory) {
        if (factory == null) {
            return;
        }

        synchronized (LOCK) {
            if (knownClasses != null) {
                String eventInterface = factory.getEventInterfaceClassName();
                String factoryName = factory.getName();

                if (eventInterface != null) {
                    HashMap instances = (HashMap) knownClasses.get(eventInterface);

                    if (instances != null) {
                        // if factory exists, release it from instance map

                        if (factoryName != null) {
                            instances.remove(factoryName);
                        }

                        // If instance map is now empty, release it from class map

                        if (instances.isEmpty()) {
                            knownClasses.remove(eventInterface);
                            instances = null;
                        }
                    }
                }

                // If class map is now empty, release it

                if (knownClasses.isEmpty()) {
                    knownClasses = null;
                }
            }

        } /* end synchronized block */

    } /* end method releaseFactory() */

} /* end class FactoryHome */
