/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JUnitGenerator.java,v 1.7 2005/05/03 22:15:12 bjiang Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.java.internal.codegen;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.test.common.internal.codegen.Helper;
import org.eclipse.hyades.test.common.internal.codegen.ImportManager;
import org.eclipse.hyades.test.common.internal.codegen.JavaGenerator;
import org.eclipse.hyades.test.java.TestJavaPlugin;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;

/**
 * @author marcelop, jcanches
 * @since 1.0.2
 */
public class JUnitGenerator 
extends JavaGenerator
{
	/**
	 * @see org.eclipse.hyades.test.common.internal.codegen.Generator#generateFile(org.eclipse.hyades.models.common.facades.behavioral.ITestSuite, org.eclipse.core.resources.IFile, org.eclipse.core.runtime.IProgressMonitor)
	 */
	protected void generateFile(ITestSuite testSuite, IFile file, IProgressMonitor monitor)
	throws Exception
	{
		GenTestSuite generator = new GenTestSuite();
		String content = Helper.formatContent(generator.generate(testSuite, new Helper()));	
		// use UTF-8 encoding - bugzilla_88029 running test with non-ASCII characters.
		file.create(new ByteArrayInputStream(content.getBytes(CHARSET_UTF8)), true, monitor);
		file.setCharset(CHARSET_UTF8, monitor);
	}
	
	protected void updateFile(ITestSuite testSuite, IFile file, IProgressMonitor monitor) {
		monitor.beginTask("", 4);
		try {
			ICompilationUnit cu = JavaCore.createCompilationUnitFrom(file);
			Helper helper = new Helper();
			String packageName = helper.getPackageName(testSuite);
			helper.setImportManager(new ImportManager(packageName));
			boolean failed = false;
			try {
				if (cu == null || !cu.isStructureKnown()) {
					failed = true;
					// TODO ask the user whether to override the code or not modify it
					return;
				}
				IType mainClass = cu.findPrimaryType();
				cu.becomeWorkingCopy(null, new SubProgressMonitor(monitor, 1));
				if (!testSuite.getImplementor().isExternalImplementor()) {
					updateSuiteMethod(testSuite, mainClass, helper);
				}
				monitor.worked(1);
				updateTestMethods(testSuite, mainClass, helper, new SubProgressMonitor(monitor, 1));
				helper.emitSortedImports(cu);
				cu.commitWorkingCopy(/*force*/false, new SubProgressMonitor(monitor, 1));
			} catch (JavaModelException e) {
				TestJavaPlugin.logError(e);
				failed = true;
			}
		} finally {
			monitor.done();
		}
	}

	/**
	 * Update the content of the suite() method for the specified class,
	 * with the behavior of the specified test suite.
	 * This implementation replaces the content of the suite() method, or creates
	 * the method if it does not exist.
	 * @throws JavaModelException 
	 */
	protected void updateSuiteMethod(ITestSuite testSuite, IType mainClass, Helper helper) throws JavaModelException {
		// Compute the suite() method body
		GenSuiteMethod generator = new GenSuiteMethod();
		String content = generator.generate(testSuite, helper);
		IMethod suiteMethod = mainClass.getMethod("suite", new String[0]);
		IMethod sibling = null; // The suite() method sits before this method
		if (suiteMethod.exists()) {
			IMethod[] methods = mainClass.getMethods();
			int index = Arrays.asList(methods).indexOf(suiteMethod);
			suiteMethod.delete(/*force*/false, new NullProgressMonitor());
			methods = mainClass.getMethods();
			if (index < methods.length) {
				sibling = methods[index];
			}
		} else {
			IMethod[] methods = mainClass.getMethods();
			if (methods.length > 0) {
				sibling = methods[0];
			}
		}
		mainClass.createMethod(Helper.formatContent(content, 1), sibling, /*force*/true, new NullProgressMonitor());
	}
	
	/**
	 * Returns whether a method is a test method.
	 * @param method
	 * @return
	 */
	private boolean isTestMethod(IMethod method) {
		String name = method.getElementName();
		String[] types = method.getParameterTypes();
		return name.startsWith("test") && types.length == 0;
	}
	
	/**
	 * Update the list of test methods in the specified class, so that the list of
	 * test methods matches the list of test cases of the specified test suite.
	 * @param testSuite
	 * @param mainClass
	 * @param helper
	 * @throws JavaModelException
	 */
	protected void updateTestMethods(ITestSuite testSuite, IType mainClass, Helper helper, IProgressMonitor monitor) throws JavaModelException {
		IMethod[] methods = mainClass.getMethods();
		monitor.beginTask("", methods.length + testSuite.getITestCases().size());
		try {
			List usedTestMethods = new ArrayList(methods.length);
			// 1) Add missing test methods
			Iterator it = testSuite.getITestCases().iterator();
			while (it.hasNext()) {
				ITestCase testCase = (ITestCase) it.next();
				String methodName = helper.getTestMethodName(testCase);
				IMethod method = mainClass.getMethod(methodName, new String[0]);
				if (!method.exists()) {
					method = createTestMethod(mainClass, testCase, helper);
				}
				usedTestMethods.add(method);
				monitor.worked(1);
			}
			// 2) Remove unused test methods
			for (int i = 0; i< methods.length; i++) {
				if (isTestMethod(methods[i]) && !usedTestMethods.contains(methods[i])) {
					methods[i].delete(/*force*/false, new NullProgressMonitor());
				}
				monitor.worked(1);
			}
		} finally {
			monitor.done();
		}
	}

	/**
	 * Creates a test method for the specified test case.
	 * @param mainClass
	 * @param testCase
	 * @param helper
	 * @throws JavaModelException 
	 */
	protected IMethod createTestMethod(IType mainClass, ITestCase testCase, Helper helper) throws JavaModelException {
		GenTestMethod generator = new GenTestMethod();
		String content = generator.generate(testCase, helper);
		return mainClass.createMethod(content, null, /*force*/false, new NullProgressMonitor());
	}
	
}