/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FileUtil.java,v 1.4 2005/02/16 22:21:22 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.models.common.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.security.AccessController;
import java.util.Vector;
import sun.security.action.GetPropertyAction;

/**
 * This class implements some utility methods for dealing with java.io.File
 * Some of the methods require JDK 1.4
 * 
 * @author jtoomey
 *
 */
public class FileUtil {

	/**
  	 * Returns the system temp directory.
  	 * @return File
  	 */
    public static File getTempDir()
    {
	    GetPropertyAction a = new GetPropertyAction("java.io.tmpdir");
	    String tmpdir = ((String)AccessController.doPrivileged(a));
	    return new File(tmpdir);
	}
    
    /**
     * This method uses JDK 1.4 nio channels to perform fast file copy.
     * 
     * @param srcFile
     * @param destFile
     * @return true if the file copy was successful, false otherwise
     */
    public static boolean copyFile( String srcFile, String destFile )
    {
    	try {
    		FileChannel srcChannel = new FileInputStream(srcFile).getChannel();
    		FileChannel destChannel = new FileOutputStream(destFile).getChannel();
    		destChannel.transferFrom(srcChannel, 0, srcChannel.size());
    		srcChannel.close();
    		destChannel.close();
		} 
    	catch (IOException e) {
    		return false;
    	}
    	return true;
    }

    
    /**
     * This method returns an array of File objects that are children of the
     * specified srcDir, and that return <code>true</code> to the .isFile() method.
     * In other words, the array is the result of a recursive search of the
     * specified directory for all files (but not directories.) 
     * @param srcDir
     * @return File[] All files that are children of the specified directory, 
     * 		   or null if srcDir does not contain any files.
     */
    public static File[] listAllFiles(File srcDir)
    {
    	Vector files = new Vector();
    	File[] ret = new File[0];
    	File[] children = srcDir.listFiles();
    	if ( children == null )
    	{
    		return null;
    	}
    	
    	for ( int i=0; i<children.length; i++ )
    	{
    		if ( children[i].isFile())
    		{
    			files.add(children[i]);
    		}
    		else
    		{
    			File[] sub = listAllFiles(children[i]);
    			for ( int j=0; j<sub.length; j++ )
    			{
    				files.add(sub[j]);
    			}
    		}
    	}
    	
    	if ( files.isEmpty() )
    		return null;
    	else
    		return (File[])files.toArray(ret);
    }
    
    /**
     * This method attempts to delete the specified root file.  If root is a directory
     * with children, this method attempts for first recursively delete all children 
     * of root.  If this methods returns true, then root has been deleted.  If this
     * method returns false, root has not been deleted, however if root is a 
     * directory, it is possible that some of its children were successfully 
     * deleted. 
     * @param root
     * @return true if root has bene deleted, false otherwise.
     */
    public static boolean deltree(File root)
    {
    	if ( root.exists() )
    	{
    		if ( root.isFile() )
    		{
    			return root.delete();
    		}
    		else
    		{
    			File[] children = root.listFiles();
    			if ( children != null )
    			{
	    			for (int i=0; i<children.length; i++)
	    			{
	    				deltree(children[i]);
	    			}
    			}
    			return root.delete();
    		}
    	}
    	return true;
    }
	
}
