/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BehaviorSection.java,v 1.6 2005/04/28 15:42:57 dguilbaud Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.common.internal.editor;

import java.util.Collection;
import java.util.Iterator;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.models.common.facades.behavioral.IAction;
import org.eclipse.hyades.models.common.facades.behavioral.IBlock;
import org.eclipse.hyades.models.common.facades.behavioral.ILoop;
import org.eclipse.hyades.models.common.facades.behavioral.ITestInvocation;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.fragments.Common_Behavior_FragmentsPackage;
import org.eclipse.hyades.models.common.interactions.Common_Behavior_InteractionsPackage;
import org.eclipse.hyades.test.common.TestCommonImages;
import org.eclipse.hyades.test.common.TestCommonPlugin;
import org.eclipse.hyades.test.common.internal.dialog.InvokedTestSelectionDialog;
import org.eclipse.hyades.test.common.internal.editor.action.AddLoop;
import org.eclipse.hyades.test.common.internal.editor.action.AddTestInvocation;
import org.eclipse.hyades.test.common.internal.editor.action.BehaviorActionGroup;
import org.eclipse.hyades.test.common.internal.editor.action.InsertLoop;
import org.eclipse.hyades.test.common.internal.editor.action.InsertTestInvocation;
import org.eclipse.hyades.test.common.internal.util.ContextIds;
import org.eclipse.hyades.test.common.internal.util.IITestSuiteProvider;
import org.eclipse.hyades.test.common.internal.util.TestCommonUtil;
import org.eclipse.hyades.test.ui.adapter.TestAdapterFactory;
import org.eclipse.hyades.test.ui.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.internal.editor.form.util.EObjectTreeSection;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.actions.ActionContext;
import org.eclipse.ui.actions.SelectionListenerAction;
import org.eclipse.ui.help.WorkbenchHelp;

/**
 * @author marcelop
 * @since 1.0.2
 */
public class BehaviorSection 
extends EObjectTreeSection implements IITestSuiteProvider
{
	protected static final int HELP_SECTION_CONTROL = 1;
	protected static final int HELP_SECTION_UP_ACTION = 2;
	protected static final int HELP_SECTION_DOWN_ACTION = 3;
	
	protected static final int HELP_SECTION_ADD_LOOP_ACTION = 4;
	protected static final int HELP_SECTION_ADD_INV_ACTION = 5;

	
	public static class BehaviorContentProvider
	extends EObjectTreeSection.EObjectContainerContentProvider
	{
		/**
		 * Constructor for BehaviorContentProvider
		 * @param editorForm
		 * @param eStructuralFeature
		 */
		public BehaviorContentProvider(EditorForm editorForm, EStructuralFeature eStructuralFeature)
		{
			super(editorForm, eStructuralFeature);
		}

		/**
		 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
		 */
		public void dispose()
		{
			if(getContainer() != null)
				EMFUtil.removeAdapter(getContainer().eResource().getAllContents(), this);
				
			super.dispose();
		}

		/**
		 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EObjectTreeSection.EObjectContainerContentProvider#deregisterInputAdapters(org.eclipse.emf.ecore.EObject, java.lang.Object)
		 */
		protected void deregisterInputAdapters(EObject oldInput, Object value)
		{
			if(oldInput.eResource() != null)
				EMFUtil.removeAdapter(oldInput.eResource().getAllContents(), this);
		}

		/**
		 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EObjectTreeSection.EObjectContainerContentProvider#registerInputAdapters(org.eclipse.emf.ecore.EObject, java.lang.Object)
		 */
		protected void registerInputAdapters(EObject newInput, Object value)
		{
			EMFUtil.addAdapter(newInput.eResource().getAllContents(), this);
		}

		/**
		 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EObjectTreeSection.EObjectContainerContentProvider#getParent(java.lang.Object)
		 */
		public Object getParent(Object element)
		{
			if(element instanceof IAction)
			{
				EObject eObject = ((EObject)element).eContainer();
				while((!(eObject instanceof ILoop)) && (eObject != null))
					eObject = eObject.eContainer();
				return eObject;
			}
			
			return super.getParent(element);
		}

		/**
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
		 */
		public boolean hasChildren(Object element)
		{
			if(super.hasChildren(element))
				return true;
				
			if(element instanceof ILoop)
				return !((ILoop)element).getBlock().getActions().isEmpty();
				
			return false;
		}

		/**
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
		 */
		public Object[] getChildren(Object parentElement)
		{
			if(parentElement instanceof ILoop)
				return ((ILoop)parentElement).getBlock().getActions().toArray();

			return super.getChildren(parentElement);
		}

		/**
		 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EObjectTreeSection.EObjectContainerContentProvider#adjustAdapter(org.eclipse.emf.common.notify.Notification)
		 */
		protected void adjustAdapter(Notification msg)
		{
			EObject eObject = null;
			switch(msg.getEventType())
			{
				case Notification.ADD:
					eObject = (EObject)msg.getNewValue();
					eObject.eAdapters().add(this);
					EMFUtil.addAdapter(eObject.eAllContents(), this);
					break;

				case Notification.ADD_MANY:
					for (Iterator i = ((Collection)msg.getNewValue()).iterator(); i.hasNext();)
						EMFUtil.addAdapter(((EObject)i.next()).eAllContents(), this);
					break;
					
				case Notification.REMOVE:
					eObject = (EObject)msg.getOldValue();
					eObject.eAdapters().remove(this);
					EMFUtil.removeAdapter(eObject.eAllContents(), this);
					break;

				case Notification.REMOVE_MANY:
				for (Iterator i = ((Collection)msg.getOldValue()).iterator(); i.hasNext();)
					EMFUtil.removeAdapter(((EObject)i.next()).eAllContents(), this);
				break;
			}
		}

		/**
		 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EObjectTreeSection.EObjectContainerContentProvider#shouldUpdateViewer(org.eclipse.emf.common.notify.Notification)
		 */
		protected boolean shouldUpdateViewer(Notification msg)
		{
			if(super.shouldUpdateViewer(msg))
				return true;
					
			return ((msg.getNotifier() instanceof IBlock) && (msg.getFeature() == Common_Behavior_FragmentsPackage.eINSTANCE.getBVRInteractionOperand_InteractionFragments()));
		}
		
		/**
		 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EObjectTreeSection.EObjectContainerContentProvider#attributeChanged(org.eclipse.emf.common.notify.Notification)
		 */
		protected void attributeChanged(Notification msg)
		{
			super.attributeChanged(msg);
			
			if((getViewer() != null) && (msg.getFeature() == Common_Behavior_InteractionsPackage.eINSTANCE.getBVRExecutionOccurrence_OtherBehavior()) && (msg.getNotifier() instanceof ITestInvocation))
			{
				IStructuredSelection structuredSelection = (IStructuredSelection)getViewer().getSelection();
				if(structuredSelection.isEmpty())
					return;
					
				if(structuredSelection.getFirstElement() == msg.getNotifier())
					UIUtil.fireCurrentSelection((StructuredViewer)getViewer());
			}
		}
	}
	
	public static class BehaviorLabelProvider
	extends LabelProvider
	{	
		/**
		 * @see org.eclipse.jface.viewers.ILabelProvider#getText(java.lang.Object)
		 */
		public String getText(Object element)
		{
			return TestAdapterFactory.INSTANCE.getHyadesWorkbenchAdapter().getLabel(element);
		}

		/**
		 * @see org.eclipse.jface.viewers.ILabelProvider#getImage(java.lang.Object)
		 */
		public Image getImage(Object element)
		{
			if(element instanceof ILoop)
				return TestCommonImages.INSTANCE.getImage(TestCommonImages.IMG_LOOP);

			if(element instanceof ITestInvocation)
				return TestCommonImages.INSTANCE.getImage(TestCommonImages.IMG_INVOCATION);

			ImageDescriptor imageDescriptor = TestAdapterFactory.INSTANCE.getHyadesWorkbenchAdapter().getImageDescriptor(element);
			if(imageDescriptor != null)
				return imageDescriptor.createImage();
				
			return null;
		}
	}
	
	private BehaviorActionGroup addBehaviorActionGroup;
	private BehaviorActionGroup insertBehaviorActionGroup;
		
	/**
	 * Constructor for CommonBehaviorSection
	 * @param editorForm
	 * @param eStructuralFeature
	 * @param addAction
	 */
	public BehaviorSection(BehaviorForm behaviorForm)
	{
		super(behaviorForm, Common_Behavior_FragmentsPackage.eINSTANCE.getBVRInteraction_InteractionFragments(), null);
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.FormSection#dispose()
	 */
	public void dispose()
	{
		if(addBehaviorActionGroup != null)
			addBehaviorActionGroup.dispose();

		if(insertBehaviorActionGroup != null)
			insertBehaviorActionGroup.dispose();

		super.dispose();
	}
	
	public ITestSuite getTestSuite()
	{
		return ((BehaviorForm)getEditorForm()).getTestSuite();
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EObjectTreeSection#adjustButtonLabels()
	 */
	protected void adjustButtonLabels(String addLabel)
	{
		super.adjustButtonLabels(TestCommonPlugin.getString("BTN_ADD"),TestCommonPlugin.getString("BTN_INSERT")); //$NON-NLS-1$ //$NON-NLS-2$
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EObjectTreeSection#adjustClient(org.eclipse.swt.widgets.Composite)
	 */
	protected void adjustClient(Composite parent)
	{
		super.adjustClient(parent);

		registerHelp(HELP_SECTION_CONTROL, getTreeViewer().getControl());				
		registerHelp(HELP_SECTION_UP_ACTION, moveUpAction);				
		registerHelp(HELP_SECTION_DOWN_ACTION, moveDownAction);
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.StructuredViewerSection#createActions()
	 */
	protected void createActions()
	{	
		addBehaviorActionGroup = createAddBehaviorActionGroup();
		insertBehaviorActionGroup = createInsertBehaviorActionGroup();
		super.createActions();
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EObjectTreeSection#createContentProvider()
	 */
	protected IContentProvider createContentProvider()
	{
		return new BehaviorContentProvider(getEditorForm(), getEStructuralFeature());
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EObjectTreeSection#createLabelProvider()
	 */
	protected ILabelProvider createLabelProvider()
	{
		return new BehaviorLabelProvider();
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.StructuredViewerSection#fillContextMenu(org.eclipse.jface.action.IMenuManager)
	 */
	protected void fillContextMenu(IMenuManager menuManager)
	{
		if(addBehaviorActionGroup != null)
		{
			addBehaviorActionGroup.fillContextMenu(menuManager);
			if(isReadOnly())
				addBehaviorActionGroup.disableActions();
		}

		if(insertBehaviorActionGroup != null)
		{
			insertBehaviorActionGroup.fillContextMenu(menuManager);
			if(isReadOnly())
				insertBehaviorActionGroup.disableActions();
		}

		super.fillContextMenu(menuManager);
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EObjectTreeSection#doAdd()
	 */
	protected void doAdd()
	{
		if(addBehaviorActionGroup != null)
		{
			Menu menu = getTreePart().getButtonMenu(0);
			addBehaviorActionGroup.fillContextMenu(menu, true);
			menu.setVisible(true);
		}
	}
	
	protected void doInsert()
	{
		if(insertBehaviorActionGroup != null)
		{
			Menu menu = getTreePart().getButtonMenu(1);
			insertBehaviorActionGroup.fillContextMenu(menu, true);
			menu.setVisible(true);
		}
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.StructuredViewerSection#updateActionsAndButtons(org.eclipse.jface.viewers.IStructuredSelection)
	 */
	protected void updateActionsAndButtons(IStructuredSelection structuredSelection)
	{
		if(addBehaviorActionGroup != null)
		{
			addBehaviorActionGroup.setContext(new ActionContext(structuredSelection));
			getTreePart().setButtonEnabled(0, addBehaviorActionGroup.isEnabled());
		}

		if(insertBehaviorActionGroup != null)
		{
			insertBehaviorActionGroup.setContext(new ActionContext(structuredSelection));
			getTreePart().setButtonEnabled(1, insertBehaviorActionGroup.isEnabled());
		}
		
		if((!structuredSelection.isEmpty()) && (((EObject)structuredSelection.getFirstElement()).eContainer() != getInput()))
		{
			moveUpAction.setEStructuralFeature(Common_Behavior_FragmentsPackage.eINSTANCE.getBVRInteractionOperand_InteractionFragments());
			moveDownAction.setEStructuralFeature(Common_Behavior_FragmentsPackage.eINSTANCE.getBVRInteractionOperand_InteractionFragments());
		}
		else
		{
			moveUpAction.setEStructuralFeature(getEStructuralFeature());
			moveDownAction.setEStructuralFeature(getEStructuralFeature());			
		}
		
		if(addAction != null)
		{
			if(addAction instanceof SelectionListenerAction)
				((SelectionListenerAction)addAction).selectionChanged(structuredSelection);
			getTreePart().setButtonEnabled(0, addAction.isEnabled());
		}

		if(insertAction != null)
		{
			if(insertAction instanceof SelectionListenerAction)
				((SelectionListenerAction)insertAction).selectionChanged(structuredSelection);
			getTreePart().setButtonEnabled(1, insertAction.isEnabled());
		}
		
		if(removeAction != null)
		{
			removeAction.selectionChanged(structuredSelection);
			getTreePart().setButtonEnabled(2, removeAction.isEnabled());
		}

		if(moveDownAction != null)
		{
			moveDownAction.selectionChanged(structuredSelection);
			getTreePart().setButtonEnabled(4, moveDownAction.isEnabled());
		}
		
		if(moveUpAction != null)
		{
			moveUpAction.selectionChanged(structuredSelection);
			getTreePart().setButtonEnabled(3, moveUpAction.isEnabled());
		}			
	}
	
	/**
	 * @see org.eclipse.hyades.test.common.internal.editor.BehaviorSection#createAddBehaviorActionGroup()
	 */
	protected BehaviorActionGroup createAddBehaviorActionGroup()
	{
		BehaviorActionGroup group = new BehaviorActionGroup(this, TestCommonPlugin.getString("LBL_ADD")); //$NON-NLS-1$

		AddLoop addLoop = new AddLoop(this, TestCommonPlugin.getString("LBL_LOOP"), TestCommonPlugin.getString("W_LOOP"));  //$NON-NLS-1$//$NON-NLS-2$
		registerHelp(HELP_SECTION_ADD_LOOP_ACTION, addLoop);
		group.addAction(addLoop);

		AddTestInvocation addTestInvocation = new AddTestInvocation(this, TestCommonPlugin.getString("W_LWR_INV")) //$NON-NLS-1$
		{
			protected InvokedTestSelectionDialog createInvokedTestSelectionDialog(ITestSuite testSuite)
			{
				return TestCommonUtil.createInvokedTestSelectionDialog(testSuite, false);
			}				
		};
		registerHelp(HELP_SECTION_ADD_INV_ACTION, addTestInvocation);
		group.addAction(addTestInvocation);
		
		return group;
	}
	
	/**
	 * @see org.eclipse.hyades.test.common.internal.editor.BehaviorSection#createAddBehaviorActionGroup()
	 */
	protected BehaviorActionGroup createInsertBehaviorActionGroup()
	{
		BehaviorActionGroup group = new BehaviorActionGroup(this, TestCommonPlugin.getString("LBL_INSERT")); //$NON-NLS-1$
		
		BehaviorContentProvider bcp = (BehaviorContentProvider)createContentProvider();

		InsertLoop insertLoop = new InsertLoop(this, TestCommonPlugin.getString("LBL_LOOP"), TestCommonPlugin.getString("W_LOOP"));  //$NON-NLS-1$//$NON-NLS-2$
		registerHelp(HELP_SECTION_ADD_LOOP_ACTION, insertLoop);
		group.addAction(insertLoop);

		InsertTestInvocation insertTestInvocation = new InsertTestInvocation(this, TestCommonPlugin.getString("W_LWR_INV")) //$NON-NLS-1$
		{
			protected InvokedTestSelectionDialog createInvokedTestSelectionDialog(ITestSuite testSuite)
			{
				return TestCommonUtil.createInvokedTestSelectionDialog(testSuite, false);
			}				
		};
		registerHelp(HELP_SECTION_ADD_INV_ACTION, insertTestInvocation);
		group.addAction(insertTestInvocation);
		
		return group;
	}
	
	/**
	 * @see org.eclipse.hyades.test.common.internal.editor.TestCasesForm#registerHelp(int, java.lang.Object)
	 */
	protected void registerHelp(int flag, Object object)
	{
		switch(flag)
		{
			case HELP_SECTION_CONTROL:
				WorkbenchHelp.setHelp((Control)object, TestCommonPlugin.getID() + ContextIds.BEHAVIOR_SECTION);				
				break;
				
			case HELP_SECTION_DOWN_ACTION:
				WorkbenchHelp.setHelp((Action)object, TestCommonPlugin.getID() + ContextIds.BEHAVIOR_SECTION_DOWN);
				break;
				
			case HELP_SECTION_UP_ACTION:
				WorkbenchHelp.setHelp((Action)object, TestCommonPlugin.getID() + ContextIds.BEHAVIOR_SECTION_UP);				
				break;

			case HELP_SECTION_ADD_LOOP_ACTION:
				WorkbenchHelp.setHelp((Action)object, TestCommonPlugin.getID() + ContextIds.ACT_ADD_LOOP);
				break;
				
			case HELP_SECTION_ADD_INV_ACTION:
				WorkbenchHelp.setHelp((Action)object, TestCommonPlugin.getID() + ContextIds.ACT_ADD_TEST_INV);				
				break;
		}
	}
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseStructuredViewerSection#buttonSelected(int)
	 */
	protected void buttonSelected(int index)
	{
		if(!isReadOnly())
		{
			switch(index)
			{
				case 0:
					doAdd();
					break;
				
				case 1:
					doInsert();
					break;
	
				case 2:
					doRemove();
					break;
	
				case 3:
					doMoveUp();
					break;
	
				case 4:
					doMoveDown();
					break;
				
			}
		}
		
		updateActionsAndButtons(getStructuredSelection());
	}

}
