/**********************************************************************
Copyright (c) 2005 IBM Corporation and others.
All rights reserved.  This program and the accompanying materials
are made available under the terms of the Eclipse Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/epl-v10.html
$Id: IExecutionComponent.java,v 1.5 2005/02/16 22:20:05 qiyanli Exp $

Contributors:
 IBM Corporation - initial implementation
 IBM Rational - initial implementation
	SCAPA Technologies - initial implementation
**********************************************************************/

package org.eclipse.hyades.execution.core;

/** 
 * 
 * <p>IExecutionComponent is the base class for all of the components in the
 * runtime system.  IExecutionComponent's are structured as a tree with
 * the specialized ISession is the root of the tree on the workbench side.</p>
 * 
 * <p>IExecutionComponent's are uniquely identified at runtime with an ID.
 * They are also named and typed but names and types are not guaranteed
 * to be unique.  Version support is provided as well.</p>
 * 
 * <p>IExecutionComponent's have at least four states.  They can be inactive,
 * ready, suspended, or dead.
 * <ul>
 * <li>Inactive IExecutionComponent's are
 * models of components that have not been created yet.</li>
 * <li>Not Configured IExecutionComponent's are those that have been created
 * but have not yet been completely initialized or properly configured
 * <li>Ready IExecutionComponent's are those that can produce or consume
 * IMessage's and IRuntimeVariable's.</li>
 * <li>Suspended IExecutionComponent's
 * are are those that have been created and are expected to become ready
 * at some point in the future.</li>
 * <li>Dead
 * IExecutionComponent's are those that will never be ready again.</li>
 * </ul></p>
 * @version 0.0.2
 */
public interface IExecutionComponent {
	
	public static final int INACTIVE = 0;
	public static final int NOT_CONFIGURED = 1; 
	public static final int READY = 2; 
	public static final int SUSPENDED = 3; 
	public static final int DEAD = 4; 
	
	
	/**
	 * This method should be called by the factory implementations
	 * after the object has been created so that it can do some 
	 * addional setup if necessary.
	 */
	public void init();
	
	/** 
	 * All IExecutionComponent instances must have an immutable
	 * name.  Names are user friendly identifiers.
	 * 
	 * @return the name of the IExecutionComponent
	 */
	public String getName();
	
	/** 
	 * All IExecutionComponent instances must have an immutable
	 * ID. Id's are globaly unique values that identify this unique
	 * IExecutionComponent runtime instance.
	 * 
	 * @return The ID of the IExecutionComponent
	 */
	public String getId();
	
	/** 
	 * All IExecutionComponent instances must have an immutable
	 * type. The type field is intended to allow you to identify groups
	 * of IExecutionComponent instances as having the same behaviour.
	 * 
	 * @return the type of the IExecutionComponent
	 */
	public String getType();
	
	/** 
	 * Retrieve the version.
	 */
	public String getVersion();
	
	/** 
	 * Get the container where this IExecutionComponent
	 * is hosted.  The typical heirarchy is a follows:
	 * ISession->IExecutor->IRemoteHyadesComponent
	 * It is possible, however, to have any level of nesting here.
	 * The ISession is always the root.
	 */
	IExecutionComponent getParent();
	
	/**
	 * Sets the container of this IExecutionComponent.
	 */
	void setParent(IExecutionComponent parent);
	
	/** 
	 * Get the first-level children that reside under this IExecutionComponent
	 * The typical heirarchy is a follows:
	 * ISession->IExecutor->IRemoteHyadesComponent
	 * It is possible, however, to have any level of nesting here.
	 * The ISession is always the root.
	 */
	IExecutionComponent[] getChildren();
	
	
	/** 
	 * Retrieve a child of this IExecutionComponent based upon its
	 * ID.
	 */
	public IExecutionComponent getChildById(String id);
	
	/** 
	 * Retrieve a child of this IExecutionComponent based upon its
	 * name.
	 * RKD:  This may not be the best idea due to name collision
	 * issues.  probably want to stick to just ID's.
	 * 
	 * @param name 
	 */
	public IExecutionComponent getChildByName(String name);

	/** 
	 * Add a child
	 * 
	 * @param child the child to add
	 */
	public void addChild(IExecutionComponent child);

	/** 
	 * Remove a child
	 * 
	 * @param child the child to remove
	 */
	public void removeChild(IExecutionComponent child);


	/** 
	 * Get the node on which this IExecutionComponent resides.
	 * 
	 * @return the node on which this IExecutionComponent resides
	 */
	public INode getNode();
	
	public int getState();
	
	/**
	 * Adds an IExecutionComponentStateChangeListener to this IExecutionComponent
	 * @param listener the listener to be informed of state change events.
	 */
	public void addExecutionComponentStateChangeListener(IExecutionComponentStateChangeListener listener);
	
	/**
	 * Removes an IExecutionComponentStateChangeListener from this IExecutionComponent
	 * @param listener the listener to be no longer informed of state change events.
	 */
	public void removeExecutionComponentStateChangeListener(IExecutionComponentStateChangeListener listener);
	
	/**
	 * Fire a state change event.  This should be in the interface because
	 * the behaviour of how to fire the event is tightly coupled with the 
	 * data structure used to hold the listeners.
	 * 
	 * @param newState
	 */
	public void fireStateChangeEvent(ExecutionComponentStateChangeEvent event);
	
	public void handleMessage(IControlMessage message) throws MessageProcessingException;
	
	public void sendMessage(IControlMessage message) throws MessageDeliveryException;
}
