/**********************************************************************
Copyright (c) 2005 IBM Corporation and others.
All rights reserved.  This program and the accompanying materials
are made available under the terms of the Eclipse Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/epl-v10.html
$Id: ProcessExecutorImpl.java,v 1.9 2005/02/16 22:20:08 qiyanli Exp $

Contributors:
 IBM Corporation - initial implementation
**********************************************************************/

package org.eclipse.hyades.execution.core.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;

import org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent;
import org.eclipse.hyades.execution.core.ExecutionComponentStateException;
import org.eclipse.hyades.execution.core.IExecutableObject;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.IExecutionEnvironment;
import org.eclipse.hyades.execution.core.IExecutor;
import org.eclipse.hyades.execution.core.IOrderedProperty;
import org.eclipse.hyades.execution.core.IProcessConsole;

/**
 * @author IBM
 *
 * This is an extension of the ExecutionEnvironmentImpl, also implements the IProcessConsole and IExecutor interfaces.
 */
public class ProcessExecutorImpl extends ExecutionComponentImpl implements IProcessConsole, IExecutor {
	private String name;
	private IExecutableObject exeObj;
	private OutputStream stdin;
	private InputStream stdout;
	private InputStream stderr;
	private ConsoleInputThread cin;
	private ConsoleOutputThread cout;
	private ConsoleErrorThread cerr;
	private IOrderedProperty[] processEnvironment = null;
	private int pid = 0;
	
	static {
		/* Loading the native runtime library */
		try {
			System.loadLibrary("hclaunch");
		}
		catch(Throwable e) {
		}
	}

	public ProcessExecutorImpl() {
		name = null;
	}

	public ProcessExecutorImpl(String _name) {
		name = _name;
	}

	/**
	 * Initialization routine.
	 */
	public void init() {
		super.init();

		cin = new ConsoleInputThread("Process console in");
		cout = new ConsoleOutputThread("Process console out");
		cerr = new ConsoleErrorThread("Process console error");
		cin.setDaemon(true);
		cout.setDaemon(true);
		cerr.setDaemon(true);
	}

	/**
	 * Set the executable object for launch
	 * @param obj IExecutableObject
	 */
	public void setExecutableObject(IExecutableObject obj) {
		exeObj = obj;
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutor#getExecutableObject()
	 */
	public IExecutableObject getExecutableObject()
	{
		return exeObj;
	}

	/**
	 * Launch the process and start the console handlers threads.
	 */
	public void launch() {
		IExecutionComponent parent;

		/* Find our execution enviornment if it exists */
		do {
			parent=getParent();

			if(parent instanceof IExecutionEnvironment) {
				processEnvironment = ((IExecutionEnvironment)parent).getEnv();
				break;
			}

		}while(parent != null);

		/* Start the consoles */
		cin.start();
		cout.start();
		cerr.start();

		/* Launch the process through JNI */
		pid = startProcess0();

		if(pid < 0) {
			System.out.println("Launch error, return code from startProcess0() = " + pid);
			return;
		}

		/* Windows does not have signal mechanism for child termination so we need a thread to monitor */
		if(System.getProperty("os.name").startsWith("Windows")) {
			ProcessStatusThread procThread = new ProcessStatusThread(this, pid);
			procThread.start();
		}
//		else {
//			while(exeListener != null) {
//				try {
//					Thread.sleep(5000);
//				} catch (InterruptedException e) {
//					e.printStackTrace();
//				}
//			}
//		}
	}

	/**
	 * Kill the active process (invoking a JNI call)
	 */
	public void kill() {
		killProcess0();
	}

	/**
	 * Get the OutputStream for writing to the process's stdin pipe. It is available after process is launched.
	 * @return OutputStream
	 * @throws ExecutionComponentStateException
	 */
	public OutputStream getStandardInputStream() throws ExecutionComponentStateException {
		if(stdin == null)
			throw new ExecutionComponentStateException(
			IExecutionComponent.NOT_CONFIGURED,
			"No standard input stream has been set");

		return stdin;
	}

	/**
	 * Get the InputStream for reading from the process's stdout pipe. It is available after process is launched.
	 * @return InputStream
	 * @throws ExecutionComponentStateException
	 */
	public InputStream getStandardOutputStream() throws ExecutionComponentStateException {
		if(stdout == null)
			throw new ExecutionComponentStateException(
			IExecutionComponent.NOT_CONFIGURED,
			"No standard output stream has been set");
		return stdout;
	}

	/**
	 * Get the InputStream for reading from the process's stderr pipe. It is available after process is launched.
	 * @return InputStream
	 * @throws ExecutionComponentStateException
	 */
	public InputStream getStandardErrorStream() throws ExecutionComponentStateException {
		if(stderr == null)
			throw new ExecutionComponentStateException(
			IExecutionComponent.NOT_CONFIGURED,
			"No standard error stream has been set");

		return stderr;
	}

	/**
	 * Called by the C/C++ code to notify a process has exited.
	 *
	 */
	private void processExited() {
		if(state==READY) {
			ExecutionComponentStateChangeEvent event = new ExecutionComponentStateChangeEvent(this, DEAD);
			fireStateChangeEvent(event);
		}

		// Bug 79026
		/*
		 * Commented out for Hyades 3.2i2 because it induced a problem (bug 81551)
		 * where the Session is locked up by the JVM on Linux. We will find a proper
		 * solution to solve this problem for Hyades 3.3.
		 */
		//cin.interrupt();
		//cout.interrupt();
		//cerr.interrupt();
	}

	/**
	 * Return the process id of the started process
	 * @return int
	 */
	public String getPid() {
		return new Integer(pid).toString();
	}

	/**
	 * Get the process status
	 * @param pid
	 * @return int
	 */
	public int getProcessStatus(int pid) {
		return getProcessStatus0(pid);
	}

	/**
	 * Get the process console associated with the current executor
	 * @return IProcessConsole 
	 */
	public IProcessConsole getProcessConsole() {
		return (IProcessConsole)this;
	}

	/**
	 * Monitors the process status
	 * @author IBM
	 *
	 */
	class ProcessStatusThread extends Thread {
		int pid;
		int cur_state = READY;
		IExecutor executor;

		public ProcessStatusThread(IExecutor _executor, int _pid) {
			setName("Process status monitor");
			executor = _executor;
			this.setDaemon(true);
			pid = _pid;
		}

		public void run() {
			while(cur_state != DEAD) {
				cur_state = getProcessStatus(pid);
				try {
					sleep(2000);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
			
			processExited();
			return;
		}

	}

	/**
	 * Thread for handling console input
	 * @author IBM
	 */
	class ConsoleInputThread extends Thread {
		private PipedInputStream reader;
		private StringBuffer strbuf;
		private char c;

		public ConsoleInputThread(String name) {
			setName(name);
			this.setDaemon(true);

			/* Create the I/O streams */
			stdin = new PipedOutputStream();

			/* Connect the pipes' read and write ends */
			try {
				reader = new PipedInputStream((PipedOutputStream)stdin);
			} catch(IOException e0) {
				e0.printStackTrace();
			}
		}

		public void run() {

			/* Listen to user's input and dump the completed line to the C/C++ layer */
			while(!isInterrupted()) { // Bug 79026
				try {
					c = (char)reader.read();
					writeToProcess0(c);
				} catch (IOException e1) {
					System.out.println("Console input closed by process");
					break;
				}
			}
		}
	}

	/**
	 * Thread for handling console output
	 * @author IBM
	 */
	class ConsoleOutputThread extends Thread {
		private PipedOutputStream writer;	
		
		public ConsoleOutputThread(String name) {
			setName(name);
			this.setDaemon(true);

			/* Create the I/O streams */
			stdout = new PipedInputStream();

			/* Connect the pipes' read and write ends */
			try {
				writer = new PipedOutputStream((PipedInputStream)stdout);
			} catch(IOException e0) {
				e0.printStackTrace();
			}
		}

		public void run() {
			int b;

			/* Listen to user's input and dump the completed line to the C/C++ layer */
			while(!isInterrupted()) { // Bug 79026
				try {
					b = readFromProcessOutput0();
					if(b != -1) {
						writer.write(b);
						writer.flush();
					}
					else {
						break;
					}
				} catch(IOException e1) {
					System.out.println("Console output closed by process");
					break;
				}
			}
		}
	}

	/**
	 * Thread for handling console error
	 * @author IBM
	 */
	class ConsoleErrorThread extends Thread {
		private PipedOutputStream writer;

		public ConsoleErrorThread(String name) {
			setName(name);
			this.setDaemon(true);

			/* Create the I/O streams */
			stderr = new PipedInputStream();

			/* Connect the pipes' read and write ends */
			try {
				writer = new PipedOutputStream((PipedInputStream)stderr);
			} catch(IOException e0) {
				e0.printStackTrace();
			}
		}

		public void run() {
			int b = 0;

			/* Listen to user's input and dump the completed line to the C/C++ layer */
			while(!isInterrupted()) { // Bug 79026
				try {
					b = readFromProcessError0();
					if(b != -1) {
						writer.write(b);
						writer.flush();
					}
					else {
						break;
					}
				} catch(Exception e1) {
					System.out.println("Console error closed by process");
					break;
				}
			}
		}
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutor.getCompatibleExecutableObject() 
	 */
	public IExecutableObject getCompatibleExecutableObject(String classname) throws ClassNotFoundException {
		IExecutableObject result=null;
		Class clazz=Class.forName(classname);
		try {
			result=(IExecutableObject)clazz.newInstance();
		}
		catch(InstantiationException e) {
	
		}
		catch(IllegalAccessException e) {
	
		}
		if(result!=null && result instanceof ProcessExecutableObjectImpl) {
			return result;
		}
		return null;
	}
	

	// ********** NATIVE METHODS **********
	private native int startProcess0();

	private native void killProcess0();

	private native void writeToProcess0(char c);

	private native int readFromProcessOutput0();

	private native int readFromProcessError0();

	private native int getProcessStatus0(int pid);

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.IExecutor#supportsControlEvent(java.lang.String)
	 */
	public boolean supportsControlEvent(String controlEvent) {
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.IExecutor#performControlEvent(java.lang.String)
	 */
	public String performControlEvent(String controlEvent, String[] params) {
		return "";
	}

}
