/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: PerformanceTimer.java,v 1.1 2005/05/03 14:32:54 sschneid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.execution.core.timer;

import java.io.PrintStream;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Stack;

/**
 * A lightweight performance timer to be used for quick situations where
 * approximate timing of a given algorithm or code is required. This class keeps
 * a pool of identified timers, retrieve timers based on identity using the
 * getInstance(...) method.
 * 
 * @author Scott E. Schneider
 */
public class PerformanceTimer implements IPerformanceTimer {

	private static final HashMap timers;

	private static final String UNNAMED = "Unnamed";

	static {
		timers = new HashMap(5);
	}

	public synchronized static IPerformanceTimer getInstance(String name) {
		return PerformanceTimer.getInstance(name, System.out);
	}

	public synchronized static IPerformanceTimer getInstance(String name, PrintStream stream) {
		PerformanceTimer timer = (PerformanceTimer) PerformanceTimer.timers.get(name);
		if (timer == null) {
			timer = new PerformanceTimer(name, stream);
			PerformanceTimer.timers.put(name, timer);
		}
		return timer;
	}

	private final String identity;

	private long size, previousSize;

	private Stack starts, names, sizes, counts;

	private LinkedList stops;

	private final PrintStream stream;

	private PerformanceTimer(String identity, PrintStream stream) {
		this.identity = identity;
		this.stream = stream;
		this.reset();
	}

	private long consumeCount() {
		return ((Long) this.counts.pop()).longValue();
	}

	private long consumeElapsed() {
		long stop = ((Long) this.stops.removeFirst()).longValue();
		long start = ((Long) this.starts.pop()).longValue();
		return stop - start;
	}

	private String consumeName() {
		return (String) this.names.pop();
	}

	private long deriveElasped(Long start, Long stop) {
		return stop.longValue() - start.longValue();
	}

	public void elapsed() {
		String name = this.consumeName();
		long elapsed = this.consumeElapsed();
		long count = this.consumeCount();
		this.stream.print("<Performance Timer " + this.identity + "> " + name + " execution is measured at " + elapsed + "ms ");
		if (count > 0) {
			long average = elapsed / count;
			this.stream.print("(" + average + "ms on average)");
		}
		this.stream.println();
	}

	private Long getTimestamp() {
		return new Long(System.currentTimeMillis());
	}

	public void reset() {
		this.names = new Stack();
		this.starts = new Stack();
		this.stops = new LinkedList();
		this.sizes = new Stack();
		this.counts = new Stack();
		this.size = 0;
		this.previousSize = 0;
	}

	public void start() {
		this.start(PerformanceTimer.UNNAMED);
	}

	public void start(String name) {
		this.starts.push(this.getTimestamp());
		this.names.push((this.names.size() > 0 ? (String) this.names.peek() + "::" : "") + name);
		this.sizes.push(new Long(this.size));
		this.size = 0;
	}

	public void stop() {
		this.stops.addLast(this.getTimestamp());
		this.previousSize = size;
		this.size = ((Long) this.sizes.pop()).longValue() + 1;
		this.counts.push(new Long(this.previousSize));
	}

}
