/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: MutableBoolean.java,v 1.1 2005/04/04 21:15:20 sschneid Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.execution.core.util;

/**
 * A mutable boolean is mainly used for the accumulation of state from local or
 * anonymously defined inner classes into the surrounding context (to be
 * accessed variables of this type must be defined final at point-of-use). A
 * java.lang.Boolean is immutable once declared final and therefore cannot be
 * used for this purpose.
 * 
 * This is almost equivalent to a pointer to an Boolean rather than just the
 * Boolean itself.
 * 
 * @author Scott E. Schneider
 * @since 3.3
 */
public class MutableBoolean {
	/**
	 * The internally stored state of the mutable Boolean
	 */
	private Boolean value;

	/**
	 * Constructs a mutable Boolean
	 */
	public MutableBoolean() {
	}

	/**
	 * Constructs a mutable Boolean with the given value.
	 * 
	 * @param value
	 *            the value the internally stored state of the mutable Boolean
	 *            will be initialized to
	 */
	public MutableBoolean(boolean value) {
		this.setValue(value);
	}

	/**
	 * Constructs a mutable Boolean with the given value
	 * 
	 * @param value
	 *            the value the internally stored state of the mutable Boolean
	 *            will be initialized to
	 */
	public MutableBoolean(Boolean value) {
		this.setValue(value.booleanValue());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object object) {
		if (object instanceof MutableBoolean) {
			if (object == this) {
				return true;
			}
			return (((MutableBoolean) object).value.equals(this.value));
		}
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode() {
		return this.value.hashCode();
	}

	/**
	 * The opposite of the is true method
	 * 
	 * @return false is the contained boolean value is false
	 */
	public boolean isFalse() {
		return !this.isTrue();
	}

	/**
	 * Indicates is the boolean value is true
	 * 
	 * @return true if the contained boolean value is true
	 */
	public boolean isTrue() {
		return this.value.booleanValue();
	}

	/**
	 * Sets the contained boolean value to false if not already false
	 */
	public void setAsFalse() {
		this.setValue(false);
	}

	/**
	 * Sets the contained boolean value to true if not already true
	 */
	public void setAsTrue() {
		this.setValue(true);
	}

	/**
	 * Set the contained boolean value to the value specified if not already,
	 * factory method for the contained boolean type used -- also included the
	 * redundancy check to avoid setting to the same value and creating another
	 * Boolean object unneedingly
	 * 
	 * @param value
	 *            the state to change to if not already in that state
	 */
	public void setValue(boolean value) {
		if (this.value == null || this.value.booleanValue() != value) {
			this.value = new Boolean(value);
		}
	}

}