/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: GetFileCommand.java,v 1.2 2005/05/05 19:52:37 sschneid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.internal.execution.core.file;

import java.io.File;
import java.io.IOException;
import java.util.Iterator;

import org.eclipse.core.internal.runtime.Assert;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option;

/**
 * The get file command is a file server command that is used to transfer files
 * from the server to the client. The main components of state that make up the
 * get file command are local identifiers of the files to transfer into, remote
 * identifiers of the files to pull from the server-side, options to tweak the
 * get file behavior and progress monitor to monitor the progress as well as
 * cancel the transfer. An advantage to the get file command is that it supports
 * the transfer of multiple files in one pass, using the same socket and
 * connection for all the files downloaded. A single file transfer is also
 * possible, basically 1 to many file identifiers can be passed in to identity
 * the local and remote files to act upon.
 * 
 * @author Scott E. Schneider
 */
class GetFileCommand extends AbstractFileTransferCommand implements IGetFileCommand {

    /**
     * The client side personality of the get file command, creates file and
     * downloads files from the server to the client
     * 
     * @author Scott E. Schneider
     */
    private class Client extends AbstractFileTransferCommand.Client {

        /**
         * Creates the client side personality for the get file command
         * 
         * @param channel
         *            the channel to communicate with the server
         */
        Client(ISocketChannel channel) {
            super(channel);
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.internal.execution.core.file.ICommand#execute()
         */
        public void execute() throws IOException {

            // Initiate client
            super.execute();

            // Retrieve local and remote identifiers
            FileIdentifierList localIdentifiers = GetFileCommand.this.localIdentifiers;
            FileIdentifierList remoteIdentifiers = GetFileCommand.this.remoteIdentifiers;
            Assert.isTrue(localIdentifiers.size() == remoteIdentifiers.size());

            // Send number of files that will be transferred
            this.send(remoteIdentifiers.size());

            // Send all remote identifier names (files to download)
            this.send(remoteIdentifiers.getArray());

            // Receive all the files, one at a time but continuous
            for (Iterator identifiers = localIdentifiers.iterator(); identifiers.hasNext();) {
                String fileName = (String) identifiers.next();
                File localFile = new File(fileName);
                boolean readyToTransfer = true;
                if (localFile.isAbsolute()) {
                    File localDirectory = new File(localFile.getParent());
                    if (!localDirectory.exists()) {
                        readyToTransfer = localDirectory.mkdirs();
                    }
                    if (readyToTransfer) {
                        localFile.createNewFile();
                        this.receive(localFile);
                    }
                }
            }

        }
    }

    /**
     * The server side personality of the get file command
     * 
     * @author Scott E. Schneider
     */
    private class Server extends AbstractFileTransferCommand.Server {

        /**
         * Constructs the server side state personality for the get file command
         * 
         * @param channel
         */
        Server(ISocketChannel channel) {
            super(channel);
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.internal.execution.core.file.ICommand#execute()
         */
        public void execute() throws IOException {

            // Initiate server
            super.execute();

            // Receive number of files that will be transferred
            int count = this.receiveInt();

            // Receive the remote identifiers from the client
            FileIdentifierList remoteIdentifiers = FileIdentifierList.create(this.receiveStrings());
            GetFileCommand.this.remoteIdentifiers = remoteIdentifiers;

            // Send all the files to the server one at a time but continuous
            for (Iterator identifiers = remoteIdentifiers.iterator(); identifiers.hasNext();) {
                String fileName = (String) identifiers.next();
                File remoteFile = new File(fileName);
                if (remoteFile.isFile()) {
                    this.send(remoteFile);
                }
            }

        }
    }

    /**
     * Constructs a get file command from a socket channel, this is the
     * constructor that the file server uses
     * 
     * @param channel
     *            the channel to communicate with the client
     */
    public GetFileCommand(ISocketChannel channel) {
        super(GetFileCommand.class);
        this.setState(new Server(channel));
    }

    /**
     * Constructs a get file command from a socket channel to the server, and
     * the command state that defines the exact purpose of this command
     * 
     * @param channel
     *            the channel to communicate with the server
     * @param localIdentifiers
     *            the local identifiers identifying the local files to transfer
     * @param remoteIdentifiers
     *            the remote identifiers identifying the remote files to copy
     *            into
     * @param options
     *            the options that tweak this command's behavior
     * @param monitor
     *            the progress monitor to use
     */
    GetFileCommand(ISocketChannel channel, FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers,
            Option[] options, IProgressMonitor monitor) {
        super(GetFileCommand.class, Cookie.NONE, localIdentifiers, remoteIdentifiers, options, monitor);
        this.setState(new Client(channel));
    }

}